/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-06 04:28:23 -0600 (Thu, 06 Aug 2009) $
--   $Revision: 4284 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file and the list_files.c file which it
--   was derived from.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <fnmatch.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../objectify_private.h"   /* do private so that nwos_get_public_objects_path is defined */
#include "../time_stamp.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}

void print_unknown(int field_width)
{
    int i;
    int spaces_before;
    int spaces_after;

    spaces_before = (field_width - 9) / 2;
    spaces_after = field_width - 9 - spaces_before;

    for (i = 0; i < spaces_before; i++) printf("&nbsp;");

    printf("&lang;Unknown&rang;");

    for (i = 0; i < spaces_after; i++) printf("&nbsp;");
}


void print_md5(ObjRef* file_ref)
{
    C_struct_File file_obj;
    C_struct_MD5sum md5_object;
    int j;

    nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj));  /* read the file object */
    nwos_read_object_from_disk(&file_obj.md5sum, &md5_object, sizeof(md5_object));

    for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_object.md5sum[j]);
}


void print_sha1(ObjRef* file_ref)
{
    C_struct_File file_obj;
    C_struct_SHA1sum sha1_object;
    int j;

    nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj));  /* read the file object */

    if (is_void_reference(&file_obj.sha1sum))
    {
             /*          1         2         3         4 */
             /* 1234567890123456789012345678901234567890 */
	print_unknown(40);
    }
    else
    {
	nwos_read_object_from_disk(&file_obj.sha1sum, &sha1_object, sizeof(sha1_object));

	for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_object.sha1sum[j]);
    }
}


void print_sha256(ObjRef* file_ref)
{
    C_struct_File file_obj;
    C_struct_SHA256sum sha256_object;
    int j;

    nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj));  /* read the file object */

    if (is_void_reference(&file_obj.sha256sum))
    {
             /*          1         2         3         4         5         6     */
             /* 1234567890123456789012345678901234567890123456789012345678901234 */
	print_unknown(64);
    }
    else
    {
	nwos_read_object_from_disk(&file_obj.sha256sum, &sha256_object, sizeof(sha256_object));

	for (j = 0; j < SHA256_DIGEST_SIZE; j++) printf("%02x", sha256_object.sha256sum[j]);
    }
}


void print_sha512(ObjRef* file_ref)
{
    C_struct_File file_obj;
    C_struct_SHA512sum sha512_object;
    int j;

    nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj));  /* read the file object */

    if (is_void_reference(&file_obj.sha512sum))
    {
	print_unknown(128);
    }
    else
    {
	nwos_read_object_from_disk(&file_obj.sha512sum, &sha512_object, sizeof(sha512_object));

	for (j = 0; j < SHA512_DIGEST_SIZE; j++) printf("%02x", sha512_object.sha512sum[j]);
    }
}


void print_time(TimeStamp ts)
{
    if (nwos_time_stamp_is_zero(ts))
    {
	print_unknown(19);
    }
    else
    {
	printf("%u-%02u-%02u %02u:%02u:%02u",
	       nwos_extract_year_from_time_stamp(ts),
	       nwos_extract_month_from_time_stamp(ts),
	       nwos_extract_day_of_month_from_time_stamp(ts),
	       nwos_extract_hour_from_time_stamp(ts),
	       nwos_extract_minute_from_time_stamp(ts),
	       nwos_extract_second_from_time_stamp(ts));
    }
}


typedef enum { Option_Error = -1, No_Option, Option_Time, Option_MD5, Option_SHA1, Option_SHA256, Option_SHA512 } Option;

void list_files(Option option)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Path_And_File_Association assoc_obj;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    C_struct_File file_obj;
    ObjRef object_class;
    ObjRef assoc_class_ref;
    ObjRef prev_ref;
    ObjRef next_ref;
    ReferenceList* ref_list;
    int num_refs;
    int num_files = 0;
    int i;
    int j;
    int revision;
    char name[256];
    bool do_revision = false;    /* currently there are no revisions in the public files */


    nwos_log_arguments(0, NULL);   /* disable logging */

    nwos_initialize_objectify(PUBLIC, NULL);

    if (!nwos_find_public_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref))
    {
	printf("ERROR: could not find public 'PATH AND FILE ASSOCIATION CLASS'.");
	nwos_terminate_objectify();
	return;
    }

    nwos_read_class_definition(&assoc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	{
	    printf("  <CODE>");

	    nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj));

	    switch (option)
	    {
	      case Option_MD5:
		print_md5(&assoc_obj.file);
		printf(" &nbsp;");
		break;

	      case Option_SHA1:
		print_sha1(&assoc_obj.file);
		printf(" &nbsp;");
		break;

	      case Option_SHA256:
		print_sha256(&assoc_obj.file);
		printf(" &nbsp;");
		break;

	      case Option_SHA512:
		print_sha512(&assoc_obj.file);
		printf(" &nbsp;");
		break;

	      case Option_Time:
		print_time(assoc_obj.modification_time);
		printf(" &nbsp;");
		break;

	      default:
		break;
	    }

	    nwos_read_variable_sized_object_from_disk(&assoc_obj.path, kludge, sizeof(kludge), &get_path_object_size);

	    /* remember ptr_path_obj points to the kludge buffer */

	    for (j = 0; j < ptr_path_obj->count; j++) name[j] = ptr_path_obj->storage[j];

	    name[j] = '\0';


	    printf("%s", name);

	    if (do_revision)
	    {
		nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj));
		revision = 1;
		while (!is_void_reference(&file_obj.header.object.prev_version))
		{
		    copy_reference(&next_ref, &file_obj.header.common.id);
		    copy_reference(&prev_ref, &file_obj.header.object.prev_version);
		    nwos_read_object_from_disk(&prev_ref, &file_obj, sizeof(file_obj));
		    assert(is_same_object(&file_obj.header.object.next_version, &next_ref));
		    revision++;
		}
		printf(";%d", revision);
	    }

	    printf("</CODE><BR>\n");

	    num_files++;

	    //    memcpy(file_ref, &ptr_path_obj->file, sizeof(*file_ref));
	}
    }

    //    printf("Number of files: %d\n", num_files);

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    nwos_terminate_objectify();
}


int main(int argc, char* argv[])
{
    char* query_string;
    char* title = "ERROR: invalid query";
    Option option = Option_Error;

    printf("Content-type: text/html\n\n");

    printf("<HTML>\n");

    query_string = getenv("QUERY_STRING");

    if (query_string == NULL || query_string[0] == '\0')
    {
	option = No_Option;
	title = "";
    }
    else if (strcmp(query_string, "time") == 0)
    {
	option = Option_Time;
	title = "with Creation Time";
    }
    else if (strcmp(query_string, "md5") == 0)
    {
	option = Option_MD5;
	title = "with MD5 Checksum";
    }
    else if (strcmp(query_string, "sha1") == 0)
    {
	option = Option_SHA1;
	title = "with SHA1 Checksum";
    }
    else if (strcmp(query_string, "sha256") == 0)
    {
	option = Option_SHA256;
	title = "with SHA256 Checksum";
    }
    else if (strcmp(query_string, "sha512") == 0)
    {
	option = Option_SHA512;
	title = "with SHA512 Checksum";
    }

    printf("<HEAD>\n");
    printf("  <TITLE>www.worldwide-database.org: List of Public Files %s</TITLE>\n", title);
    printf("</HEAD>\n");

    printf("<BODY BGCOLOR=\"#ffffff\">\n");

    printf("<P>\n");

    if (option == Option_Error)
    {
	printf("ERROR: invalid QUERY_STRING");
    }
    else
    {
	list_files(option);
    }

    printf("</P>\n");

    printf("</BODY>\n");
    printf("</HTML>\n");

    return 0;
}


