/*
--          This file is part of the New World OS and Objectify projects
--               Copyright (C) 2006, 2007, 2008, 2009  QRW Software
--               J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   For the latest information, source code (SVN), releases, bug and feature
--   request tracking go to:
--      http://sourceforge.net/projects/objectify
--
--   For older bug tracking, releases and source code (CVS) prior to the
--   Alpha_30 release go to:
--      http://sourceforge.net/projects/nwos
--
--   Other related websites:
--      http://www.qrwsoftware.com
--      http://www.worldwide-database.org
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--   $Author: jsedwards $
--   $Date: 2009-08-04 07:13:14 -0600 (Tue, 04 Aug 2009) $
--   $Revision: 4263 $
--
--   NOTE: Subversion does not support the Log keyword so I have removed the
--   logs that were here when I was using CVS.  Use the "svn log" command to
--   see the revision history of this file.  Also this file was created in the
--   alpha_05_branch so check the logs for this file in it too.
--   (See http://subversion.tigris.org/faq.html#log-in-source)
--
*/

#include <assert.h>
#include <dirent.h>
#include <limits.h>    /* define PATH_MAX */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>

#include "bit_map.h"
#include "objectify.h"


#ifdef CHANGE_SECURITY_TO_DENSITY
struct options {
  Security_Level level;
  char* option;
  char* description;
} security_options[] = 
  {
    { Security_Minimal,  "--minimal-security",  "minimal"  },
    { Security_Very_Low, "--very-low-security", "very low" },
    { Security_Low,      "--low-security",      "low"      },
    { Security_Medium,   "--medium-security",   "medium"   },
    { Security_High,     "--high-security",     "high"   },
    { Security_Extreme,  "--extreme-security",  "extreme"  },
  };

#define NUM_OPTIONS (sizeof(security_options)/sizeof(struct options))
#endif

static int result_counts[NUM_OBJ_RESULTS + 1];   /* allow one extra space for files that wouldn't fit */


static void display_usage(char* prog_name)
{
#ifdef CHANGE_SECURITY_TO_DENSITY
    int i;
#endif

    fprintf(stderr, "usage: %s [option] file [option] file...\n", prog_name);
    fprintf(stderr, "   or: %s --help\n", prog_name);
    fprintf(stderr, "\n");
    fprintf(stderr, "  --add-revision\tlink this file to a previous version.\n");
    fprintf(stderr, "  --help\t\tdisplay this message.\n");
    fprintf(stderr, "  -r, --recursive\trecurse through directories.\n");

#ifdef CHANGE_SECURITY_TO_DENSITY
    for (i = 0; i < NUM_OPTIONS; i++)
    {
	fprintf(stderr, "  %s\tuse %s security level for following files.\n",
		security_options[i].option, security_options[i].description);
    }

    fprintf(stderr, "\n");

    fprintf(stderr, "See the file doc/Security_Levels.txt for more info on the security levels.\n");
#endif

    fprintf(stderr, "\n");
}


typedef enum { Do_Not_Add_Revision, Add_Revision, Recursive_Add_Revision} Add_Revision_Type;

void process_file(const char* path, Add_Revision_Type add_revision, bool do_recursive)
{
    struct stat stat_struct;
    DIR* dp;
    struct dirent *dir_entry;
    const char* p;
    char full_path[PATH_MAX];
    int i;
    time_t start_time;
    ObjCreateResult result = ERROR_OCCURRED;
    ObjRef ref;
    ObjRef path_ref;
    bool Unknown_result_in_switch_statement = false;

    assert(strstr(path, "//") == NULL);

    if (lstat(path, &stat_struct) != 0)
    {
	perror(path);
	result_counts[ERROR_OCCURRED]++;
	return;
    }

    if (S_ISDIR(stat_struct.st_mode))
    {
	dp = opendir(path);
	if (dp == NULL)
	{
	    perror(path);
	    result_counts[ERROR_OCCURRED]++;
	    return;
	}

	dir_entry = readdir(dp);
	while (dir_entry != NULL)
	{
	    if (strcmp(dir_entry->d_name, ".") != 0 && strcmp(dir_entry->d_name, "..") != 0)
	    {
		// Construct the full path name

		i = 0;

		if (strcmp(path, ".") != 0)
		{
		    for (p = path; *p != '\0' && i < PATH_MAX; p++) full_path[i++] = *p;

		    if (i < PATH_MAX && full_path[i-1] != '/') full_path[i++] = '/';
		}

		for (p = dir_entry->d_name; *p != '\0' && i < PATH_MAX; p++) full_path[i++] = *p;

		if (i < PATH_MAX)
		{
		    full_path[i] = '\0';
		}
		else
		{
		    fprintf(stderr, "ERROR: path %s/%s exceeds maximum size of a path (PATH_MAX = %d)\n",
			    path, dir_entry->d_name, PATH_MAX);
		    result_counts[ERROR_OCCURRED]++;
		    return;
		}

		assert(strstr(full_path, "//") == NULL);

		if (add_revision == Do_Not_Add_Revision)
		{
		    process_file(full_path, Do_Not_Add_Revision, do_recursive);
		}
		else
		{
		    process_file(full_path, Recursive_Add_Revision, do_recursive);
		}
	    }

	    dir_entry = readdir(dp);
	}

	if (closedir(dp) != 0)
	{
	    perror(path);
	    result_counts[ERROR_OCCURRED]++;
	}
    }    
    else    // must be a regular file
    {
	assert(S_ISREG(stat_struct.st_mode));

	if (nwos_check_blocks_available(nwos_estimate_blocks_for_file(path)))
	{
	    start_time = time(NULL);

	    switch (add_revision)
	    {
	      case Do_Not_Add_Revision:
		result = nwos_create_file(path, &ref, do_recursive); /* allow multiple paths for the same file if recursive */
		break;

	      case Add_Revision:
		result = nwos_add_new_revision_of_file(path, &ref);
		break;

	      case Recursive_Add_Revision:
		if (nwos_find_file_path(path, &path_ref))   /* file with the same name exists */
		{
		    result = nwos_add_new_revision_of_file(path, &ref);
		}
		else
		{
		    result = nwos_create_file(path, &ref, true); /* no file with the same name exists, allow multiple paths */
		}
		break;
	    }

	    result_counts[result]++;

	    switch(result)
	    {
	      case CREATED_NEW:
		printf("file created: %02x%02x%02x%02x   time: %d seconds\n", 
		       ref.id[0], ref.id[1], ref.id[2], ref.id[3], 
		       (int) (time(NULL) - start_time));
		break;

	      case IMPORTED_DATA:
		printf("imported data: %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]);
		break;

	      case CREATED_NEW_REVISION:
		printf("file created: %02x%02x%02x%02x   time: %d seconds\n", 
		       ref.id[0], ref.id[1], ref.id[2], ref.id[3], 
		       (int) (time(NULL) - start_time));
		break;

	      case FOUND_EXISTING:
		printf("existing file found: %02x%02x%02x%02x\n", ref.id[0], ref.id[1], ref.id[2], ref.id[3]);
		break;

	      case DUPLICATE_PATH:
		printf("WARNING: a file by the name of: %s already exists and --add-revision.\n", path);
		printf("         was not specified, it will be skipped!!\n");
		break;

	      case PATH_NOT_FOUND:
		fprintf(stderr, "Error: no previous version of '%s' exists, will be skipped\n", path);
		break;

	      case DUPLICATE_FILE:
		printf("WARNING: a file with the same size, md5, and sha1 sums already exists.\n");
		printf("         In this version it will be skipped!!\n");
		break;

	      case MULTIPLE_FILES:
		fprintf(stderr, "Error: multiple files with the name %s exist, this version", path);
		fprintf(stderr, "cannot cope with them.\n");
		break;

	      case CREATED_NEW_PATH_FOR_EXISTING_FILE:
		printf("Created new path: %s for existing file: %02x%02x%02x%02x\n",
		       path, ref.id[0], ref.id[1], ref.id[2], ref.id[3]);
		break;

	      case ERROR_OCCURRED:
		break;

	      default:   /* keep the compiler form complaining */
		assert(Unknown_result_in_switch_statement);
		break;
	    }

	    nwos_flush_to_disk();
	}
	else  /* not enough space for this file */
	{
	    fprintf(stderr, "Error: not enough space left for this file, skipped\n");
	    result_counts[NUM_OBJ_RESULTS]++;   /* use last slot to count too full errors */
	}
    }
}


int main(int argc, char* argv[])
{
    struct stat stat_struct;
    int i;
    int j;
    char path[PATH_MAX];
    bool do_recursive = false;
    const char* invalid_type;

    for (i = 0; i < (NUM_OBJ_RESULTS + 1); i++) result_counts[i] = 0;

    printf("\n");

    if (argc < 2)
    {
	display_usage(argv[0]);
	exit(1);
    }

    for (i = 1; i < argc; i++)
    {
	if (argv[i][0] == '-')
	{
#ifdef CHANGE_SECURITY_TO_DENSITY
	    for (j = 0; j < NUM_OPTIONS; j++)
	    {
		if (strcmp(argv[i], security_options[j].option) == 0) break;
	    }
#else
#define NUM_OPTIONS 1234567890

	    j = NUM_OPTIONS;
#endif

	    if (j == NUM_OPTIONS)
	    {
		if (strcmp(argv[i], "--add-revision") == 0)
		{
		    i++;

		    if (i == argc || argv[i][0] == '-')
		    {
			fprintf(stderr, "Error: a file must be specified after --add-revision\n");
			exit(1);
		    }
		}
		else if (strcmp(argv[i], "-r") == 0 || strcmp(argv[i], "--recursive") == 0)
		{
		    do_recursive = true;
		}
		else if (strcmp(argv[i], "--help") == 0)
		{
		    display_usage(argv[0]);
		    exit(1);
		}
		else
		{
		    fprintf(stderr, "Unknown option: %s\n", argv[i]);
		    display_usage(argv[0]);
		    exit(1);
		}
	    }
	}
	else
	{
	    if (lstat(argv[i], &stat_struct) != 0)
	    {
		perror(argv[i]);
		exit(1);
	    }

	    invalid_type = nwos_check_invalid_type(stat_struct.st_mode);

	    if (invalid_type != NULL)
	    {
		fprintf(stderr, "ERROR: %s is a %s.\n", argv[i], invalid_type);
	    }

	    if (S_ISDIR(stat_struct.st_mode))    // it's not a normal file, investigate further
	    {
		if (do_recursive)
		{
		    nwos_check_directory_is_all_ok(argv[i]);
		}
		else
		{
		    fprintf(stderr, "ERROR: %s is a directory and -r or --recursive was not specified.\n", argv[i]);
		    exit(1);
		}
	    }    
	    else if (!S_ISREG(stat_struct.st_mode))
	    {
		fprintf(stderr, "ERROR: %s is unrecognized type of file.\n", argv[i]);
		exit(1);
	    }
	}
    }


    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(READ_WRITE, DEFAULT_FILE);

#ifdef CHANGE_SECURITY_TO_DENSITY
    nwos_set_security_level(Security_Low);
#endif

    for (i = 1; i < argc; i++)
    {
#ifdef CHANGE_SECURITY_TO_DENSITY
	nwos_restart_id_generation();   /* make sure each file starts with it's own sequence of id's */
#endif

	if (argv[i][0] == '-')
	{
	    if (strcmp(argv[i], "--add-revision") == 0)
	    {
		i++;    /* point to the file argument */

		assert(i < argc && argv[i][0] != '-');

		if (strlen(argv[i]) + 1 > sizeof(path))
		{
		    fprintf(stderr, "WARNING: the path '%s' is too long for the buffer (exceeds %zd characters) - ignored.\n", argv[i], sizeof(path));
		}
		else if (*argv[i] == '/')
		{
		    fprintf(stderr, "WARNING: the path '%s' is absolute and this version cannot process it - ignored.\n", argv[i]);
		}
		else
		{
		    nwos_normalize_path(path, argv[i], sizeof(path));

		    if (path[0] == '.' && path[1] == '.' && path[2] == '/')
		    {
			fprintf(stderr, "ERROR: this version cannot import relative to the parent directory: %s\n", argv[i]);
			fprintf(stderr, "Will be skipped!\n");
		    }
		    else
		    {
			process_file(path, Add_Revision, do_recursive);
		    }
		}
	    }
	    else
	    {
#ifdef CHANGE_SECURITY_TO_DENSITY
		for (j = 0; j < NUM_OPTIONS; j++)
		{
		    if (strcmp(argv[i], security_options[j].option) == 0) break;
		}

		if (j < NUM_OPTIONS)
		{
		    printf("setting security level to: %s\n", security_options[j].description);

		    nwos_set_security_level(security_options[j].level);
		}
		else
#endif
		{
		    assert(strcmp(argv[i], "-r") == 0 || strcmp(argv[i], "--recursive") == 0);
		}
	    }
	}
	else
	{
	    if (strlen(argv[i]) + 1 > sizeof(path))
	    {
		fprintf(stderr, "WARNING: the path '%s' is too long for the buffer (exceeds %zd characters) - ignored.\n", argv[i], sizeof(path));
	    }
	    else if (*argv[i] == '/')
	    {
		fprintf(stderr, "WARNING: the path '%s' is absolute and this version cannot process it - ignored.\n", argv[i]);
	    }
	    else
	    {
		nwos_normalize_path(path, argv[i], sizeof(path));

		if (path[0] == '.' && path[1] == '.' && path[2] == '/')
		{
		    fprintf(stderr, "ERROR: this version cannot import relative to the parent directory: %s\n", argv[i]);
		    fprintf(stderr, "Will be skipped!\n");
		}
		else
		{
		    process_file(path, Do_Not_Add_Revision, do_recursive);
		}
	    }
	}
    }

    printf("files created:   %d\n", result_counts[CREATED_NEW]);
    printf("imported data:   %d\n", result_counts[IMPORTED_DATA]);
    printf("new revisions:   %d\n", result_counts[CREATED_NEW_REVISION]);
    printf("new paths:       %d\n", result_counts[CREATED_NEW_PATH_FOR_EXISTING_FILE]);
    printf("existing files:  %d\n", result_counts[FOUND_EXISTING]);
    printf("path not found:  %d\n", result_counts[PATH_NOT_FOUND]);
    printf("duplicate paths: %d\n", result_counts[DUPLICATE_PATH]);
    printf("duplicate files: %d\n", result_counts[DUPLICATE_FILE]);
    printf("multiple files:  %d\n", result_counts[MULTIPLE_FILES]);
    printf("files w/errors:  %d\n", result_counts[ERROR_OCCURRED]);
    printf("too full to add: %d\n", result_counts[NUM_OBJ_RESULTS]);

    nwos_terminate_objectify();

    return (result_counts[PATH_NOT_FOUND] |
	    result_counts[DUPLICATE_PATH] |
	    result_counts[DUPLICATE_FILE] |
	    result_counts[MULTIPLE_FILES] |
	    result_counts[ERROR_OCCURRED] |
	    result_counts[NUM_OBJ_RESULTS]) != 0;
}


