/*
--             This file is part of the New World OS project
--                 Copyright (C) 2005-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: dump_obj.c,v $
-- Revision 1.23  2009/03/14 19:14:21  jsedwards
-- Added include of security.h file, renamed read_object_and_decrypt functions
-- to read_block_and_decrypt, and changed parameters to just pass sequence
-- number.
--
-- Revision 1.22  2009/03/13 12:31:18  jsedwards
-- Added include of objectify.h file and added comment about needing disk_io.h
-- just for IVEC_SIZE.
--
-- Revision 1.21  2009/03/08 00:06:18  jsedwards
-- Changed include objectify_private.h to disk_io.h.
--
-- Revision 1.20  2008/09/01 03:13:41  jsedwards
-- Change for new nwos_initialize_objectify calling convention (doesn't pass
-- back root_object_reference anymore) and removed call to nwos_set_root_object
-- because initialize calls it now.
--
-- Revision 1.19  2008/08/31 17:58:26  jsedwards
-- Add assert around call to nwos_read_object_from_disk because it now returns
-- false when it fails, whereas before it would assert itself.
--
-- Revision 1.18  2008/08/30 12:46:09  jsedwards
-- Removed code and variables to read pass phrase and pass it to initialize,
-- and change parameters passed to initialize.
--
-- Revision 1.17  2008/08/02 11:02:24  jsedwards
-- Added call to log arguments to disable logging and fixed typo in print
-- statement text.
--
-- Revision 1.16  2008/08/01 14:36:54  jsedwards
-- Added code to print class name and to dump object data in hex.
--
-- Revision 1.15  2008/08/01 00:35:04  jsedwards
-- Changed to try old decryption algorithm if checksum fails after reading
-- with new decryption algorithm.
--
-- Revision 1.14  2008/07/19 14:19:29  jsedwards
-- Removed some unused variables.
--
-- Revision 1.13  2008/07/19 10:58:05  jsedwards
-- Added printing of header and object checksums.
--
-- Revision 1.12  2008/07/15 13:21:34  jsedwards
-- Added code to allow dumping object from compressed file, changed to use
-- strtoul instead of strtol for converting reference id, and added new line
-- character to usage.
--
-- Revision 1.11  2008/07/15 12:45:19  jsedwards
-- Updated to work with current version, was at version 0004 - 29-Dec-2005.
--
-- Revision 1.10  2007/07/13 21:04:32  jsedwards
-- Change HEADER_VERSION (which wasn't used anywhere else) to VERSION_STRING
-- which is defined in objectify.h.
--
-- Revision 1.9  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.8  2006/12/01 14:37:23  jsedwards
-- Fix the year in the copyright.
--
-- Revision 1.7  2006/11/11 12:01:03  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.6  2006/10/26 01:51:26  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.5.2.1  2006/10/25 12:22:28  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.5  2005/12/29 14:33:14  jsedwards
-- Updated to work with version Alpha_04 (0004) WITHOUT encryption turned on!
--
-- Revision 1.4  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.3  2005/12/10 15:03:36  jsedwards
-- Fixed header to say the GPL is in the LICENSE file instead of COPYING.
--
-- Revision 1.2  2005/12/02 14:12:09  jsedwards
-- Modified for new header layouts.
--
-- Revision 1.1.1.1  2005/11/25 12:44:27  jsedwards
-- Copied from 'lab'.
--
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "crc32.h"
#include "disk_io.h"       /* define IVEC_SIZE */
#include "objectify.h"
#include "security.h"


#define SIZE_EXTRA_BLOCK_DATA (FILE_BLOCK_SIZE - sizeof(uint32) - sizeof(ObjRef) - sizeof(ObjRef))

typedef struct
{
    uint8  flags[4];
    ObjRef id;
    uint8  data[SIZE_EXTRA_BLOCK_DATA];
    ObjRef next_block;
} Extra_Data_Block;


void print_reference(char* name, ObjRef* ref)
{
  int i;

  printf("%s ", name);

  for (i = 0; i < 4; i++) printf("%02x", ref->id[i]);
}


int main(int argc, char* argv[])
{
    ObjRef ref;
    uint32 hex;
    char* path = DEFAULT_FILE;
    char* endptr;
    uint8 ivec[IVEC_SIZE];
    int seq = 0;
    Extra_Data_Block extra;
    EveryObject* header;
    uint8 chksum[4];
    C_struct_Class_Definition class_def_obj;
    char class_name[64];
    size_t size;
    uint8 *data;
    int i;

    nwos_log_arguments(0, NULL);   /* disable logging */

    printf("\n");

    if (argc < 2 || argc == 3 || argc > 4 || (argc == 2 && *argv[1] == '-') || (argc == 4 && (strcmp(argv[1], "--compressed") != 0 || *argv[2] == '-' || *argv[3] == '-')))
    {
	fprintf(stderr, "usage: %s [--compressed file] object_id_in_hexadecimal\n", argv[0]);
	exit(1);
    }

    hex = strtoul(argv[argc-1], &endptr, 16);

    if (argc == 4)
    {
	path = argv[2];
    }

    if (*endptr != '\0')
    {
	fprintf(stderr, "Invalid reference: %s\n", argv[1]);
	exit(1);
    }

    nwos_word_to_ref(hex, &ref);

    if (nwos_reference_type(&ref) == Public_Reference)
    {
	nwos_initialize_objectify(PUBLIC, NULL);
    }
    else
    {
	nwos_initialize_objectify(READ_ONLY, path);
    }


    //    nwos_read_object_headers_from_disk(ObjRef* ref, EveryObject* header);

    printf("Id: %08x\n", nwos_ref_to_word(&ref));

    memset(ivec, 0, sizeof(ivec));

    if (!nwos_read_block_from_disk_and_decrypt(&ref, &extra, sizeof(extra), ivec, seq))
    {
	fprintf(stderr, "Error: nwos_read_object_from_disk_and_decrypt failed\n");
	exit(1);
    }

    header = (EveryObject*) &extra;

    nwos_crc32_calculate((uint8*)&header->object, sizeof(header->object), chksum);

    if (memcmp(&header->common.header_chksum, chksum, 4) != 0)
    {
	fprintf(stderr, "Warning: checksum with new decryption algorithm failed, trying old algorithm...\n");

	memset(ivec, 0, sizeof(ivec));

	if (!nwos_read_block_from_disk_and_old_decrypt(&ref, &extra, sizeof(extra), ivec, seq))
	{
	    fprintf(stderr, "Error: nwos_read_object_from_disk_and_old_decrypt failed\n");
	    exit(1);
	}

	nwos_crc32_calculate((uint8*)&header->object, sizeof(header->object), chksum);

	if (memcmp(&header->common.header_chksum, chksum, 4) != 0)
	{
	    fprintf(stderr, "Error: checksum with old decryption algorithm failed as well.\n");
	}
    }

    printf("Creation: %s\n", nwos_time_stamp_to_string(header->common.creation_time));

    printf("Checksums:\n");
    printf("  Header: %02x%02x%02x%02x\n", header->common.header_chksum[0], header->common.header_chksum[1],
	   header->common.header_chksum[2], header->common.header_chksum[3]);
    printf("    Data: %02x%02x%02x%02x\n", header->common.data_chksum[0], header->common.data_chksum[1],
	   header->common.data_chksum[2], header->common.data_chksum[3]);

    nwos_read_class_definition(&header->common.class_definition, &class_def_obj);
    nwos_name_to_string(&class_def_obj.name, class_name, sizeof(class_name));

    printf("Class: %08x - %s\n", nwos_ref_to_word(&header->common.class_definition), class_name);

    printf("Clone Of: %08x\n", nwos_ref_to_word(&header->object.clone_of));
    printf("References: %08x\n", nwos_ref_to_word(&header->object.references));
    printf("Context: %08x\n", nwos_ref_to_word(&header->object.context));
    printf("Creation: %08x\n", nwos_ref_to_word(&header->object.creation));
    printf("Source: %08x\n", nwos_ref_to_word(&header->object.source));
    printf("Prev_version: %08x\n", nwos_ref_to_word(&header->object.prev_version));
    printf("Next_version: %08x\n", nwos_ref_to_word(&header->object.next_version));

    size = nwos_get_object_size(&extra);

    data = malloc(size);

    if (data == NULL)
    {
	perror("allocating space for object");
	exit(1);
    }

    assert(nwos_read_object_from_disk(&ref, data, size));

    printf("\n");

    for (i = sizeof(EveryObject); i < size; i++)
    {
	printf("%02x", data[i]);

	if (i % 16 == 3 || i % 16 == 7 || i % 16 == 11)
	{
	    printf("  ");
	}
	else if (i % 16 == 15)
	{
	    printf("\n");
	}
	else
	{
	    printf(" ");
	}
    }

    printf("\n");

    nwos_terminate_objectify();

    return 0;
}


