/*
--             This file is part of the New World OS project
--                 Copyright (C) 2004-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: chunk_info.c,v $
-- Revision 1.7  2009/03/14 23:14:04  jsedwards
-- Added include of the new chunk_info.h file.
--
-- Revision 1.6  2009/03/14 11:46:46  jsedwards
-- Added include of gen_id.h file.
--
-- Revision 1.5  2009/03/13 12:14:18  jsedwards
-- Added includes for log.h and mem_alloc.h files.
--
-- Revision 1.4  2009/03/08 00:06:18  jsedwards
-- Changed include objectify_private.h to disk_io.h.
--
-- Revision 1.3  2009/03/04 14:56:25  jsedwards
-- Moved uint32_ref_to_info_index function from disk_io.c and renamed to
-- nwos_uint32_ref_to_info_index.
--
-- Revision 1.2  2009/03/02 15:32:11  jsedwards
-- Changed so allocate_new_chunk only calls write_empty_chunk if the archive
-- is stored in a block device.
--
-- Revision 1.1  2009/03/01 18:02:48  jsedwards
-- New file created from code taken from the disk_io.c file.
--
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "chunk_info.h"
#include "disk_io.h"
#include "gen_id.h"
#include "log.h"
#include "mem_alloc.h"


/* Warning: if the total_number_of_chunks changes, all of these have to be reallocated. */

Chunk_Info*    nwos_chunk_info;

static int*    chunk_info_reverse_index;   /* index from chunk index to info table */
static uint16* chunk_used_index;  /* order of chunks sorted by blocks used, fewest to most used */

static uint32 capacity;
static bool chunk_info_modified;



/*************************************************************************************/
/* This updates the chunk_info_reverse_index table to point to the chunk_info table. */
/*                                                                                   */
/*  chunk_info   reverse                                                             */
/*    0 [5]       0 [4]                                                              */
/*    1 [3]       1 [5]                                                              */
/*    2 [4]       2 [3]                                                              */
/*    3 [2]       3 [1]                                                              */
/*    4 [0]       4 [2]                                                              */
/*    5 [1]       5 [0]                                                              */
/*                                                                                   */
/*************************************************************************************/

static void update_chunk_info_reverse_index()
{
    int i;

    assert(capacity > 0);

    for (i = 0; i < nwos_used_private_chunks; i++)
    {
	assert(nwos_chunk_info[i].index < nwos_used_private_chunks);
	chunk_info_reverse_index[nwos_chunk_info[i].index] = i;
    }

    /* verify it is correct */
    for (i = 0; i < nwos_used_private_chunks; i++)
    {
	assert(nwos_chunk_info[chunk_info_reverse_index[i]].index == i);
    }
}


/****************************************/
/* Read the chunk_info table from disk. */
/****************************************/

void nwos_initialize_chunk_info(int private_file_desc)
{
    size_t chunk_info_size_in_bytes;
    uint32 used_blocks;
    int i;
    int j;
    int k;
    uint16 left;
    uint16 right;
    uint16 save;
    char log_msg[256];

    assert(capacity == 0);
    assert(nwos_chunk_info == NULL);
    assert(chunk_info_reverse_index == NULL);
    assert(chunk_used_index == NULL);

    capacity = nwos_total_private_chunks;   /* save the size of the chunk_info table */

    if (lseek(private_file_desc, FILE_BLOCK_SIZE, SEEK_SET) < 0)
    {
	perror("chunk index");
	exit(1);
    }

    chunk_info_size_in_bytes = capacity * sizeof(Chunk_Info);
    nwos_chunk_info = nwos_malloc(chunk_info_size_in_bytes);

    if (read(private_file_desc, nwos_chunk_info, chunk_info_size_in_bytes) != chunk_info_size_in_bytes)
    {
	snprintf(log_msg, sizeof(log_msg), "reading chunk info from: %s", nwos_private_path);
	perror(log_msg);
	exit(1);
    }

    used_blocks = 0;
    for (i = 0; i < (int)capacity; i++)
    {
#ifndef WORDS_BIGENDIAN
	nwos_chunk_info[i].ref = byteswap_uint32(nwos_chunk_info[i].ref);
	nwos_chunk_info[i].used = byteswap_uint16(nwos_chunk_info[i].used);
	nwos_chunk_info[i].index = byteswap_uint16(nwos_chunk_info[i].index);
#endif
	used_blocks += nwos_chunk_info[i].used;

	if (nwos_chunk_info[i].ref > MAXIMUM_PRIVATE_REFERENCE)
	{
	    snprintf(log_msg, sizeof(log_msg), 
		     "WARNING: you have a chunk of data located at %08x that is above %08x",
		     nwos_chunk_info[i].ref, MAXIMUM_PRIVATE_REFERENCE);
	    nwos_log(log_msg);
	    fprintf(stderr, "%s,\n", log_msg);
	    fprintf(stderr, "which is used for temporary storage in versions after Alpha_29.4.  Please\n"
		            "e-mail me at qrwsoftware@gmail.com for help fixing this.\n\n");
	    sleep(4);
	}
    }

    if (used_blocks != nwos_used_private_blocks)
    {
	snprintf(log_msg, sizeof(log_msg), 
		 "Warning: calculated sum of used blocks (%u) doesn't match stored (%u)",
		 used_blocks, nwos_used_private_blocks);
	nwos_log(log_msg);
	fprintf(stderr, "%s\n", log_msg);
    }

    chunk_info_modified = false;

    chunk_info_reverse_index = nwos_malloc(capacity * sizeof(int));

    update_chunk_info_reverse_index();

    /* now create the chunk_used_index */

    chunk_used_index = nwos_malloc(capacity * sizeof(uint16));

    /* insert them into a heap to do a heap sort */
    for (i = 0; i < (int)nwos_used_private_chunks; i++)
    {
	j = i + 1;
	while (j / 2 > 0 && nwos_chunk_info[chunk_used_index[(j / 2) - 1]].used < nwos_chunk_info[i].used)
	{
	    /* parent is smaller, move it lower in the tree */
	    chunk_used_index[j - 1] = chunk_used_index[(j / 2) - 1];
	    j = j / 2;
	}

	chunk_used_index[j - 1] = i;
    }

    for (i = (int)nwos_used_private_chunks; i > 1; i--)
    {
	/* save the index at the top of the tree (smallest) */
	save = chunk_used_index[i - 1];

	/* move the root of the the (largest) to the top */
	chunk_used_index[i - 1] = chunk_used_index[0];

	j = 1;
	while (j < (i / 2) - 1)
	{
	    k = j * 2;

	    /* if there is a right child */
	    if (k < i - 1)
	    {
		left = nwos_chunk_info[chunk_used_index[k - 1]].used;
		right = nwos_chunk_info[chunk_used_index[k + 1 - 1]].used;

		if (left < right)
		{
		    k++;
		}
	    }

	    if (nwos_chunk_info[save].used >= nwos_chunk_info[chunk_used_index[k-1]].used) break;

	    chunk_used_index[j - 1] = chunk_used_index[k - 1];

	    j = k;
	}

	chunk_used_index[j - 1] = save;
    }
}


/*********************************************************************/
/* If the chunk_info table has been modified, write it back to disk. */
/*********************************************************************/

void nwos_terminate_chunk_info(int private_file_desc)
{
    int i;
#ifndef PUBLIC_MODE
    size_t chunk_info_size_in_bytes;
#endif
    char log_msg[128];

    if (chunk_info_modified)
    {
	snprintf(log_msg, sizeof(log_msg), "  chunk index modified - new size: %u", nwos_used_private_chunks);
	nwos_log(log_msg);

	chunk_info_size_in_bytes = capacity * sizeof(Chunk_Info);

#ifndef WORDS_BIGENDIAN
	for (i = 0; i < (int)capacity; i++)
	{
	    nwos_chunk_info[i].ref = byteswap_uint32(nwos_chunk_info[i].ref);
	    nwos_chunk_info[i].used = byteswap_uint16(nwos_chunk_info[i].used);
	    nwos_chunk_info[i].index = byteswap_uint16(nwos_chunk_info[i].index);
	}
#endif
	if (lseek(private_file_desc, (off_t)FILE_BLOCK_SIZE, SEEK_SET) < 0)
	{
	    perror(nwos_private_path);
	    exit(1);
	}

	if (write(private_file_desc, nwos_chunk_info, chunk_info_size_in_bytes) != chunk_info_size_in_bytes)
	{
	    perror(nwos_private_path);
	    exit(1);
	}

	nwos_uint32_to_4_uint8(&nwos_used_private_chunks, nwos_disk_header.used_chunks);

	chunk_info_modified = false;
    }

    nwos_free(nwos_chunk_info);
    nwos_chunk_info = NULL;

    nwos_free(chunk_info_reverse_index);
    chunk_info_reverse_index = NULL;

    nwos_free(chunk_used_index);
    chunk_used_index = NULL;

    capacity = 0;
}


/********************************************/
/* Functions to update the block used count */
/********************************************/

void nwos_increment_chunk_info_block_used(int info_index)
{
    assert(0 <= info_index && info_index < nwos_used_private_chunks);
    assert(nwos_chunk_info[info_index].used < USABLE_BLOCKS_PER_CHUNK);

    nwos_chunk_info[info_index].used++;

    chunk_info_modified = true;
}


void nwos_decrement_chunk_info_block_used(int info_index)
{
    assert(0 <= info_index && info_index < nwos_used_private_chunks);
    assert(nwos_chunk_info[info_index].used < USABLE_BLOCKS_PER_CHUNK);

    nwos_chunk_info[info_index].used--;

    chunk_info_modified = true;
}


void nwos_reset_chunk_info_blocks_used(int info_index, uint32 blocks_used)
{
    char log_msg[128];

    assert(0 <= info_index && info_index < nwos_used_private_chunks);
    assert(0 <= blocks_used && blocks_used <= USABLE_BLOCKS_PER_CHUNK);

    snprintf(log_msg, sizeof(log_msg),
	     "Warning: blocks used in chunk mismatch - map: %u  stored: %u, updated stored to match.",
	     blocks_used, nwos_chunk_info[info_index].used);
    nwos_log(log_msg);
    fprintf(stderr, "%s\n", log_msg);

    nwos_chunk_info[info_index].used = blocks_used;

    chunk_info_modified = true;
}


/**********************************************************************************************/
/* This function returns the index into the chunk_info table for the chunk index passed in or */
/* -1 if not found.                                                                           */
/**********************************************************************************************/

int nwos_chunk_index_to_info_index(uint16 chunk_index)
{
    assert(chunk_index < nwos_used_private_chunks);

    return chunk_info_reverse_index[chunk_index];
}


/**********************************************************************************************/
/* This function returns the index into the chunk_info table for the reference passed in or   */
/* -1 if not found.                                                                           */
/**********************************************************************************************/

#ifndef PUBLIC_MODE
int nwos_uint32_ref_to_info_index(uint32 ref)
{
    int lwr;
    int upr;
    int mid = 0;

    assert(nwos_used_private_chunks > 0);
    assert(MINIMUM_PRIVATE_REFERENCE <= ref && ref <= MAXIMUM_PRIVATE_REFERENCE);

    lwr = 1;

    upr = nwos_used_private_chunks;

    while (lwr <= upr)
    {
	mid = (upr + lwr) / 2;

	if (ref < nwos_chunk_info[mid - 1].ref)
	{
	    upr = mid - 1;
	}
	else if (ref >= nwos_chunk_info[mid - 1].ref + USABLE_BLOCKS_PER_CHUNK)
	{
	    lwr = mid + 1;
	}
	else
	{
	    mid = mid - 1;  /* adjust to 0 based index */
	    break;
	}
    }

    if (lwr <= upr)
    {
	assert(0 <= mid && mid < nwos_used_private_chunks);
	assert(nwos_chunk_info[mid].ref <= ref && ref < nwos_chunk_info[mid].ref + USABLE_BLOCKS_PER_CHUNK);

	return mid;
    }

    return -1;   /* not found */
}
#endif


/*************************************/
/* This function creates a new chunk */
/*************************************/

void nwos_allocate_new_chunk(uint32 ref)
{
    uint32 offset;
    int i;

    assert(nwos_used_private_chunks < nwos_total_private_chunks);

    assert(MINIMUM_PRIVATE_REFERENCE <= ref && ref <= MAXIMUM_PRIVATE_REFERENCE);

    assert(nwos_hash_uint32_ref(ref) == 0);

    /* first make sure the total_private_chunks hasn't been increased (if it is a file). */

    if (capacity != nwos_total_private_chunks)
    {
	nwos_chunk_info = nwos_realloc(nwos_chunk_info, nwos_total_private_chunks * sizeof(Chunk_Info));

	chunk_info_reverse_index = nwos_realloc(chunk_info_reverse_index, nwos_total_private_chunks * sizeof(int));

	chunk_used_index = nwos_realloc(chunk_used_index, nwos_total_private_chunks * sizeof(uint16));

	for (i = capacity; i < nwos_total_private_chunks; i++)
	{
	    memset(&nwos_chunk_info[i], 0, sizeof(Chunk_Info));
	    chunk_info_reverse_index[i] = 0;
	    chunk_used_index[i] = 0;
	}

	capacity = nwos_total_private_chunks;
    }

    /* figure out where this ref goes in the chunk_info table */
    for (i = nwos_used_private_chunks - 1; i >= 0; i--)
    {
	if (nwos_chunk_info[i].ref < ref) break;

	nwos_chunk_info[i+1].ref = nwos_chunk_info[i].ref;
	nwos_chunk_info[i+1].used = nwos_chunk_info[i].used;
	nwos_chunk_info[i+1].index = nwos_chunk_info[i].index;
    }
    i++;

    offset = (ref - MINIMUM_PRIVATE_REFERENCE) % USABLE_BLOCKS_PER_CHUNK;

    nwos_chunk_info[i].ref = ref - offset;

    assert(i == 0 || nwos_chunk_info[i-1].ref < nwos_chunk_info[i].ref);
    assert(i == nwos_used_private_chunks || nwos_chunk_info[i+1].ref > nwos_chunk_info[i].ref);

    /* make sure chunk_info.ref is an even multiple of USABLE_BLOCKS_PER_CHUNK */

    offset = nwos_chunk_info[i].ref - MINIMUM_PRIVATE_REFERENCE;

    nwos_chunk_info[i].used = 0;
    nwos_chunk_info[i].index = nwos_used_private_chunks;

    assert(nwos_chunk_info[i].index < nwos_total_private_chunks);

    assert((offset % USABLE_BLOCKS_PER_CHUNK) == 0);

    if (nwos_archive_is_block_device())
    {
	nwos_write_empty_chunk(nwos_chunk_info[i].index);
    }

    nwos_used_private_chunks += 1;

    update_chunk_info_reverse_index();

    chunk_info_modified = true;
}


