/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: list_files.c,v $
-- Revision 1.35  2008/09/01 03:13:43  jsedwards
-- Change for new nwos_initialize_objectify calling convention (doesn't pass
-- back root_object_reference anymore) and removed call to nwos_set_root_object
-- because initialize calls it now.
--
-- Revision 1.34  2008/08/31 21:53:52  jsedwards
-- Added an assert around calls to nwos_read_variable_sized_object_from_disk
-- and nwos_read_object_from_disk because now they return false when they fail
-- instead of asserting themselves.
--
-- Revision 1.33  2008/08/30 12:46:13  jsedwards
-- Removed code and variables to read pass phrase and pass it to initialize,
-- and change parameters passed to initialize.
--
-- Revision 1.32  2008/06/01 04:10:49  jsedwards
-- Fix assert to verify the correct reference for the previous version.
--
-- Revision 1.31  2008/05/31 14:37:54  jsedwards
-- Fix so that revision is always initialized.
--
-- Revision 1.30  2008/05/31 14:32:07  jsedwards
-- Fixed so that old name gets the ;0 revision NOT the new name.
--
-- Revision 1.29  2008/05/31 13:47:56  jsedwards
-- Changed code to flag old revision files that were a name change, with a ;0
-- revision.
--
-- Revision 1.28  2008/05/22 11:35:58  jsedwards
-- Changed so that it deals with pre Alpha_28 revised files, but warns of
-- problem at the end.
--
-- Revision 1.27  2008/05/21 13:26:40  jsedwards
-- Added code to deal with revisied file links being in PATH AND FILE
-- ASSOCIATION objects instead of FILE objects.  Still need to figure out
-- how to deal with reading compressed files that are the old way.
--
-- Revision 1.26  2008/04/09 12:35:21  jsedwards
-- Added call to nwos_log_arguments to disable logging.
--
-- Revision 1.25  2008/03/14 19:11:19  jsedwards
-- Added --sha512 option.
--
-- Revision 1.24  2008/02/16 17:34:34  jsedwards
-- Added if statements in print_sha1 to print "not available" if it is void in
-- the file object and print_time to print "unknown" if modification time is
-- zero.
--
-- Revision 1.23  2008/02/11 04:15:12  jsedwards
-- Added the --public option.
--
-- Revision 1.22  2008/02/03 01:05:23  jsedwards
-- Changed DEFAULT_TYPE_RO to READ_ONLY.
--
-- Revision 1.21  2008/01/22 13:40:03  jsedwards
-- Updated year in copyright.
--
-- Revision 1.20  2008/01/22 13:30:33  jsedwards
-- Added missing options to usage print and expanded to explain options.
--
-- Revision 1.19  2007/12/24 02:25:51  jsedwards
-- Added --time and --revision options.
--
-- Revision 1.18  2007/08/25 22:48:27  jsedwards
-- Fixed option name in usage print statement.
--
-- Revision 1.17  2007/08/25 21:55:09  jsedwards
-- Added --list-old-files option and code to only print the old names of files
-- that have been renamed if that option is set.
--
-- Revision 1.16  2007/08/23 13:43:58  jsedwards
-- Removed unused code to read File Path class reference and emit error message
-- when there isn't a private File and Path Association class (meaning no files
-- exist in the system), instead of asserting.
--
-- Revision 1.15  2007/08/17 17:59:52  jsedwards
-- Added code to deal with the case where a file object doesn't have a SHA256
-- checksum, but the --sha256 option is enabled.
--
-- Revision 1.14  2007/08/17 16:48:54  jsedwards
-- Added code to optionally print SHA256 checksums.
--
-- Revision 1.13  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.12  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.11  2007/01/14 22:22:03  jsedwards
-- Add --sha1 option to do print SHA1 checksum instead of MD5.
--
-- Revision 1.10  2007/01/10 13:58:54  jsedwards
-- Changed to scan the path_and_file_association objects instead of the
-- file_path objects.
--
-- Revision 1.9  2007/01/09 13:06:18  jsedwards
-- Change to use new path_and_file_association object.
--
-- Revision 1.8  2006/12/21 13:08:36  jsedwards
-- Hack to make it compile with new split public and private classes,
-- NON-FUNCTIONAL!
--
-- Revision 1.7  2006/12/19 14:52:18  jsedwards
-- Kludged version to compile with new 'file path' structure without 'file'
-- feature.  DOES NOT function correctly!
--
-- Revision 1.6  2006/12/02 03:44:03  jsedwards
-- Change to print md5 sums when there are multiple files with the same name.
--
-- Revision 1.5  2006/12/01 14:31:30  jsedwards
-- Changed to use new malloc_reference_list and free_reference_list functions
-- instead of inlining the code.
--
-- Revision 1.4  2006/12/01 05:20:26  jsedwards
-- Changed to handle the case where a file path points to more than one file.
-- The case where printing MD5 sums is not handled yet.
--
-- Revision 1.3  2006/11/18 15:09:10  jsedwards
-- Added "max_size" parameter to read_variable_sized_object_from_disk because
-- objects are no longer limited to one file block.
--
-- Revision 1.2  2006/10/26 01:51:27  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.1.2.4  2006/10/25 12:22:28  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.1.2.3  2006/10/07 12:56:27  jsedwards
-- Added file name globing so you specify which files to list.
--
-- Revision 1.1.2.2  2006/10/07 12:27:59  jsedwards
-- Added the ability to print md5sums in the listing with the --md5 option.
--
-- Revision 1.1.2.1  2006/10/06 12:06:57  jsedwards
-- Program to list all files (actually file_paths) in system.
--
*/

#include <fnmatch.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify_private.h"   /* do private so that nwos_get_public_objects_path is defined */
#include "time_stamp.h"


static size_t get_path_object_size(void* file_path_obj)
{
    assert(((C_struct_File_Path*)file_path_obj)->count > 0);

    return sizeof(C_struct_File_Path) + ((C_struct_File_Path*)file_path_obj)->count;
}


void print_md5(ObjRef* file_ref)
{
    C_struct_File file_obj;
    C_struct_MD5sum md5_object;
    int j;

    assert(nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj)));  /* read the file object */
    assert(nwos_read_object_from_disk(&file_obj.md5sum, &md5_object, sizeof(md5_object)));

    for (j = 0; j < MD5_DIGEST_SIZE; j++) printf("%02x", md5_object.md5sum[j]);
    printf("  ");
}


void print_sha1(ObjRef* file_ref)
{
    C_struct_File file_obj;
    C_struct_SHA1sum sha1_object;
    int j;

    assert(nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj)));  /* read the file object */

    if (is_void_reference(&file_obj.sha1sum))
    {
             /*          1         2         3         4 */
             /* 1234567890123456789012345678901234567890 */

	printf("* No SHA1 checksum stored for this file * ");
    }
    else
    {
	assert(nwos_read_object_from_disk(&file_obj.sha1sum, &sha1_object, sizeof(sha1_object)));

	for (j = 0; j < SHA1_DIGEST_SIZE; j++) printf("%02x", sha1_object.sha1sum[j]);
	printf("  ");
    }
}


void print_sha256(ObjRef* file_ref)
{
    C_struct_File file_obj;
    C_struct_SHA256sum sha256_object;
    int j;

    assert(nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj)));  /* read the file object */

    if (is_void_reference(&file_obj.sha256sum))
    {
             /*          1         2         3         4         5         6     */
             /* 1234567890123456789012345678901234567890123456789012345678901234 */

	printf("*******  No SHA256 checksum is available for this file  ******* ");
    }
    else
    {
	assert(nwos_read_object_from_disk(&file_obj.sha256sum, &sha256_object, sizeof(sha256_object)));

	for (j = 0; j < SHA256_DIGEST_SIZE; j++) printf("%02x", sha256_object.sha256sum[j]);
    }

    printf("  ");
}


void print_sha512(ObjRef* file_ref)
{
    C_struct_File file_obj;
    C_struct_SHA512sum sha512_object;
    int j;

    assert(nwos_read_object_from_disk(file_ref, &file_obj, sizeof(file_obj)));  /* read the file object */

    if (is_void_reference(&file_obj.sha512sum))
    {
	printf("***************************************  No SHA512 checksum is available for this file  *************************************** ");
    }
    else
    {
	assert(nwos_read_object_from_disk(&file_obj.sha512sum, &sha512_object, sizeof(sha512_object)));

	for (j = 0; j < SHA512_DIGEST_SIZE; j++) printf("%02x", sha512_object.sha512sum[j]);
    }

    printf("  ");
}


void print_time(TimeStamp ts)
{
    if (nwos_time_stamp_is_zero(ts))
    {
	printf("  unknown");
    }
    else
    {
	printf("  %u-%02u-%02u %02u:%02u:%02u",
	       nwos_extract_year_from_time_stamp(ts),
	       nwos_extract_month_from_time_stamp(ts),
	       nwos_extract_day_of_month_from_time_stamp(ts),
	       nwos_extract_hour_from_time_stamp(ts),
	       nwos_extract_minute_from_time_stamp(ts),
	       nwos_extract_second_from_time_stamp(ts));
    }
}


int main(int argc, char* argv[])
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Path_And_File_Association assoc_obj;
    C_struct_Path_And_File_Association prev_assoc_obj;
    C_struct_Path_And_File_Association next_assoc_obj;
    uint8 kludge[FILE_BLOCK_SIZE];
    C_struct_File_Path* ptr_path_obj = (C_struct_File_Path*)kludge;
    C_struct_File file_obj;
    ObjRef object_class;
    ObjRef assoc_class_ref;
    ObjRef prev_ref;
    ObjRef next_ref;
    ReferenceList* ref_list;
    int num_refs;
    int num_files = 0;
    int argn;
    int i;
    int j;
    int revision;
    int old_file_revisions = 0;
    char name[256];
    bool do_md5 = false;
    bool do_sha1 = false;
    bool do_sha256 = false;
    bool do_sha512 = false;
    bool do_list_old_names = false;
    bool do_time = false;
    bool do_revision = false;
    bool do_public = false;
    bool match;


    nwos_log_arguments(0, NULL);   /* disable logging */

    char* path = DEFAULT_FILE;

    argn = 1;

    while (argn < argc && *argv[argn] == '-')
    {
	if (strcmp(argv[argn], "--compressed") == 0 && argc > argn + 1)
	{
	    argn++;
	    path = argv[argn];
	    argn++;
	}
	else if (strcmp(argv[argn], "--md5") == 0)
	{
	    do_md5 = true;
	    argn++;
	}
	else if (strcmp(argv[argn], "--public") == 0)
	{
	    do_public = true;
	    argn++;
	}
	else if (strcmp(argv[argn], "--sha1") == 0)
	{
	    do_sha1 = true;
	    argn++;
	}
	else if (strcmp(argv[argn], "--sha256") == 0)
	{
	    do_sha256 = true;
	    argn++;
	}
	else if (strcmp(argv[argn], "--sha512") == 0)
	{
	    do_sha512 = true;
	    argn++;
	}
	else if (strcmp(argv[argn], "--time") == 0)
	{
	    do_time = true;
	    argn++;
	}
	else if (strcmp(argv[argn], "--list-old-names") == 0)
	{
	    do_list_old_names = true;
	    argn++;
	}
	else if (strcmp(argv[argn], "--revision") == 0)
	{
	    do_revision = true;
	    argn++;
	}
	else
	{
	    fprintf(stderr, "usage: %s [--public] [--compressed compressed-file] [--time] [--revision]\n", argv[0]);
	    fprintf(stderr, "  [--md5 | --sha1 | --sha256 | --sha512] [--list-old-names] [FILES]...\n");
	    fprintf(stderr, "\n");
	    fprintf(stderr, "List information about the files stored or logged in Objectify.\n");
	    fprintf(stderr, "If no files are specified, all files stored and logged are printed.\n");
	    fprintf(stderr, "\n");
	    fprintf(stderr, "  --compresed file  read from 'file' instead of normal storage\n");
	    fprintf(stderr, "  --list-old-names  print old names of files (that have been renamed)\n");
	    fprintf(stderr, "  --md5             print md5 sum of each file\n");
	    fprintf(stderr, "  --public          list public files instead of private\n");
	    fprintf(stderr, "  --revision        print revision of each file\n");
	    fprintf(stderr, "  --sha1            print sha1 sum of each file\n");
	    fprintf(stderr, "  --sha256          print sha256 sum of each file\n");
	    fprintf(stderr, "  --sha512          print sha512 sum of each file\n");
	    fprintf(stderr, "  --time            print modification time for each file\n");
	    fprintf(stderr, "\n");
	    fprintf(stderr, "Only one of --md5, --sha1, --sha256, --sha512 can be specified at a time.\n\n");
	    exit(1);
	}

    }

    if (do_public && path != NULL)   /* both --public and --compressed were specified */
    {
	fprintf(stderr, "WARNING: because --public was specified --compressed file is ignored!\n");
	path = NULL;
    }

    printf("\n");

    if (do_public)
    {
	nwos_initialize_objectify(PUBLIC, NULL);

	if (!nwos_find_public_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref))
	{
	    printf("No public files found, please update your Public Objects file: %s!\n", nwos_get_public_objects_path());
	    exit(1);
	}
    }
    else
    {
	nwos_initialize_objectify(READ_ONLY, path);

	if (!nwos_find_private_class_definition("PATH AND FILE ASSOCIATION", &assoc_class_ref))
	{
	    printf("No files are stored in the system!\n");
	    exit(1);
	}
    }

    nwos_read_class_definition(&assoc_class_ref, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_refs = ref_list->common_header.num_refs;

    printf("num_refs: %d\n", num_refs);

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);

	if (is_same_object(&object_class, &assoc_class_ref))
	  {
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &assoc_obj, sizeof(assoc_obj)));

	    match = false;

	    if (do_list_old_names || do_revision || is_void_reference(&assoc_obj.header.object.next_version))
	    {
		assert(nwos_read_variable_sized_object_from_disk(&assoc_obj.path, kludge, sizeof(kludge), &get_path_object_size));

		/* remember ptr_path_obj points to the kludge buffer */

		for (j = 0; j < ptr_path_obj->count; j++) name[j] = ptr_path_obj->storage[j];

		name[j] = '\0';

		if (argn < argc)    /* file names were passed to the program on the command line */
		{
		    for (j = argn; j < argc; j++)
		    {
			if (fnmatch(argv[j], name, 0) == 0)   /* found a match */
			{
			    match = true;
			    break;
			}
		    }
		}
		else             /* no filenames specified on command line */
		{
		    match = true;   /* list all files */
		}
	    }

	    if (match)
	    {
		if (do_md5)
		{
		    print_md5(&assoc_obj.file);
		}

		if (do_sha1)
		{
		    print_sha1(&assoc_obj.file);
		}

		if (do_sha256)
		{
		    print_sha256(&assoc_obj.file);
		}

		if (do_sha512)
		{
		    print_sha512(&assoc_obj.file);
		}

		printf("%s", name);

		if (do_revision)
		{
		    revision = 1;

		    /* This is complicated because of my earlier bad descision about handling file revisions */
		    /* (putting them in the file objects).  Then when I moved them to the path and file      */
		    /* association objects, that made multiple possiblities for what the revision means.     */
		    /*                                                                                       */
		    /* 1) Rename - code in rename_file will not allow a path that had more than one file     */
		    /*    reference be renamed so we can assume that if a the path changed it is a rename.   */
		    /*                                                                                       */
		    /* 2) Revision - a file could have been renamed before, but not after because of the     */
		    /*    code in rename_file.                                                               */

		    /* Check to see if this is an old name for a file */
		    if (!is_void_reference(&assoc_obj.header.object.next_version))
		    {
			assert(nwos_read_object_from_disk(&assoc_obj.header.object.next_version, &next_assoc_obj, sizeof(next_assoc_obj)));
			assert(is_same_object(&next_assoc_obj.header.object.prev_version, &ref_list->references[i]));
			if (!is_same_object(&next_assoc_obj.path, &assoc_obj.path))  /* path changed, must be a rename */
			{
			    assert(is_same_object(&next_assoc_obj.file, &assoc_obj.file));
			    assert(is_void_reference(&assoc_obj.header.object.prev_version));
			    revision = 0;    /* flag old names as revision 0 */
			}
		    }

		    /* Check to see if it has a new style revision (previous/next links in the association object */
		    if (!is_void_reference(&assoc_obj.header.object.prev_version))
		    {
			copy_reference(&prev_ref, &assoc_obj.header.object.prev_version);
			copy_reference(&next_ref, &assoc_obj.header.common.id);
			while (!is_void_reference(&prev_ref))
			{
			    assert(nwos_read_object_from_disk(&prev_ref, &prev_assoc_obj, sizeof(prev_assoc_obj)));
			    assert(is_same_object(&prev_assoc_obj.header.object.next_version, &next_ref));

			    /* don't count it if it was a name change */
			    if (is_same_object(&prev_assoc_obj.path, &assoc_obj.path))
			    {
				revision++;
			    }

			    copy_reference(&prev_ref, &prev_assoc_obj.header.object.prev_version);
			    copy_reference(&next_ref, &prev_assoc_obj.header.common.id);
			}
		    }
		    /* Check to see if it has an old style revision (previous/next links in the file object) */
		    /* NOTE: no need to check for an old style revision if the file was renamed because we   */
		    /*       know the file is the same (assert verified above) so we will do the revision    */
		    /*       when we get to the new name.                                                    */
		    else if (revision != 0)
		    {
			assert(nwos_read_object_from_disk(&assoc_obj.file, &file_obj, sizeof(file_obj)));

			while (!is_void_reference(&file_obj.header.object.prev_version))
			{
			    copy_reference(&next_ref, &file_obj.header.common.id);
			    copy_reference(&prev_ref, &file_obj.header.object.prev_version);
			    assert(nwos_read_object_from_disk(&prev_ref, &file_obj, sizeof(file_obj)));
			    assert(is_same_object(&file_obj.header.object.next_version, &next_ref));
			    if (revision > 0)
			    {
				revision = -revision;    /* make it negative to indicate it is an old style */
			    }
			    revision--;
			}
		    }

		    if (revision < 0)   /* old style */
		    {
			printf(";%d*", -revision);
			old_file_revisions++;
		    }
		    else
		    {
			printf(";%d", revision);
		    }
		}

		if (do_time)
		{
		    print_time(assoc_obj.modification_time);
		}

		printf("\n");

		num_files++;
	    }

	    //    memcpy(file_ref, &ptr_path_obj->file, sizeof(*file_ref));
	}
    }

    printf("Number of files: %d\n", num_files);
    fflush(stdout);

    if (old_file_revisions > 0)
    {
	fprintf(stderr, "\n");

	if (path == DEFAULT_FILE)   /* reading from normal archive */
	{
	    fprintf(stderr, "WARNING: the %d files with an asterisk (*) following the revision number are\n",
		    old_file_revisions);
	    fprintf(stderr, "         using an old method for storing the file revision information (pre\n");
	    fprintf(stderr, "         Alpha_28).  You should run the program: update_files_0028 on this\n");
	    fprintf(stderr, "         archive to update them to the current method.\n");
	}
	else if (memcmp(nwos_disk_header.version_string, "0028", 4) < 0)    /* reading from old compressed file */
	{
	    fprintf(stderr, "NOTE: the %d files with an asterisk (*) following the revision number used\n",
		    old_file_revisions);
	    fprintf(stderr, "      an old method for storing the file revision information (pre Alpha_28).\n");
	    fprintf(stderr, "      Since this compressed file predates Alpha_28 no action is necessary.\n");
	}
	else  /* reading from newer compressed file */
	{
	    fprintf(stderr, "WARNING: the %d files with an asterisk (*) following the revision number used\n",
		    old_file_revisions);
	    fprintf(stderr, "         an old method for storing the file revision information (pre\n");
	    fprintf(stderr, "         Alpha_28).  Since this compressed file postdates that change, you\n");
	    fprintf(stderr, "         should verify that the archive it was created from has been updated.\n");
	    fprintf(stderr, "         You can verify it by running the program 'update_files_0028' on it.\n");
	}

	fprintf(stderr, "\n");
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    nwos_terminate_objectify();

    return 0;
}


