/*
--             This file is part of the New World OS project
--                 Copyright (C) 2006-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: mem_alloc.c,v $
-- Revision 1.1  2009/03/11 12:29:57  jsedwards
-- Created by moving nwos_malloc, nwos_realloc, and nwos_free functions from
-- the objectify.c file.
--
*/

#include <assert.h>
#include <stdio.h>

#include "mem_alloc.h"


/**************************************************************************/
/* Special malloc and free routines that can help check for memory leaks. */
/**************************************************************************/

#ifdef CHECK_MEMORY_LEAKS
#define MAX_MEM_ADDR_SAVED 16384
void* save_malloc_addr[MAX_MEM_ADDR_SAVED];
int cur_malloc_index;
#endif

void* nwos_malloc(size_t size)
{
#ifdef CHECK_MEMORY_LEAKS
    int i;
#endif

    void* result = malloc(size);

    if (result == NULL)
    {
	char msg[64];
	snprintf(msg, sizeof(msg), "%zd bytes", size);
	perror(msg);
	assert(result);
	exit(1);
    }

#ifdef CHECK_MEMORY_LEAKS
    for (i = 0; i < MAX_MEM_ADDR_SAVED; i++)
    {
	if (save_malloc_addr[cur_malloc_index] == NULL) break;
	cur_malloc_index++;
	if (cur_malloc_index >= MAX_MEM_ADDR_SAVED) cur_malloc_index = 0;  /* wrap if necessary */
    }

    assert(i < MAX_MEM_ADDR_SAVED);

    save_malloc_addr[cur_malloc_index] = result;
    cur_malloc_index++;
    if (cur_malloc_index >= MAX_MEM_ADDR_SAVED) cur_malloc_index = 0;  /* wrap if necessary */
#endif

    return result;
}


void* nwos_realloc(void* oldptr, size_t size)
{
    void* result = realloc(oldptr, size);

    if (result == NULL)
    {
	char msg[64];
	snprintf(msg, sizeof(msg), "%zd bytes", size);
	perror(msg);
	assert(result);
	exit(1);
    }

#ifdef CHECK_MEMORY_LEAKS
    int i;

    for (i = 0; i < MAX_MEM_ADDR_SAVED; i++)
    {
	if (save_malloc_addr[i] == oldptr) break;
    }

    assert(i < MAX_MEM_ADDR_SAVED);

    save_malloc_addr[i] = oldptr;
#endif

    return result;
}


void nwos_free(void* ptr)
{
#ifdef CHECK_MEMORY_LEAKS
    int i;
#endif

    free(ptr);

#ifdef CHECK_MEMORY_LEAKS
    for (i = 0; i < MAX_MEM_ADDR_SAVED; i++)
    {
	cur_malloc_index--;
	if (cur_malloc_index < 0) cur_malloc_index = MAX_MEM_ADDR_SAVED - 1;  /* wrap if necessary */
	if (save_malloc_addr[cur_malloc_index] == ptr) break;
    }

    assert(i < MAX_MEM_ADDR_SAVED);

    save_malloc_addr[cur_malloc_index] = NULL;    /* erase it */
#endif
}


