/*
--             This file is part of the New World OS project
--                 Copyright (C) 2007-2009  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: financial.c,v $
-- Revision 1.10.2.1  2009/06/06 14:03:54  jsedwards
-- Moved get_input function to new dialog.c file, renamed it to nwos_ask_user,
-- and changed to call it.
--
-- Revision 1.10  2009/03/10 11:19:00  jsedwards
-- Moved MAX_NUMBER_SIZE, MAX_ID_SIZE, MAX_ACCOUNT_NUMBER_SIZE, and
-- MAX_CHECKING_SIZE to the objectify.h file and renamed.
--
-- Revision 1.9  2009/03/09 14:12:35  jsedwards
-- Eliminated use of FILE_BLOCK_SIZE for kludge buffers.
--
-- Revision 1.8  2009/03/08 00:50:30  jsedwards
-- Changed include objectify_private.h to objectify.h and changed to call
-- nwos_generate_new_id instead of nwos_generate_new_completely_random_id.
--
-- Revision 1.7  2008/09/01 00:10:51  jsedwards
-- Fix copyright year.  NO code changes.
--
-- Revision 1.6  2008/08/31 21:53:52  jsedwards
-- Added an assert around calls to nwos_read_variable_sized_object_from_disk
-- and nwos_read_object_from_disk because now they return false when they fail
-- instead of asserting themselves.
--
-- Revision 1.5  2007/07/01 19:44:12  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.4  2007/05/24 11:48:45  jsedwards
-- Add acronym to credit union.
--
-- Revision 1.3  2007/05/06 17:20:36  jsedwards
-- Add 'list_accounts' function.
--
-- Revision 1.2  2007/05/06 16:37:28  jsedwards
-- Added all the code to create checking accounts.
--
-- Revision 1.1  2007/04/22 15:16:23  jsedwards
-- Initial version created from person.c.
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"

#define MAX_NAME_SIZE 128


static size_t get_account_number_object_size(void* account_num_obj)
{
    uint8 count = ((C_struct_Account_Number*)account_num_obj)->count;
    assert(0 < count && count <= MAX_ACCOUNT_NUMBER_SIZE);

    return sizeof(C_struct_Account_Number) + count;
}


static size_t get_checking_object_size(void* checking_obj)
{
    uint8 count = ((C_struct_Checking*)checking_obj)->count;
    assert(0 < count && count <= MAX_CHECKING_ID_SIZE);

    return sizeof(C_struct_Checking) + count;
}


static bool valid_number(char* number)
{
    char *p;
    size_t length;

    length = strlen(number);

    assert(length <= MAX_ACCOUNT_NUMBER_SIZE);

    for (p  = number; *p != '\0'; p++)
    {
	if (!isdigit(*p) && *p != '-') return false;
    }

    return true;
}


static bool valid_id(char* number)
{
    char *p;

    for (p  = number; *p != '\0'; p++)
    {
	if (!isdigit(*p)) return false;
    }

    return true;
}



bool nwos_find_public_credit_union(char* name, ObjRef* ref)
{
    C_struct_Credit_Union cu_obj;
    EveryObject name_header;
    ObjRef cu_class_ref;
    ObjRef nora_ref;       /* name or acronym reference */
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    bool result = false;


    assert(nwos_find_public_class_definition("CREDIT UNION", &cu_class_ref));

    if (nwos_find_public_name(name, &nora_ref) || nwos_find_public_acronym(name, &nora_ref))
    {
	nwos_read_object_headers_from_disk(&nora_ref, &name_header);

	ref_list = nwos_malloc_reference_list(&name_header.object.references);

	num_refs = ref_list->common_header.num_refs;

	/* printf("number of public credit unions: %d\n", num_refs); */

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &cu_class_ref))   /* it is a word object */
	    {
		assert(nwos_read_object_from_disk(&ref_list->references[i], &cu_obj, sizeof(cu_obj)));

		memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		result = true;
		break;
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;
}


void nwos_list_public_credit_unions()
{
    C_struct_Credit_Union cu_obj;
    ObjRef object_class;
    EveryObject class_ref_header;
    ObjRef cu_class_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    char name[32];
    ReferenceList* url_list;
    int num_urls;

    assert(nwos_find_public_class_definition("CREDIT UNION", &cu_class_ref));

    nwos_read_object_headers_from_disk(&cu_class_ref, &class_ref_header);

    ref_list = nwos_malloc_reference_list(&class_ref_header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("cu class num refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &cu_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &cu_obj, sizeof(cu_obj)));

	    if (!is_void_reference(&cu_obj.short_name))
	    {
		nwos_name_to_string(&cu_obj.short_name, name, sizeof(name));
	    }
	    else
	    {
		nwos_name_to_string(&cu_obj.full_name, name, sizeof(name));
	    }

	    printf("%s", name);

	    if (!is_void_reference(&cu_obj.acronym))
	    {
		assert(nwos_acronym_to_string(&cu_obj.acronym, name, sizeof(name)));
		printf(" - %s", name);
	    }

	    if (!is_void_reference(&cu_obj.url_list))
	    {
		url_list = nwos_malloc_reference_list(&cu_obj.url_list);

		num_urls = url_list->common_header.num_refs;

		if (num_urls > 0)
		{
		    assert(nwos_url_to_string(&url_list->references[0], name, sizeof(name)));
		    printf(" - %s", name);
		}

		nwos_free_reference_list(url_list);
		url_list = NULL;
	    }

	    printf("\n");
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;
}


bool nwos_find_private_credit_union(char* name, ObjRef* ref)
{
    ObjRef private_class_ref;
    ObjRef public_ref;
    ObjRef object_class;
    C_struct_Class_Definition class_def_obj;
    C_struct_Credit_Union cu_obj;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    bool result = false;

    assert(nwos_find_public_credit_union(name, &public_ref));

    if (nwos_find_private_class_definition("CREDIT UNION", &private_class_ref))
    {
	nwos_read_class_definition(&private_class_ref, &class_def_obj);

	ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

	num_refs = ref_list->common_header.num_refs;

	/* printf("num_refs: %d\n", num_refs); */

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&object_class, &private_class_ref))
	    {
		assert(nwos_read_object_from_disk(&ref_list->references[i], &cu_obj, sizeof(cu_obj)));

		if (is_same_object(&cu_obj.header.object.clone_of, &public_ref))   /* found a match */
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    result = true;
		    break;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;
}


/**********************************************************************************************/
/* This routine finds a private credit union object or creates one by cloning the public one. */
/**********************************************************************************************/

ObjCreateResult nwos_find_or_create_private_credit_union(char* name, ObjRef* ref)
{
    ObjRef public_cu_ref;
    ObjRef private_class_ref;
    C_struct_Credit_Union cu_obj;
    char msg[128];
    ObjCreateResult result = FOUND_EXISTING;

    assert(nwos_find_public_credit_union(name, &public_cu_ref));

    if (!nwos_find_private_credit_union(name, ref))
    {
	nwos_find_or_create_private_class_definition("CREDIT UNION", &private_class_ref);

	assert(nwos_read_object_from_disk(&public_cu_ref, &cu_obj, sizeof(cu_obj)));

	nwos_generate_new_id(ref);

	snprintf(msg, sizeof(msg), "clone_credit_union(%02x%02x%02x%02x) -> %02x%02x%02x%02x\n",
		 public_cu_ref.id[0], public_cu_ref.id[1], public_cu_ref.id[2], public_cu_ref.id[3],
		 ref->id[0], ref->id[1], ref->id[2], ref->id[3]);
	nwos_log(msg);

	copy_reference(&cu_obj.header.common.id, ref);

	nwos_get_time_stamp(cu_obj.header.common.creation_time);

	copy_reference(&cu_obj.header.common.class_definition, &private_class_ref);

	copy_reference(&cu_obj.header.object.clone_of, &public_cu_ref);

	nwos_create_reference_list(ref, &cu_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &cu_obj.header.object, sizeof(ObjectHeader), cu_obj.header.common.header_chksum);

	nwos_write_object_to_disk(ref, &cu_obj, sizeof(cu_obj));

	nwos_add_to_references(ref, &private_class_ref);

	result = CREATED_NEW;
    }

    return result;
}


bool nwos_find_account_number(char* number, ObjRef* ref)
{
    ObjRef number_class_ref;
    C_struct_Class_Definition class_def_obj;
    uint8 kludge[MAX_ACCOUNT_NUMBER_OBJ_SIZE];
    C_struct_Account_Number* ptr_number_obj = (C_struct_Account_Number*)kludge;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    size_t length;
    int i;
    bool result = false;


    assert(valid_number(number));

    length = strlen(number);

    if (nwos_find_private_class_definition("ACCOUNT NUMBER", &number_class_ref))
    {
	nwos_read_class_definition(&number_class_ref, &class_def_obj);

	ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

	num_refs = ref_list->common_header.num_refs;

	/* printf("num_refs: %d\n", num_refs); */

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&object_class, &number_class_ref))
	    {
		assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_account_number_object_size));

		/* remember ptr_number_obj points to the kludge buffer */

		if (ptr_number_obj->count == length && strncmp((char*)ptr_number_obj->storage, number, length) == 0)   /* found a match */
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    result = true;
		    break;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;   /* return true if we found it */
}



ObjCreateResult nwos_find_or_create_account_number(char* number, ObjRef* ref)
{
    ObjRef number_class_ref;
    uint8 kludge[MAX_ACCOUNT_NUMBER_OBJ_SIZE];
    C_struct_Account_Number* ptr_number_obj = (C_struct_Account_Number*)kludge;
    size_t length;
    ObjCreateResult result = FOUND_EXISTING;

    assert(valid_number(number));

    if (!nwos_find_account_number(number, ref))
    {
	memset(kludge, 0, sizeof(kludge));

	nwos_find_or_create_private_class_definition("ACCOUNT NUMBER", &number_class_ref);

	nwos_generate_new_id(ref);

	nwos_fill_in_common_header(&ptr_number_obj->header.common, ref, &number_class_ref);

	nwos_create_reference_list(ref, &ptr_number_obj->header.object.references);

	length = strlen(number);

	ptr_number_obj->count = length;

	memcpy(ptr_number_obj->storage, number, length);

	nwos_crc32_calculate((uint8*) &ptr_number_obj->header.object, sizeof(ObjectHeader), ptr_number_obj->header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &ptr_number_obj->count, sizeof(C_struct_Account_Number) + length - sizeof(EveryObject), ptr_number_obj->header.common.data_chksum);

	nwos_write_object_to_disk(ref, kludge, sizeof(C_struct_Account_Number) + length);

	nwos_add_to_references(ref, &number_class_ref);

	ptr_number_obj = malloc(sizeof(C_struct_Account_Number) + length);
	assert(nwos_read_object_from_disk(ref, ptr_number_obj, sizeof(C_struct_Account_Number) + length));
	assert(memcmp(kludge, ptr_number_obj, sizeof(C_struct_Account_Number) + length) == 0);

	memset(kludge, 0, sizeof(kludge));  /* clear it */
	assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_account_number_object_size));  /* read the other way */
	assert(memcmp(ptr_number_obj, kludge, sizeof(C_struct_Account_Number) + length) == 0);

	free(ptr_number_obj);
	ptr_number_obj = NULL;

	result = CREATED_NEW;
    }

    return result;
}


void nwos_account_number_to_string(ObjRef* ref, char* string, size_t size)
{
    uint8 kludge[MAX_ACCOUNT_NUMBER_OBJ_SIZE];
    C_struct_Account_Number* ptr_number_obj = (C_struct_Account_Number*)kludge;

    assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_account_number_object_size));

    assert(ptr_number_obj->count < size);

    memcpy(string, ptr_number_obj->storage, ptr_number_obj->count);
    string[ptr_number_obj->count] = '\0';
}



bool nwos_find_account(ObjRef* credit_union_ref, ObjRef* account_num_ref, ObjRef* ref)
{
    ObjRef account_class_ref;
    C_struct_Class_Definition class_def_obj;
    C_struct_Account account_obj;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    bool result = false;


    if (nwos_find_private_class_definition("ACCOUNT", &account_class_ref))
    {
	nwos_read_class_definition(&account_class_ref, &class_def_obj);

	ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

	num_refs = ref_list->common_header.num_refs;

	/* printf("num_refs: %d\n", num_refs); */

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);

	    if (is_same_object(&object_class, &account_class_ref))
	    {
		assert(nwos_read_object_from_disk(&ref_list->references[i], &account_obj, sizeof(account_obj)));

		if (is_same_object(&account_obj.institution, credit_union_ref)
		    && is_same_object(&account_obj.number, account_num_ref))   /* found a match */
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    result = true;
		    break;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;   /* return true if we found it */
}


void nwos_create_account(ObjRef* credit_union_ref, ObjRef* account_num_ref, ObjRef* ref)
{
    ObjRef account_class_ref;
    C_struct_Account account_obj;
    C_struct_Account test_obj;

    assert(!nwos_find_account(credit_union_ref, account_num_ref, ref));

    memset(&account_obj, 0, sizeof(account_obj));

    nwos_find_or_create_private_class_definition("ACCOUNT", &account_class_ref);

    nwos_generate_new_id(ref);

    nwos_fill_in_common_header(&account_obj.header.common, ref, &account_class_ref);

    nwos_create_reference_list(ref, &account_obj.header.object.references);

    copy_reference(&account_obj.institution, credit_union_ref);
    copy_reference(&account_obj.number, account_num_ref);

    nwos_crc32_calculate((uint8*) &account_obj.header.object, sizeof(ObjectHeader), account_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &account_obj.institution, sizeof(C_struct_Account) - sizeof(EveryObject), account_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &account_obj, sizeof(account_obj));

    nwos_add_to_references(ref, &account_class_ref);

    nwos_add_to_references(ref, &account_obj.institution);
    nwos_add_to_references(ref, &account_obj.number);

    assert(nwos_read_object_from_disk(ref, &test_obj, sizeof(C_struct_Account)));
    assert(memcmp(&account_obj, &test_obj, sizeof(C_struct_Account)) == 0);
}



void nwos_create_checking(ObjRef* account_ref, char* id, ObjRef* ref)
{
    ObjRef checking_class_ref;
    uint8 kludge[MAX_CHECKING_OBJ_SIZE];
    C_struct_Checking* ptr_checking_obj = (C_struct_Checking*)kludge;
    size_t length;


    assert(valid_id(id));

    memset(kludge, 0, sizeof(kludge));

    nwos_find_or_create_private_class_definition("CHECKING", &checking_class_ref);

    nwos_generate_new_id(ref);

    nwos_fill_in_common_header(&ptr_checking_obj->header.common, ref, &checking_class_ref);

    nwos_create_reference_list(ref, &ptr_checking_obj->header.object.references);

    copy_reference(&ptr_checking_obj->account, account_ref);

    length = strlen(id);

    ptr_checking_obj->count = length;

    memcpy(ptr_checking_obj->id, id, length);

    nwos_crc32_calculate((uint8*) &ptr_checking_obj->header.object, sizeof(ObjectHeader), ptr_checking_obj->header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &ptr_checking_obj->account, sizeof(C_struct_Checking) + length - sizeof(EveryObject), ptr_checking_obj->header.common.data_chksum);

    nwos_write_object_to_disk(ref, kludge, sizeof(C_struct_Checking) + length);

    nwos_add_to_references(ref, &checking_class_ref);

    nwos_add_to_references(ref, &ptr_checking_obj->account);

    ptr_checking_obj = malloc(sizeof(C_struct_Checking) + length);
    assert(nwos_read_object_from_disk(ref, ptr_checking_obj, sizeof(C_struct_Checking) + length));
    assert(memcmp(kludge, ptr_checking_obj, sizeof(C_struct_Checking) + length) == 0);

    memset(kludge, 0, sizeof(kludge));  /* clear it */
    assert(nwos_read_variable_sized_object_from_disk(ref, kludge, sizeof(kludge), &get_checking_object_size));  /* read the other way */
    assert(memcmp(ptr_checking_obj, kludge, sizeof(C_struct_Checking) + length) == 0);

    free(ptr_checking_obj);
    ptr_checking_obj = NULL;
}



void nwos_add_account()
{
    char name[MAX_NAME_SIZE+2];
    char number[MAX_ACCOUNT_NUMBER_SIZE+2];
    char id[MAX_CHECKING_ID_SIZE+2];
    ObjRef cu_ref;
    ObjRef number_ref;
    ObjRef account_ref;
    ObjRef checking_ref;
    bool ok;
    bool cu_existed;
    bool num_existed;

#if 0
    nwos_find_or_create_private_class_definition("PERSON", &person_class_ref);
#endif

    ok = false;
    while (!ok)
    {
	nwos_ask_user("name of credit union", name, sizeof(name));
	if (*name == '\0') return;
	ok = nwos_find_public_credit_union(name, &cu_ref);
	if (!ok) printf("name not found in public credit union list.\n");
    }

    printf("public credit union: %02x%02x%02x%02x\n",
	   cu_ref.id[0], cu_ref.id[1], cu_ref.id[2], cu_ref.id[3]);


    ok = false;
    while (!ok)
    {
	nwos_ask_user("account number", number, sizeof(number));
	if (*number == '\0') return;
	ok = valid_number(number);
	if (!ok) printf("number can only contain digits 0 to 9 and dash '-'.\n");
    }

    ok = false;
    while (!ok)
    {
	nwos_ask_user("checking id", id, sizeof(id));
	if (*id == '\0') return;
	ok = valid_id(id);
	if (!ok) printf("id can only contain digits 0 to 9.\n");
    }

    printf("\n");
    printf("credit union: %s\n", name);
    printf("account number: %s\n", number);
    printf("checking id: %s\n", id);
    printf("\n");

    if (nwos_find_account_number(number, &number_ref))
    {
	printf("WARNING: account number: %s already exists\n\n", number);
    }

    if (nwos_ask_yes_or_no(NULL, "Create account"))
    {
	cu_existed = (nwos_find_or_create_private_credit_union(name, &cu_ref) == FOUND_EXISTING);

	num_existed = (nwos_find_or_create_account_number(number, &number_ref) == FOUND_EXISTING);

	if (cu_existed && num_existed && nwos_find_account(&cu_ref, &number_ref, &account_ref))
	{
	    printf("ERROR: that account already exists\n");
	    return;
	}

	nwos_create_account(&cu_ref, &number_ref, &account_ref);

	nwos_create_checking(&account_ref, id, &checking_ref);

	
	printf("Object created: %02x%02x%02x%02x\n",
	       checking_ref.id[0], checking_ref.id[1], checking_ref.id[2], checking_ref.id[3]);
    }
}


void nwos_list_accounts()
{
    ObjRef checking_class_ref;
    EveryObject class_ref_header;
    uint8 kludge[MAX_CHECKING_OBJ_SIZE];
    C_struct_Checking* ptr_checking_obj = (C_struct_Checking*)kludge;
    C_struct_Account account_obj;
    C_struct_Credit_Union cu_obj;
    ObjRef object_class;
    ReferenceList* ref_list;
    int num_refs;
    char name[MAX_NAME_SIZE+1];
    char number[MAX_ACCOUNT_NUMBER_SIZE+1];
    int i;

    if (!nwos_find_private_class_definition("CHECKING", &checking_class_ref))
    {
	printf("No checking accounts found\n");
    }
    else
    {
	nwos_read_object_headers_from_disk(&checking_class_ref, &class_ref_header);

	ref_list = nwos_malloc_reference_list(&class_ref_header.object.references);

	num_refs = ref_list->common_header.num_refs;

	for (i = 0; i < num_refs; i++)
	{
	    nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	    if (is_same_object(&object_class, &checking_class_ref))
	    {
		assert(nwos_read_variable_sized_object_from_disk(&ref_list->references[i], kludge, sizeof(kludge), &get_checking_object_size));

		assert(nwos_read_object_from_disk(&ptr_checking_obj->account, &account_obj, sizeof(account_obj)));

		assert(nwos_read_object_from_disk(&account_obj.institution, &cu_obj, sizeof(cu_obj)));

		nwos_account_number_to_string(&account_obj.number, number, sizeof(number));

		if (!is_void_reference(&cu_obj.short_name))
		{
		    nwos_name_to_string(&cu_obj.short_name, name, sizeof(name));
		}
		else
		{
		    nwos_name_to_string(&cu_obj.full_name, name, sizeof(name));
		}

		printf("%s - %s\n", name, number);
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }
}


#if 0
void nwos_list_credit_unions()
{
    C_struct_Person person_obj;
    ObjRef object_class;
    EveryObject class_ref_header;
    ObjRef person_class_ref;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    char name[32];

    assert(nwos_find_private_class_definition("PERSON", &person_class_ref));

    nwos_read_object_headers_from_disk(&person_class_ref, &class_ref_header);

    ref_list = nwos_malloc_reference_list(&class_ref_header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("person class num refs: %d\n", num_refs); */

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &person_class_ref))
	{
	    assert(nwos_read_object_from_disk(&ref_list->references[i], &person_obj, sizeof(person_obj)));

	    nwos_name_to_string(&person_obj.last_name, name, sizeof(name));

	    printf("%s, ", name);

	    nwos_name_to_string(&person_obj.first_name, name, sizeof(name));

	    printf("%s ", name);

	    if (!is_void_reference(&person_obj.middle_name))
	    {
		nwos_name_to_string(&person_obj.middle_name, name, sizeof(name));

		printf("%s", name);
	    }

	    printf("\n");
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;
}


void nwos_describe_person(char* name)
{
    C_struct_Person person_obj;
    ObjRef person_ref;
    char temp[64];
    
    if (!nwos_find_person(name, &person_ref))
    {
	printf("\nI'm sorry I don't know anyone named %s.\n\n", name);
    }
    else
    {
	assert(nwos_read_object_from_disk(&person_ref, &person_obj, sizeof(person_obj)));

	printf("full name: ");

	nwos_name_to_string(&person_obj.first_name, temp, sizeof(temp));
	printf("%s ", temp);

	if (!is_void_reference(&person_obj.middle_name))
	{
	    nwos_name_to_string(&person_obj.middle_name, temp, sizeof(temp));
	    printf("%s ", temp);
	}

	nwos_name_to_string(&person_obj.last_name, temp, sizeof(temp));
	printf("%s\n", temp);


	if (!is_void_reference(&person_obj.goes_by))
	{
	    printf("goes by: ");
	    nwos_name_to_string(&person_obj.goes_by, temp, sizeof(temp));
	    printf("%s\n", temp);
	}

	if (!is_void_reference(&person_obj.nickname))
	{
	    printf("nickname: ");
	    nwos_name_to_string(&person_obj.nickname, temp, sizeof(temp));
	    printf("%s\n", temp);
	}

	if (!is_void_reference(&person_obj.maiden_name))
	{
	    printf("maiden_name: ");
	    nwos_name_to_string(&person_obj.maiden_name, temp, sizeof(temp));
	    printf("%s\n", temp);
	}

	if (!is_void_reference(&person_obj.birth_date))
	{
	    printf("born: ");
	    nwos_date_to_string(&person_obj.birth_date, temp, sizeof(temp));
	    printf("%s\n", temp);
	}
    }

    printf("\n");
}
#endif

