/*
--             This file is part of the New World OS project
--                   Copyright (C) 2008  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: user_config.c,v $
-- Revision 1.13  2008/06/08 16:33:08  jsedwards
-- Added #ifdef HTTP around code that tries to open a config file in the user's
-- home directory, because when running under apache $HOME isn't defined.
--
-- Revision 1.12  2008/02/24 13:24:33  jsedwards
-- Fix so that if DEFAULT_USER_CONFIG_PATH contains a leading '~' the user's
-- home directory will correctly be substituted.
--
-- Revision 1.11  2008/02/23 02:06:26  jsedwards
-- Changed to search for a configuration file in the users home directory
-- first and then read the system configuration file if it doesn't exist.
--
-- Revision 1.10  2008/02/06 04:19:27  jsedwards
-- Fix Bug #1887042 - changed so that all environment variables are checked
-- before reading the config file.
--
-- Revision 1.9  2008/02/05 04:30:05  jsedwards
-- Added nwos_get_log_file_path function.
--
-- Revision 1.8  2008/02/05 04:09:10  jsedwards
-- Fix home directory substitution so it doesn't leave the tilde in.
--
-- Revision 1.7  2008/02/04 03:14:01  jsedwards
-- Added nwos_get_backup_directory_path function.
--
-- Revision 1.6  2008/02/04 02:48:32  jsedwards
-- Added code to convert a path that begins with ~ to the $HOME directory.
--
-- Revision 1.5  2008/02/02 18:20:33  jsedwards
-- Added nwos_get_private_objects_path function.
--
-- Revision 1.4  2008/02/02 16:27:56  jsedwards
-- Added the environment variable name and default path to path_table and
-- made a generic function to find a given path.
--
-- Revision 1.3  2008/02/02 16:11:13  jsedwards
-- Eliminated public_objects_path variable and just stored the path in the
-- path_table.
--
-- Revision 1.2  2008/02/02 15:53:06  jsedwards
-- Added new function to read the configuration file and changed the
-- nwos_get_public_objects_path function to check for environment variable
-- first, then read the configuration file and finally use default path.
--
-- Revision 1.1  2008/01/22 14:51:35  jsedwards
-- Initial version, just returns default public path.
--
*/

#include <ctype.h>
#include <limits.h>    /* define PATH_MAX */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "objectify.h"


static bool config_file_has_been_read;

static struct path_table
{
    char* name;
    char* env;
    char* default_path;
    char* path;
    size_t path_length;     /* if non-zero path was malloc'd */
} path_table[] = {
    { "public",  "OBJECTIFY_PUBLIC_PATH",   DEFAULT_PUBLIC_PATH,   NULL, 0 },
    { "private", "OBJECTIFY_PRIVATE_PATH",  DEFAULT_PRIVATE_PATH,  NULL, 0 },
    { "log",     "OBJECTIFY_LOG_FILE_PATH", DEFAULT_LOG_FILE_PATH, NULL, 0 },
    { "backup",  "OBJECTIFY_BACKUP_PATH",   NULL,                  NULL, 0 },
};

#define PATH_TABLE_SIZE (sizeof(path_table) / sizeof(struct path_table))


/* get the user's home directory */

const char* get_users_home_dir(const char* path)
{
    static char* result = NULL;
    char msg[128];

    if (result == NULL)
    {
	result = getenv("HOME");

	if (result == NULL)
	{
	    snprintf(msg, sizeof(msg), "WARNING: $HOME environment variable not defined, cannot resolve path: %s", path);

	    nwos_log(msg);
	    fprintf(stderr, "%s\n", msg);
	}
    }

    return result;
}


static void read_configuration_file()
{
    FILE* fp = NULL;
    char config_path[PATH_MAX];
    char buffer[1024];
    int line_number = 0;
    char *p;
    int i;

    assert(DEFAULT_SYSTEM_CONFIG_PATH[0] == '/');

    assert(DEFAULT_USER_CONFIG_PATH[0] == '/' || (DEFAULT_USER_CONFIG_PATH[0] == '~' && DEFAULT_USER_CONFIG_PATH[1] == '/'));

    if (!config_file_has_been_read)
    {
	config_file_has_been_read = true;

	/* check all environment variables first */

	for (i = 0; i < PATH_TABLE_SIZE; i++)
	{
	    path_table[i].path = getenv(path_table[i].env);
	}	    

	/* then read the file */

#ifndef HTTP
	/* first try to open a users config file */
	if (*DEFAULT_USER_CONFIG_PATH == '~')
	{
	    strlcpy(config_path, get_users_home_dir(DEFAULT_USER_CONFIG_PATH), sizeof(config_path));
	    strlcat(config_path, DEFAULT_USER_CONFIG_PATH+1, sizeof(config_path));
	}
	else
	{
	    strlcpy(config_path, DEFAULT_USER_CONFIG_PATH, sizeof(config_path));
	}

	fp = fopen(config_path, "r");
#endif

	/* if that failed try to open the system config file */
	if (fp == NULL)
	{
	    strlcpy(config_path, DEFAULT_SYSTEM_CONFIG_PATH, sizeof(config_path));
	    fp = fopen(config_path, "r");
	}

	if (fp != NULL)
	{
	    while (fgets(buffer, sizeof(buffer), fp) != NULL)
	    {
		line_number++;

		p = strchr(buffer, '\n');

		if (p == NULL)    /* line too long */
		{
		    if (buffer[0] != '#')
		    {
			snprintf(buffer, sizeof(buffer), "WARNING: config file '%s' line %d is too long, ignored",
				 config_path, line_number);

			nwos_log(buffer);
			fprintf(stderr, "%s\n", buffer);
		    }

		    while (fgets(buffer, sizeof(buffer), fp) != NULL && strchr(buffer, '\n') == NULL); /* skip over */
		}
		else
		{
		    *p = '\0';    /* eliminate new line character */

		    for (p = buffer; *p != '\0' && *p != '#'; p++)   /* skip over leading spaces */
		    {
			assert(p < buffer + sizeof(buffer));

			if (isspace(*p))
			{
			    *p++ = '\0';

			    while (*p != '\0' && isspace(*p)) p++;   /* skip over any extra whitespace */

			    for (i = 0; i < PATH_TABLE_SIZE; i++)
			    {
				if (strcmp(path_table[i].name, buffer) == 0) break;
			    }

			    if (i < PATH_TABLE_SIZE)   /* found it */
			    {
				if (path_table[i].path == NULL)
				{
				    path_table[i].path_length = strlen(p) + 1;

				    path_table[i].path = malloc(path_table[i].path_length);

				    if (path_table[i].path == NULL)
				    {
					perror("allocating memory for configuration path");
					exit(1);
				    }

				    strlcpy(path_table[i].path, p, path_table[i].path_length);
				}
			    }
			    else
			    {
				snprintf(buffer, sizeof(buffer), "WARNING: config file '%s' line %d is invalid, ignored",
					 config_path, line_number);

				nwos_log(buffer);
				fprintf(stderr, "%s\n", buffer);
			    }

			    break;
			}
		    }
		}
	    }

	    fclose(fp);
	}

	/* finally fill in any null values with the default */

	for (i = 0; i < PATH_TABLE_SIZE; i++)
	{
	    if (path_table[i].path == NULL)
	    {
		path_table[i].path = path_table[i].default_path;
	    }
	}	    

    }
}

static const char* get_path(char* name)
{
    int i;
    const char* home;
    char* combined;
    size_t combined_length;


    for (i = 0; i < PATH_TABLE_SIZE; i++)
    {
	if (strcmp(path_table[i].name, name) == 0) break;
    }

    assert(i < PATH_TABLE_SIZE);

    if (path_table[i].path == NULL)
    {
	read_configuration_file();     /* read the environment variables and the config file */
    }

    if (path_table[i].path != NULL && path_table[i].path[0] == '~')
    {
	home = get_users_home_dir(path_table[i].path);

	if (home == NULL)
	{
	    if (path_table[i].path_length != 0)  /* path was malloc'd */
	    {
		free(path_table[i].path);
	    }

	    path_table[i].path = NULL;
	}
	else
	{
	    combined_length = strlen(home) + 1 + strlen(path_table[i].path) + 1;
	    combined = malloc(combined_length);

	    strlcpy(combined, home, combined_length);
	    if (path_table[i].path[1] != '/')
	    {
		strlcat(combined, "/", combined_length);
	    }
	    strlcat(combined, path_table[i].path + 1, combined_length);

	    if (path_table[i].path_length != 0)  /* path was malloc'd */
	    {
		free(path_table[i].path);   /* better free it */
	    }

	    path_table[i].path = combined;
	    path_table[i].path_length = combined_length;
	}
    }

    return path_table[i].path;
}


const char* nwos_get_public_objects_path()
{
    return get_path("public");
}


const char* nwos_get_private_objects_path()
{
    return get_path("private");
}


const char* nwos_get_log_file_path()
{
    return get_path("log");
}


const char* nwos_get_backup_directory_path()
{
    return get_path("backup");
}


