/*
--             This file is part of the New World OS project
--                Copyright (C) 2006, 2007  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
--   This program is free software: you can redistribute it and/or modify
--   it under the terms of the GNU General Public License as published by
--   the Free Software Foundation, either version 3 of the License, or
--   (at your option) any later version.
--
--   This program is distributed in the hope that it will be useful,
--   but WITHOUT ANY WARRANTY; without even the implied warranty of
--   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
--   GNU General Public License for more details.
--
--   You should have received a copy of the GNU General Public License
--   along with this program, in the file LICENSE.  If not, see 
--   <http://www.gnu.org/licenses/>.
--
--   You can also contact me via paper mail at:
--
--      QRW Software
--      P.O. Box 27511
--      Salt Lake City, UT 84127-0511, USA.
--
--
-- $Log: disc_list.c,v $
-- Revision 1.12  2007/07/01 19:44:11  jsedwards
-- Upgrade to GPLv3.
--
-- Revision 1.11  2007/06/13 21:15:03  jsedwards
-- Comment out debugging print statement accidentally left in.
--
-- Revision 1.10  2007/06/13 20:51:39  jsedwards
-- Added code to free memory used for file names.
--
-- Revision 1.9  2007/06/13 20:13:54  jsedwards
-- Change to take a directory as a parameter instead of a list of files.
--
-- Revision 1.8  2007/02/11 16:58:26  jsedwards
-- Changed so DEFAULT_TYPE has to specify RO (Read-Only) or RW (Read-Write).
--
-- Revision 1.7  2007/01/20 13:07:26  jsedwards
-- Print blank line after entering pass phrase and move log_arguments call.
--
-- Revision 1.6  2007/01/07 20:31:55  jsedwards
-- Added call to log arguments.
--
-- Revision 1.5  2007/01/07 03:26:19  jsedwards
-- Changed default encryption level to 'Very_Low' instead of 'Low'.
--
-- Revision 1.4  2007/01/04 17:39:14  jsedwards
-- Add code to set encryption level (always "Low" for now).
--
-- Revision 1.3  2006/12/07 14:10:25  jsedwards
-- Removed call to file_setup, no longer needed.
--
-- Revision 1.2  2006/12/02 17:26:09  jsedwards
-- Changed to deal with file paths that can point to different files.
--
-- Revision 1.1  2006/11/19 13:53:12  jsedwards
-- Created "disc_list.c" from "log_disc.c".  This program just creates a disc
-- list and does not create a disc copy object.
--
-- Revision 1.6  2006/11/18 14:34:58  jsedwards
-- Changed so that discs can be stored in one of three different binders.
--
-- Revision 1.5  2006/11/11 12:01:04  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.4  2006/11/07 14:03:35  jsedwards
-- Add assert to verify file_path didn't come back void.
--
-- Revision 1.3  2006/11/06 13:54:07  jsedwards
-- Changed wording in print statement because disc_list now stores the
-- references to the file_path object instead of the file object.
--
-- Revision 1.2  2006/11/05 21:32:18  jsedwards
-- Add code to create storage_location and disc_copy objects.
--
-- Revision 1.1  2006/11/04 18:56:22  jsedwards
-- Program to scan all the files on a CD or DVD and make a disk_list object
-- of it.
--
*/


#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include "objectify.h"



int main(int argc, char* argv[])
{
    ObjRef root_object_ref;
    uint8 big_key[16 + 8 + 4];
    uint8 bf_key[16];
    uint32 linear;
    uint32 serial;
    ObjRef ref;
    ObjCreateResult result;
    int i;
    time_t start_time;
    char id[14];
    char *p;
    bool ok = false;
    char* file_names[MAX_FILES_PER_DISC_LIST];
    int num_files;
    ObjRef files[MAX_FILES_PER_DISC_LIST];
    ObjRef disc_list_ref;
    DIR* dp;
    struct dirent *dir_entry;


    if (argc != 2)
    {
	fprintf(stderr, "usage: %s directory\n", argv[0]);
	exit(1);
    }

    dp = opendir(argv[1]);
    if (dp == NULL)
    {
	perror(argv[1]);
	exit(1);
    }

    num_files = 0;
    dir_entry = readdir(dp);
    while (dir_entry != NULL)
    {
	if (strcmp(dir_entry->d_name, ".") != 0 && strcmp(dir_entry->d_name, "..") != 0)
	{
	    if (num_files == MAX_FILES_PER_DISC_LIST)
	    {
		fprintf(stderr, "Too many files in directory: %d, currently can only handle %d files max\n",
			num_files, MAX_FILES_PER_DISC_LIST);
		exit(1);
	    }

	    file_names[num_files] = malloc(strlen(dir_entry->d_name) + 1);

	    if (file_names[num_files] == NULL)
	    {
		perror("allocating memory for file name");
		exit(1);
	    }

	    strcpy(file_names[num_files], dir_entry->d_name);

	    /* printf("%s\n", file_names[num_files]); */

	    num_files++;
	}

	dir_entry = readdir(dp);
    }

    if (closedir(dp) != 0)
    {
	perror(argv[1]);
	exit(1);
    }

    printf("\n");

    nwos_get_key_from_password(big_key, sizeof(big_key));

    printf("\n");

    memcpy(bf_key, big_key, 16);
    linear = ((uint32)big_key[16] << 24) | ((uint32)big_key[17] << 16) | ((uint32)big_key[18] << 8) | (uint32)big_key[19];
    memcpy(root_object_ref.id, big_key+20, 4);
    serial = ((uint32)big_key[24] << 24) | ((uint32)big_key[25] << 16) | ((uint32)big_key[26] << 8) | (uint32)big_key[27];

    nwos_log_arguments(argc, argv);

    nwos_initialize_objectify(bf_key, linear, serial, DEFAULT_TYPE_RW, DEFAULT_FILE);

    nwos_set_root_object(&root_object_ref);

    nwos_set_encryption_level(Encryption_Very_Low);

    /*******************/
    /* Get the disc id */
    /*******************/

    while (!ok)
    {
	printf("Disk id: ");
	fflush(stdout);

	fgets(id, sizeof(id), stdin);

	p = strchr(id, '\n');   /* find the newline char */

	if (p == NULL)    /* line was tool long */
	{
	    while (p == NULL) 
	    {
		fgets(id, sizeof(id), stdin);
		p = strchr(id, '\n');   /* find the newline char */
	    }
	    printf("id was too long, must be less than %zd characters\n", sizeof(id) - 2);
	}
	else     /* line was ok */
	{
	    *p = '\0';   /* eliminate the newline character */

	    if (strlen(id) < 12)
	    {
		printf("id was too short, must be %zd characters\n", sizeof(id) - 2);
	    }
	    else
	    {
		ok = true;   /* we should be good to go */
	    }
	}
    }

    if (nwos_find_disc_list(id, &ref))
    {
	printf("That disc already exists!\n");
	exit(1);
    }

    memset(files, 0, sizeof(files));

    for (i = 0; i < num_files; i++)
    {
	start_time = time(NULL);

	result = nwos_create_file_without_storing_data(argv[1], file_names[i], &files[i]);

	assert(!is_void_reference(&files[i]));

	if (result == CREATED_NEW)
	{
	    printf("created new file path: %02x%02x%02x%02x   time: %d seconds\n", 
		   files[i].id[0], files[i].id[1], files[i].id[2], files[i].id[3], 
		   (int) (time(NULL) - start_time));
	}
	else
	{
	    printf("result: %d\n", result);
	}

	nwos_flush_bit_maps();
    }

    assert(nwos_create_disc_list(id, files, &disc_list_ref) == CREATED_NEW);

    printf("disc list: %02x%02x%02x%02x\n",
	   disc_list_ref.id[0], disc_list_ref.id[1], disc_list_ref.id[2], disc_list_ref.id[3]);

    for (i = 0; i < num_files; i++)
    {
	free(file_names[i]);
    }

    nwos_terminate_objectify();

    return 0;
}


