/*             This file is part of the New World OS project
--                Copyright (C) 2005, 2006  QRW Software
--           J. Scott Edwards - j.scott.edwards.nwos@gmail.com 
--                      http://www.qrwsoftware.com
--                      http://nwos.sourceforge.com
--
-- NWOS is free software;  you can redistribute it and/or modify it under the
-- terms of the GNU General Public License  as published by the Free Software
-- Foundation; either version 2, or (at your option) any later version.  This
-- software is distributed with the hope that it will be useful,  but WITHOUT
-- ANY WARRANTY;  without  even the  implied warranty  of MERCHANTABILITY  or
-- FITNESS FOR A PARTICULAR PURPOSE.   See the GNU General Public License for
-- more  details.  You should have received a copy  of the GNU General Public
-- License along with this package;  see the file LICENSE.  If not, write to:
--
--      Free Software Foundation, Inc.
--      59 Temple Place - Suite 330
--      Boston, MA 02111-1307, USA.
--
-- $Log: phone.c,v $
-- Revision 1.22  2007/04/10 13:27:32  jsedwards
-- Changed to find or create phone_number and mobile_phone classes.
--
-- Revision 1.21  2007/04/08 18:15:49  jsedwards
-- Made separate public and private create area code functions.
--
-- Revision 1.20  2007/04/07 14:05:54  jsedwards
-- Split find_area_code function into public and private variants.
--
-- Revision 1.19  2007/02/11 15:15:20  jsedwards
-- Change 'sprintf' calls to 'snprintf' calls so the OpenBSD linker will stop
-- whining.
--
-- Revision 1.18  2006/12/25 12:05:41  jsedwards
-- Ifdef out code that isn't used in "public_mode".
--
-- Revision 1.17  2006/12/21 12:59:47  jsedwards
-- Hack to make it compile with new split public and private classes,
-- NON-FUNCTIONAL!
--
-- Revision 1.16  2006/12/07 14:08:46  jsedwards
-- Moved setup routine and associated functions to attic/big_bang.c.
--
-- Revision 1.15  2006/12/01 14:31:30  jsedwards
-- Changed to use new malloc_reference_list and free_reference_list functions
-- instead of inlining the code.
--
-- Revision 1.14  2006/11/11 12:01:06  jsedwards
-- Update e-mail address to something that works.
--
-- Revision 1.13  2006/10/26 01:51:28  jsedwards
-- Merged alpha_05_branch back into main trunk.
--
-- Revision 1.12.2.2  2006/10/25 12:22:29  jsedwards
-- Changed C_struct_class_definition to C_struct_Class_Definition so the case
-- is consistent with all the other C_struct objects.
--
-- Revision 1.12.2.1  2006/09/01 13:27:20  jsedwards
-- Changed "nwos_object_size" to "nwos_reference_list_size" and added the
-- object reference to "nwos_fill_in_common_header" so it can put the "id"
-- in the header now.
--
-- Revision 1.12  2006/01/12 03:43:07  jsedwards
-- Commented out print statement.
--
-- Revision 1.11  2006/01/10 14:09:17  jsedwards
-- Added "find phone for person" routine.
--
-- Revision 1.10  2006/01/01 21:49:01  jsedwards
-- Moved date, phone, us_state, and word class creations out of "big bang"
-- and into the respective files.
--
-- Revision 1.9  2006/01/01 21:08:38  jsedwards
-- Removed debugging print statement and added a blank line before printing
-- give up (can't find person and phone number looks bad) message.
--
-- Revision 1.8  2006/01/01 19:55:24  jsedwards
-- Added "parse phone number" and "parse phone number wo area code" routines
-- to deal with phone numbers and reworked the whole problems with phone
-- numbers section of getting a mobile phone.
--
-- Revision 1.7  2006/01/01 14:52:51  jsedwards
-- Added phone number creation stuff in part taken from "create_phone.c".
--
-- Revision 1.6  2005/12/29 17:50:53  jsedwards
-- Commented out printf debugging statements, that aren't useful now.
--
-- Revision 1.5  2005/12/27 18:32:36  jsedwards
-- Changed to look up class definition instead of using a fixed file name.
-- Also changed so that object id is random instead of based upon contents.
--
-- Revision 1.4  2005/12/24 16:18:26  jsedwards
-- Removed "host" id from object references (ObjRef).  Host redirection will
-- be done using a "redirection" object in the future.
--
-- Revision 1.3  2005/12/21 23:31:07  jsedwards
-- Added create_phone_number, find_phone_number, check phone number, etc.
--
-- Revision 1.2  2005/12/21 17:10:31  jsedwards
-- Add 'nwos' to 'create_all_area_codes' routine name.
--
-- Revision 1.1  2005/12/21 16:55:01  jsedwards
-- Initial version that adds all US area codes.
--
*/


#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>   /* define memset */

#include "crc32.h"
#include "objectify.h"
#include "objectify_private.h"



bool nwos_find_public_area_code(char* area_code, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Area_Code area_code_obj;
    ObjRef reference;
    ReferenceList* ref_list;
    int num_area_codes;
    int i;

    assert(strlen(area_code) == 3);

    assert(isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]));

    /* first find out if we already have this area_code */

    assert(nwos_find_public_class_definition("AREA CODE", &reference));

    nwos_read_class_definition(&reference, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_area_codes = ref_list->common_header.num_refs;

    /* printf("num_area_codes: %d\n", num_area_codes); */

    for (i = 0; i < num_area_codes; i++)
    {
	nwos_read_object_from_disk(&ref_list->references[i], &area_code_obj, sizeof(area_code_obj));

	if (strncmp(area_code_obj.storage, area_code, 3) == 0)   /* found a match */
	{
	    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
	    break;
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i < num_area_codes);  /* return true if we found it */
}


bool nwos_find_private_area_code(char* area_code, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Area_Code area_code_obj;
    ObjRef reference;
    ReferenceList* ref_list;
    int num_area_codes;
    int i;

    assert(strlen(area_code) == 3);

    assert(isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]));

    /* first find out if we already have this area_code */

    assert(nwos_find_private_class_definition("AREA CODE", &reference));

    nwos_read_class_definition(&reference, &class_def_obj);

    ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

    num_area_codes = ref_list->common_header.num_refs;

    /* printf("num_area_codes: %d\n", num_area_codes); */

    for (i = 0; i < num_area_codes; i++)
    {
	nwos_read_object_from_disk(&ref_list->references[i], &area_code_obj, sizeof(area_code_obj));

	if (strncmp(area_code_obj.storage, area_code, 3) == 0)   /* found a match */
	{
	    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
	    break;
	}
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;

    return (i < num_area_codes);  /* return true if we found it */
}


#ifdef PUBLIC_MODE

ObjCreateResult nwos_create_public_area_code(char* area_code, char* state_code, ObjRef* ref)
{
    C_struct_Area_Code area_code_obj;
    ObjRef area_code_class_ref;

    assert(strlen(area_code) == 3);
    assert(isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]));
    assert(strlen(state_code) == 2);
    assert(isalpha(state_code[0]) && isalpha(state_code[1]));

    assert(nwos_find_public_class_definition("AREA CODE", &area_code_class_ref));

    assert(!nwos_find_public_area_code(area_code, ref));

    memset(&area_code_obj, 0, sizeof(area_code_obj));  /* zero it out */

    nwos_generate_new_id(ref);

    nwos_fill_in_common_header(&area_code_obj.header.common, ref, &area_code_class_ref);

    memcpy(area_code_obj.storage, area_code, 3);

    nwos_create_reference_list(ref, &area_code_obj.header.object.references);

    nwos_find_state_from_postal_code(state_code, &area_code_obj.state);
    nwos_add_to_references(ref, &area_code_obj.state);

    nwos_crc32_calculate((uint8*) &area_code_obj.header.object, sizeof(ObjectHeader), area_code_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &area_code_obj.state, sizeof(area_code_obj) - sizeof(EveryObject), area_code_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &area_code_obj, sizeof(area_code_obj));

    nwos_add_to_references(ref, &area_code_class_ref);

    return CREATED_NEW;
}
#else

ObjCreateResult nwos_find_or_create_private_area_code(char* area_code, ObjRef* ref)
{
    C_struct_Area_Code area_code_obj;
    ObjRef area_code_class_ref;
    ObjRef public_ref;
    ObjCreateResult result = FOUND_EXISTING;

    assert(strlen(area_code) == 3);
    assert(isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]));

    if (!nwos_find_private_area_code(area_code, ref) == false)   /* didn't find it */
    {
	assert(nwos_find_public_area_code(area_code, &public_ref));

	nwos_read_object_from_disk(&public_ref, &area_code_obj, sizeof(area_code_obj));

	nwos_find_or_create_private_class_definition("AREA CODE", &area_code_class_ref);

	nwos_generate_new_id(ref);

	copy_reference(&area_code_obj.header.common.id, ref);
	copy_reference(&area_code_obj.header.common.class_definition, &area_code_class_ref);

	copy_reference(&area_code_obj.header.object.clone_of, &public_ref);

	nwos_create_reference_list(ref, &area_code_obj.header.object.references);

	nwos_crc32_calculate((uint8*) &area_code_obj.header.object, sizeof(ObjectHeader), area_code_obj.header.common.header_chksum);

	nwos_write_object_to_disk(ref, &area_code_obj, sizeof(area_code_obj));

        nwos_add_to_references(ref, &area_code_class_ref);

	result = CREATED_NEW;
    }

    return result;
}
#endif


bool nwos_is_valid_phone_number(char* number)
{
    return nwos_invalid_phone_number_msg(number) == NULL;
}

const char* nwos_invalid_phone_number_msg(char* number)
{
    static char msg[64];
    int i;
    size_t length = strlen(number);
    bool separator = false;

    for (i = 0; number[i] != '\0'; i++)
    {
	if (!isdigit(number[i]))
	{
	    /* ok to have hyphen or period in the fourth place*/
	    if (number[i] == '-' || number[i] == '.')
	    {
		if (separator)  /* already have one separator */
		{
		    return "multiple separators (- or .)";
		}

		if (i != 3)
		{
		    snprintf(msg, sizeof(msg), "separator '%c' in the wrong place", number[i]);
		    return msg;
		}

		separator = true;
	    }
	    else
	    {
		if (isgraph(number[i]))
		{
		    snprintf(msg, sizeof(msg), "unexpected character: %c\n", number[i]);
		}
		else
		{
		    snprintf(msg, sizeof(msg), "unexpected character: 0x%02x\n", (uint8)number[i]);
		}

		return msg;
	    }
	}
    }

    if (separator)
    {
	if (length < 8) return "not enough digits";
	if (length > 8) return "too many digits";
    }
    else
    {
	if (length < 7) return "not enough digits";
	if (length > 7) return "too many digits";
    }

    return NULL;   /* no errors */
}


static void copy_phone_number(char number[7], char* phone_number)
{
    assert(nwos_is_valid_phone_number(phone_number));

    memcpy(number, phone_number, 3);   /* copy first 3 */
    if (!isdigit(phone_number[3]))
    {
	memcpy(number+3, phone_number+4, 4);
    }
    else
    {
	memcpy(number+3, phone_number+3, 4);
    }
}


bool nwos_find_phone_number(char* area_code, char* phone_number, ObjRef* ref)
{
    C_struct_Class_Definition class_def_obj;
    C_struct_Phone_Number phone_number_obj;
    ObjRef reference;
    ObjRef area_code_ref;
    ReferenceList* ref_list;
    int num_phone_numbers;
    int i;
    char number[7];
    bool result = true;

    assert(nwos_is_valid_phone_number(phone_number));

    copy_phone_number(number, phone_number);

    if (area_code != NULL)
    {
#ifdef PUBLIC_MODE
	result = nwos_find_public_area_code(area_code, &area_code_ref);
#else
	result = nwos_find_private_area_code(area_code, &area_code_ref);
#endif
    }

    if (result)
    {
	assert(nwos_find_private_class_definition("PHONE NUMBER", &reference));

	nwos_read_class_definition(&reference, &class_def_obj);

	ref_list = nwos_malloc_reference_list(&class_def_obj.header.object.references);

	num_phone_numbers = ref_list->common_header.num_refs;

	/* printf("num_phone_numbers: %d\n", num_phone_numbers); */

	result = false;

	for (i = 0; i < num_phone_numbers; i++)
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &phone_number_obj, sizeof(phone_number_obj));

	    if (area_code == NULL || is_same_object(&phone_number_obj.area_code, &area_code_ref))
	    {
		if (memcmp(phone_number_obj.storage, number, 7) == 0)   /* found a match */
		{
		    memcpy(ref, &ref_list->references[i], sizeof(ObjRef));
		    result = true;
		    break;
		}
	    }
	}

	nwos_free_reference_list(ref_list);
	ref_list = NULL;
    }

    return result;
}


ObjCreateResult nwos_create_phone_number(char* area_code, char* phone_number, ObjRef* ref)
{
    C_struct_Phone_Number phone_number_obj;
    ObjRef reference;

    assert(nwos_is_valid_phone_number(phone_number));

    assert(!nwos_find_phone_number(area_code, phone_number, ref));

    memset(&phone_number_obj, 0, sizeof(phone_number_obj));  /* zero it out */

    nwos_find_or_create_private_class_definition("PHONE NUMBER", &reference);

    nwos_generate_new_id(ref);

    nwos_fill_in_common_header(&phone_number_obj.header.common, ref, &reference);

    nwos_create_reference_list(ref, &phone_number_obj.header.object.references);

    copy_phone_number(phone_number_obj.storage, phone_number);

    if (!nwos_find_private_area_code(area_code, &phone_number_obj.area_code))   /* didn't find it */
    {
	fprintf(stderr, "Unknown area code: %s\n", area_code);
	exit(1);
    }
    nwos_add_to_references(ref, &phone_number_obj.area_code);

    nwos_crc32_calculate((uint8*) &phone_number_obj.header.object, sizeof(ObjectHeader), phone_number_obj.header.common.header_chksum);

    nwos_crc32_calculate((uint8*) &phone_number_obj.country, sizeof(phone_number_obj) - sizeof(EveryObject), phone_number_obj.header.common.data_chksum);

    nwos_write_object_to_disk(ref, &phone_number_obj, sizeof(phone_number_obj));

    nwos_add_to_references(ref, &reference);

    return CREATED_NEW;
}


bool nwos_phone_number_to_string(ObjRef* ref, char* string, size_t size)
{
    C_struct_Phone_Number number_obj;
    C_struct_Area_Code area_code_obj;
    int i;
    int j;

    nwos_read_object_from_disk(ref, &number_obj, sizeof(number_obj));

    nwos_read_object_from_disk(&number_obj.area_code, &area_code_obj, sizeof(area_code_obj));

    i = 0;

    assert(i < size);
    string[i++] = '(';

    for (j = 0; j < 3; j++) 
    {
	assert(i < size);
	string[i++] = area_code_obj.storage[j];
    }

    assert(i < size);
    string[i++] = ')';

    assert(i < size);
    string[i++] = ' ';

    for (j = 0; j < 3; j++) 
    {
	assert(i < size);
	string[i++] = number_obj.storage[j];
    }

    assert(i < size);
    string[i++] = '-';

    for (j = 3; j < 7; j++) 
    {
	assert(i < size);
	string[i++] = number_obj.storage[j];
    }

    assert(i < size);
    string[i++] = '\0';

    return true;
}




/* FIX THIS! - need a common get input routine, instead of duplicating this in every file! */

static void get_input(char* descr, char* buffer, size_t size)
{
    char *ptr;

    while (1)
    {
	printf("%s: ", descr);
	fflush(stdout);
	fgets(buffer, size, stdin);
	ptr = strchr(buffer, '\n');
	if (ptr != NULL)
	{
	    *ptr = '\0';
	    break;
	}
	do { fgets(buffer, size, stdin); } while (strchr(buffer, '\n') == NULL);
	printf("input too long - try again!\n");
    }
}



static bool parse_phone_number(char* phone_number, char area_code[], char number[])
{
    static char* patterns[] =
    {
	"(AAA) NNN-NNNN",
	"AAA-NNN-NNNN",
	"AAA.NNN.NNNN",
	"AAA NNN NNNN",
    };
    const int num_patterns = sizeof(patterns) / sizeof(char*);
    char* ptrpat;
    char* ptrin;
    char* ptrac;
    char* ptrnum;
    int i;

    for (i = 0; i < num_patterns; i++)
    {
	ptrpat = patterns[i];
	ptrin = phone_number;
	ptrac = area_code;
	ptrnum = number;

	while (ptrpat != NULL && *ptrpat != '\0' && *ptrin != '\0')
	{
	    switch (*ptrpat)
	    {
	      case 'A':
		if (isdigit(*ptrin))
		{
		    *ptrac = *ptrin;
		    ptrin++;
		    ptrac++;
		    ptrpat++;
		}
		else    /* no match */
		{
		    ptrpat = NULL;
		}
		break;

	      case ' ':   /* spaces are optional */
		while (*ptrin == ' ') ptrin++;
		ptrpat++;
		break;

	      case 'N':
		if (isalnum(*ptrin))
		{
		    *ptrnum = *ptrin;
		    ptrin++;
		    ptrnum++;
		    ptrpat++;
		}
		else    /* no match */
		{
		    ptrpat = NULL;
		}
		break;

	      default:    /* verify it matches the pattern exactly */
		if (*ptrin == *ptrpat)
		{
		    ptrin++;
		    ptrpat++;
		}
		else   /* no match */
		{
		    ptrpat = NULL;
		}
		break;
	    }
	}

	if (ptrpat != NULL && *ptrpat == '\0' && *ptrin == '\0')    /* it matched */
	{
	    *ptrac = '\0';
	    *ptrnum = '\0';
	    break;
	}
    }

    assert(i == num_patterns || strlen(area_code) == 3);
    assert(i == num_patterns || strlen(number) == 7);

    return (i < num_patterns);
}


static bool parse_phone_number_wo_area_code(char* phone_number, char number[])
{
    static char* patterns[] =
    {
	"NNN-NNNN",
	"NNN.NNNN",
	"NNN NNNN",
    };
    const int num_patterns = sizeof(patterns) / sizeof(char*);
    char* ptrpat;
    char* ptrin;
    char* ptrnum;
    int i;

    for (i = 0; i < num_patterns; i++)
    {
	ptrpat = patterns[i];
	ptrin = phone_number;
	ptrnum = number;

	while (ptrpat != NULL && *ptrpat != '\0' && *ptrin != '\0')
	{
	    switch (*ptrpat)
	    {
	      case ' ':   /* spaces are optional */
		while (*ptrin == ' ') ptrin++;
		ptrpat++;
		break;

	      case 'N':
		if (isalnum(*ptrin))
		{
		    *ptrnum = *ptrin;
		    ptrin++;
		    ptrnum++;
		    ptrpat++;
		}
		else    /* no match */
		{
		    ptrpat = NULL;
		}
		break;

	      default:    /* verify it matches the pattern exactly */
		if (*ptrin == *ptrpat)
		{
		    ptrin++;
		    ptrpat++;
		}
		else   /* no match */
		{
		    ptrpat = NULL;
		}
		break;
	    }
	}

	if (ptrpat != NULL && *ptrpat == '\0' && *ptrin == '\0')    /* it matched */
	{
	    *ptrnum = '\0';
	    break;
	}
    }

    assert(i == num_patterns || strlen(number) == 7);

    return (i < num_patterns);
}


#ifndef PUBLIC_MODE
void nwos_add_mobile_phone(char* name, char* phone_number)
{
    ObjRef reference;
    char area_code[16];
    char number[16];
    C_struct_Area_Code area_code_obj;
    C_struct_US_State state_obj;
    C_struct_Mobile_Phone phone_obj;
    ObjRef area_code_ref;
    ObjRef phone_class_ref;
    ObjRef person_ref;
    char temp[64];
    bool number_ok;
    int i;

    /* parse the number first in case there is something wrong with it */

    number_ok = parse_phone_number(phone_number, area_code, number);

    if (!number_ok)    /* try parsing without the area code */
    {
	if (parse_phone_number_wo_area_code(phone_number, number))
	{
	    while (!number_ok)
	    {
		get_input("What is the area code (enter `quit' to return to main command)", area_code, 16);

		printf("\n");

		if (nwos_is_quit_command(area_code))    /* want's to bail out */
		{
		    printf("Ok, returning to main command.\n");
		    return;
		}

		if (isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]) && area_code[3] == '\0')
		{
		    number_ok = true;
		}
		else
		{
		    printf("Invalid area code: %s.  Please re-enter.\n\n", area_code);
		}
	    }
	}
    }

    while (number_ok && !nwos_find_public_area_code(area_code, &area_code_ref))
    {
	char* unimplemented_area_codes[] = { "800", "866", "877", "900" };
	const int num_unimplemented_area_codes = sizeof(unimplemented_area_codes) / sizeof(char*);
	for (i = 0; i < num_unimplemented_area_codes; i++)
	{
	    if (strcmp(area_code, unimplemented_area_codes[i]) == 0)
	    {
		printf("I'm sorry, this version of software cannot deal with toll free or pay\n");
		printf("numbers such as %s.  Returning to main command.\n", area_code);
		return;
	    }
	}

	printf("That area code is not in the database.  Unfortunately this version of the\n");
	printf("software does not have the capability to add an area code.  If it was a\n");
	printf("mistake, please re-enter the correct area code.  If not please check for\n");
	printf("a newer version of this software at http://sourceforge.net/projects/nwos/.\n");
	printf("\n");

	number_ok = false;

	while (!number_ok)
	{
	    get_input("What is the area code (enter `quit' to return to main command)", area_code, 16);

	    printf("\n");

	    if (nwos_is_quit_command(area_code))    /* want's to bail out */
	    {
		printf("Ok, returning to main command.\n");
		return;
	    }

	    if (isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]) && area_code[3] == '\0')
	    {
		number_ok = true;
	    }
	    else
	    {
		printf("Invalid area code: %s.  Please re-enter.\n\n", area_code);
	    }
	}
    }

    /* Not sure about the phone number at this point, let's see about the person */

    if (!nwos_find_person(name, &person_ref))
    {
	if (!number_ok)
	{
	    printf("I'm sorry I can't find %s in the database and there is something\n", name);
	    printf("wrong with the phone number given: %s.  Please try again.\n", phone_number);
	}

	return;
    }

    /* At this point we have a person, so make sure the phone number is sane */

    if (!number_ok)
    {
	if (!nwos_ask_yes_or_no("I can't figure out the phone number given", "Would you like to re-enter it"))
	{
	    printf("Ok, returning to main command.\n");
	    return;
	}

	while (!number_ok)
	{
	    get_input("What is the area code", area_code, 16);

	    printf("\n");

	    if (nwos_is_quit_command(area_code))    /* want's to bail out */
	    {
		printf("Ok, returning to main command.\n");
		return;
	    }

	    if (isdigit(area_code[0]) && isdigit(area_code[1]) && isdigit(area_code[2]) && area_code[3] == '\0')
	    {
		number_ok = true;
	    }
	    else
	    {
		printf("Invalid area code: %s.  Please re-enter.\n\n", area_code);
	    }
	}

	number_ok = false;

	while (!number_ok)
	{
	    get_input("What is the phone number", number, 16);

	    number_ok = nwos_is_valid_phone_number(number);

	    if (!number_ok)
	    {
		printf("Invalid phone number: %s - %s\n", number, nwos_invalid_phone_number_msg(number));
		printf("Please try again.\n");
	    }
	}
    }

    nwos_read_object_from_disk(&area_code_ref, &area_code_obj, sizeof(area_code_obj));

    nwos_read_object_from_disk(&area_code_obj.state, &state_obj, sizeof(state_obj));

    nwos_name_to_string(&state_obj.name, temp, 64);

    printf("Area code is in state: %s\n", temp);


    nwos_find_or_create_private_class_definition("MOBILE PHONE", &phone_class_ref);

    memset(&phone_obj, 0, sizeof(phone_obj));

    nwos_generate_new_id(&reference);

    nwos_fill_in_common_header(&phone_obj.header.common, &reference, &phone_class_ref);

    memcpy(&phone_obj.person, &person_ref, sizeof(ObjRef));

    if (nwos_find_phone_number(area_code, number, &phone_obj.number))
    {
	printf("Sorry, that phone number already exists!\n");
    }
    else
      {
	nwos_create_phone_number(area_code, number, &phone_obj.number);

	nwos_crc32_calculate((uint8*) &phone_obj.header.object, sizeof(ObjectHeader), phone_obj.header.common.header_chksum);

	nwos_crc32_calculate((uint8*) &phone_obj.person, sizeof(phone_obj) - sizeof(EveryObject), phone_obj.header.common.data_chksum);

	nwos_write_object_to_disk(&reference, &phone_obj, sizeof(phone_obj));

	nwos_create_reference_list(&reference, &phone_obj.header.object.references);

	nwos_add_to_references(&reference, &phone_obj.number);
	nwos_add_to_references(&reference, &phone_obj.person);
	    
	nwos_add_to_references(&reference, &phone_class_ref);
	
	printf("Mobile phone created: %02x%02x%02x%02x\n", reference.id[0], reference.id[1], reference.id[2], reference.id[3]);
    }
}


void nwos_find_phone_for_person(char* name)
{
    ObjRef person_ref;
    ObjRef phone_class_ref;
    ObjRef object_class;
    EveryObject header;
    C_struct_Mobile_Phone phone_obj;
    ReferenceList* ref_list;
    int num_refs;
    int i;
    char temp[20];

    if (!nwos_find_person(name, &person_ref))
    {
	printf("I'm sorry I can't find anyone with the name %s.\n", name);
	return;
    }

    /**********************************************/
    /* Now look for a phone in the reference list */
    /**********************************************/

    nwos_read_object_headers_from_disk(&person_ref, &header);

    ref_list = nwos_malloc_reference_list(&header.object.references);

    num_refs = ref_list->common_header.num_refs;

    /* printf("person num refs: %d\n", num_refs); */

    assert(nwos_find_private_class_definition("MOBILE PHONE", &phone_class_ref));

    for (i = 0; i < num_refs; i++)
    {
	nwos_get_object_class(&ref_list->references[i], &object_class);   /* find out what kind of object it is */

	if (is_same_object(&object_class, &phone_class_ref))
	{
	    nwos_read_object_from_disk(&ref_list->references[i], &phone_obj, sizeof(phone_obj));

	    nwos_phone_number_to_string(&phone_obj.number, temp, 20);
	    printf("%s ", temp);
	}

	printf("\n");
    }

    nwos_free_reference_list(ref_list);
    ref_list = NULL;
}
#endif

