# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020,2021,2022,2023,2024,2025 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import libxml2
import pymobius.p2p
import pymobius.p2p.account

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Iterate through nodes, getting attributes' IDs and values
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def iter_attributes (node, namespace=None):
  
  # node full namespace
  if namespace:
    name = namespace + '.' + node.name.lower ()
  else:
    name = node.name.lower ()

  # list properties
  for p in node.properties or []:
    if p.type == 'attribute':
      yield name + '.' + p.name.lower (), p.content

  # navigate through children
  child = node.children
  
  while child:
    if child.type == 'element':
      for attr_id, attr_value in iter_attributes (child, name):
        yield attr_id, attr_value

    child = child.next

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Retrieve data from Profile.xml
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
def retrieve (item, model, f, username):

  # Check file
  if not f:
    return

  reader = f.new_reader ()
  if not reader:
    return

  # Retrieve attributes
  xml = reader.read ().decode ('utf-8')
  doc = libxml2.parseDoc (xml)
  node = doc.getRootElement ()
  attr = {}

  for attr_id, attr_value in iter_attributes (node):
    attr[attr_id] = attr_value

  # Retrieve BitTorrent account
  bt_guid = attr.get ('gprofile.bittorrent.guid')

  if bt_guid:
    account = pymobius.p2p.account.account ()
    account.username = username
    account.app_id = 'shareaza'
    account.app = 'Shareaza'
    account.network = 'BitTorrent'
    account.guid = bt_guid.upper ()
    model.accounts.append (account)

  # Retrieve Gnutella account
  gnutella_guid = attr.get ('gprofile.gnutella.guid')

  if gnutella_guid:
    account = pymobius.p2p.account.account ()
    account.username = username
    account.app_id = 'shareaza'
    account.app = 'Shareaza'
    account.network = 'Gnutella'
    account.guid = gnutella_guid.upper ()
    model.accounts.append (account)

  # Set file handled
  pymobius.p2p.set_handled (item, f)
