/*===========================================================================*/
/*
 * This file is part of libogg++ - a c++ library for transport of the Ogg format
 *
 * Copyright (C) 2006, 2007, 2008 Elaine Tsiang YueLien
 *
 * libogg++ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/** @file multiThreadedDemux.C                                               *
 *                                                                           */
/* multiThreadedDemux - multi-threaded demultiplexor                         */
/*                                                                           */
/*===========================================================================*/
#include	<Ogg/Ogg.H>
#include	<Ogg/Debug.H>
#include	<Thread.H>


extern "C" 
{
#include	<stdlib.h>
#include	<string.h>
}

namespace	Examples
{
  using namespace	Ogg;

  class Parser : public Logical, public Thread
  {
    const char * codecName;

  public:
    Parser(
	   Transport &		transport
	   ,const char *	name
	   )
      : Logical(transport)
	, codecName(name)
    {}

    void
    run()
    {
      try
	{
	  size_t total = 0;
	  Reader & r = reader();
	  total += r->size();

	  // print first header packet
	  debug(true, codecName,
		": serial no ", serialNo()
		," got first header"
		);

	  do
	    {
	      ++r;
	      total += r->size();
	      // print subsequent packet #, size
	      debug(true, codecName,
		    " got ",r->packetNo(),"th packet, "
		    ,"size = ",r->size()
		    ," granule position = ",r.granulePosition()
		    ," total size = ",total
		    );
	    }
	  while ( !r.ending() );
	  debug(true, codecName,
		": last packet no.=",r->packetNo()
		," size = ",r->size()
		," total size=",total
		);

	}
      catch ( exception ex
	      )
	{
	  debug(true, codecName,
		ex.what()
		);
	  throw;
	}

      return;
    }
      
  }
  ;
  
  class VorbisParser : public Parser
  {
  public:
    VorbisParser(
		 Transport & 	transport
		 )
      : Parser(transport
	       ,"vorbis"
	       )
    {}

    bool
    selectCallback(
		   Packet &	pkt
		   )
    {
      const  char * header = static_cast<const  char *>(pkt.data());
      const unsigned char * uheader = static_cast<const unsigned char *>(pkt.data());

      if ( (pkt.size() < 30)
	   ||
	   (uheader[0] != 0x01)
	   ||
	   (strncmp (header+1, "vorbis", 6))
	   )
	return(false);
      else
	return(true);
    }

  };

  
  class TheoraParser : public Parser
  {
  public:
    TheoraParser(
		 Transport & transport
		 )
      : Parser(transport
	       ,"theora"
	       )
    {}


    bool
    selectCallback(
		   Packet &	pkt
		   )
    { 
      const  char * header = static_cast<const  char *>(pkt.data());
      const unsigned char * uheader = static_cast<const unsigned char *>(pkt.data());

      if ( (pkt.size() < 41)
	   ||
	   (uheader[0] != 0x80)
	   ||
	   (strncmp (header+1, "theora", 6))
	   )
	return(false);
      else
	return(true);
    }

  }
  ;

  class Demuxer: public Transport, public Thread
  {
  public:
    Demuxer()
      : Transport(false)
    {}

    ~Demuxer() throw(Ogg::Transport::NoDeletion())
    {}

    void
    run()
    {
      loop();
    }

    size_t
    recvCallback(
		 Page &	pg
		 )
    {
      size_t count = Transport::recvCallback(pg);

      if ( std::cin.eof()
	   ||
	   (count <= 0 )
	   )
	{
	  terminate();
	}

      return(count);
    }
  }
  ;

}

int main(int argc, const char* argv[])
{
  using namespace	Examples;
  using namespace	Ogg;

  Demuxer * transport = new Demuxer;
  VorbisParser * vorparser = new VorbisParser(*transport);
  TheoraParser * theoparser = new TheoraParser(*transport);

  transport->start();
  vorparser->start();
  theoparser->start();

  vorparser->finish();
  theoparser->finish();
  transport->finish();

  exit(0);
}
