/*===========================================================================*/
/*
 * This file is part of libogg++ - a c++ library for the Ogg transport format
 *
 * Copyright (C) 2006, 2007, 2008 Elaine Tsiang YueLien
 *
 * libogg++ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/** @file Logical.H                                                          *
 *                                                                           *
 *  Ogg::Logical to application interface                                    *
 *                                                                           */
/*===========================================================================*/
#ifndef LogicalToApplicationInterface
#define	LogicalToApplicationInterface

#ifdef __GNUG__
#pragma interface
#endif


namespace	Ogg
{
  class Logical;
  class Packet;
}

#include	<Ogg/Ogg.H>
#include	<Ogg/Transport.H>
#include	<Ogg/Exception.H>

namespace	Ogg
{
  /// @defgroup logical		Logical Interface
  //@{

  /// @brief Logical Packet interface
  ///
  /// @ingroup derivables
  ///
  /// Do not derive from Packet.
  /// Cast Packet::data() to suit codec needs.
  class Packet
  {
  private:
    Packet(const Packet &);

    Packet &
    operator =(const Packet &);

  protected:
    Packet(){};

  public:
    void
    data(void *
	 ,size_t
	 );

    void *
    data();		///< data area of packet
    
    size_t
    size();		///< size of data area

    PacketNo		///< decapsulated packet no.
    packetNo();

  };

  /// @brief Logical interface
  ///
  /// @ingroup derivables
  ///
  /// Codecs may derive from Logical.
  class Logical
  {
  private:
    void *		impl_;

    Logical(const Logical &);

    Logical &
    operator =(const Logical &);

  protected:
    /// @brief Despite appearance, derived classes should NOT construct with this
    Logical(
	    bool	dummy
	    )
      : impl_(0)
    {}

  public:
    void *
    impl()
    {
      return(impl_);
    }

    /*----- Exceptions thrown by a Logical thread ---------------------------------*/

    /// @brief Exception thrown by Logical::Logical
    class SerialNotPositive : public Exception
    {
    public:
      SerialNotPositive(
			const long	serialNo
			) throw()
	: Exception("Serial no. ")
      {
	*this << serialNo << "is not positive" << std::endl;
      }
      
      SerialNotPositive(
			const SerialNotPositive & ex
			) throw()
	: Exception(ex)
      {}
      
      ~SerialNotPositive() throw()
      {}
    }
    ;

    /// @brief Exception thrown by Logical::writer
    class WriterAlreadyExists : public Exception
    {
    public:
      WriterAlreadyExists(
			  const long	serialNo
			  ) throw()
	: Exception("Writer for this Logical already exists,")
      {
	*this << " serial no. = " << serialNo << std::endl;
      }
      
      WriterAlreadyExists(
			  const WriterAlreadyExists & ex
			  ) throw()
	: Exception(ex)
      {}
      
      ~WriterAlreadyExists() throw()
      {}
    }
    ;

    /// @brief Exception thrown by Logical::writer
    class NotForWriting : public Exception
    {
    public:
      NotForWriting(
		    const long	serialNo
		    ) throw()
	: Exception("This Logical not for writing")
      {
	*this << " serial no. = " << serialNo << std::endl;
      }

      NotForWriting(
		    const NotForWriting & ex
		    ) throw()
	: Exception(ex)
      {}

      ~NotForWriting() throw()
      {}
    }
    ;

    /// @brief Exception thrown by Logical::reader
    class ReaderAlreadyExists : public Exception
    {
    public:
      ReaderAlreadyExists(
			  const long	serialNo
			  ) throw()
	: Exception("Reader for this Logical already exists,")
      {
	*this << " serial no. = " << serialNo << std::endl;
      }
      
      ReaderAlreadyExists(
			  const ReaderAlreadyExists & ex
			  ) throw()
	: Exception(ex)
      {}
      
      ~ReaderAlreadyExists() throw()
      {}
    }
    ;

    /// @brief Exception thrown by Logical::reader
    class NotForReading : public Exception
    {
    public:
      NotForReading(
		    const long	serialNo
		    ) throw()
	: Exception("This Logical not for reading,")
      {
	*this << " serial no. = " << serialNo << std::endl;
      }

      NotForReading(
		    const NotForReading & ex
		    ) throw()
	: Exception(ex)
      {}

      ~NotForReading() throw()
      {}
    }
    ;

    ///@brief Exception thrown by muxing Logical::Logical
    class MuxTooLate : public Exception
    {
    public:
      MuxTooLate(
		 const long	serialNo
		 ) throw()
	: Exception("This Logical stream comes in too late - synchronize better,")
      {
	*this << " serial no. = " << serialNo << std::endl;
      }

      MuxTooLate(
		 const MuxTooLate & ex
		 ) throw()
	: Exception(ex)
      {}

      ~MuxTooLate() throw()
      {}
    }
    ;

    ///@brief Exception thrown by muxing Logical::Logical
    class SerialNoAlreadyUsed : public Exception
    {
    public:
      SerialNoAlreadyUsed(
			  const long	serialNo
			  ) throw()
	: Exception("This serial no. has been taken already,")
      {
	*this << " serial no. = " << serialNo << std::endl;
      }

      SerialNoAlreadyUsed(
			  const SerialNoAlreadyUsed & ex
			  ) throw()
	: Exception(ex)
      {}
      
      ~SerialNoAlreadyUsed() throw()
      {}
    }
    ;

    /// @brief Exception thrown by Logical::reader
    class StreamAlreadySelected : public Exception
    {
    public:
      StreamAlreadySelected(
			   const long	serialNo
			  ) throw()
	: Exception("Stream already has been selected,")
      {
	*this << " serial no. = " << serialNo << std::endl;
      }
      
      StreamAlreadySelected(
			   const StreamAlreadySelected & ex
			  ) throw()
	: Exception(ex)
      {}
      
      ~StreamAlreadySelected() throw()
      {}
    }
    ;

    /// @brief Exception thrown by Logical::reader
    class NoSuchStream : public Exception
    {
    public:
      NoSuchStream(
		   const long	serialNo
		   ) throw()
	: Exception("No stream found,")
      {
	*this << " serial no. = " << serialNo << std::endl;
      }

      NoSuchStream(
		   const NoSuchStream & ex
		   ) throw()
	: Exception(ex)
      {}
      
      ~NoSuchStream() throw()
      {}
    }
    ;

    /// @brief Exception thrown by Logical::reader
    class FailedToSelectStream : public Exception
    {
    public:
      FailedToSelectStream(
			   const long	serialNo
			   ) throw()
	: Exception("Failed to find a stream with ")
      {
	*this << " serial no. = " << serialNo << std::endl;
      }

      FailedToSelectStream(
			   const FailedToSelectStream & ex
			   ) throw()
	: Exception(ex)
      {}
      
      ~FailedToSelectStream() throw()
      {}
    }
    ;

    ///@brief Exception thrown by Logical::Reader:: seek operators
    class SeekInternalError : public Exception
    {
    public:
      SeekInternalError(
			const long	serialNo
			) throw()
	: Exception("Please let developers know this happened.")
      {
	*this << " serial no. = " << serialNo << std::endl;
      }

      SeekInternalError(
			const SeekInternalError & ex
			) throw()
	: Exception(ex)
      {}
      
      ~SeekInternalError() throw()
      {}
    }
    ;

    /// @brief Exception thrown by Logical::reader
    class PageReaderTooLate : public Exception
    {
    public:
      PageReaderTooLate(
			const long	serialNo
			) throw()
	: Exception("Request for PageReader must precede that for Reader.")
      {
	*this << " serial no. = " << serialNo << std::endl;
      }

      PageReaderTooLate(
			const PageReaderTooLate & ex
			) throw()
	: Exception(ex)
      {}
      
      ~PageReaderTooLate() throw()
      {}
    }
    ;

    /// @brief Exception thrown by Logical::reader
    class ReaderTooLate : public Exception
    {
    public:
      ReaderTooLate(
		    const long	serialNo
		    ) throw()
	: Exception("Logical::reader() must precede PageReader::get()")
      {
	*this << " serial no. = " << serialNo << std::endl;
      }

      ReaderTooLate(
		    const ReaderTooLate & ex
		    ) throw()
	: Exception(ex)
      {}
      
      ~ReaderTooLate() throw()
      {}
    }
    ;

    /*- end Exceptions thrown by a Logical thread ---------------------------------*/

    /// @brief Add a writing Logical to an encapsulating Transport
    Logical(
	    Transport & transport
	    ,long	serialNo
	    );
    
    /// @brief Construct a Logical for reading
    Logical(
	    Transport & transport
	    );

    /// @brief Construct a Logical for reading a particular logical stream
    Logical(
	    long	serialNo
	    ,Transport&	transport
	    );


    /// @brief Removes this Logical from Transport.
    ///
    /// @ingroup overrides
    virtual
    ~Logical();

    /// @brief Call back for logical stream selection based on first packet.
    ///
    /// @ingroup overrides
    ///
    /// Default: returns true for selection of whatever is available
    ///
    virtual
    bool
    selectCallback(
		   Packet &	firstPacket
		   )
    {
      return(true);
    }

    /// @brief Get the serialNo for this Logical
    ///
    /// Returns a writing Logical instance's set serialNo.
    /// If the Logical instance is reading, then
    /// this will return 0 if there is no current Reader.
    long
    serialNo() const;

    /// @brief Get the Transport for this Logical
    Transport &
    transport() const;

    /// @brief Logical stream writer
    /// @ingroup derivables
    ///
    /// Do not derive from Writer.
    /// It is an iterator-like class for writing packets.
    /// 
    class Writer
    {
    private:
      Writer(const Writer &);
      
      Writer &
      operator =(const Writer &);

    protected:
      Writer(){};

    public:
      /// @brief Ends writing this Logical stream.
      ///
      /// The user deletes the Writer instance,
      /// or the Writer instance falls out of scrope,
      /// which implicitly calls this destructor
      /// which closes the Logical stream.
      ///
      ~Writer();

      /// @brief Get the current size of the data payload.
      size_t
      dataSize() const;

      /// @brief Set the size of the next packet's data payload.
      /// 
      /// Be aware that the payload may be at a different address,
      /// invalidating any previous operator->() calls.
      void
      dataSize(size_t);

      /// @brief Flush previously enqued packets.
      void
      flush();
      
      /// @brief Set the current packet to begin data packets.
      ///
      /// Writer assumes header packets until beginData() is called.
      ///
      void
      beginData();

      /// @brief Set granule position of current packet.
      ///
      /// Default is last granulePosition, beginning with 0
      ///
      void
      granulePosition(
		      Position
		      );

      /// @brief Get granule position of the last enqued packet.
      Position
      granulePosition() const;

      /// @brief Get the current page size for transport.
      size_t
      pageSize() const;

      /// @brief Set size for transport pages 
      ///
      /// beginning with the page where the current packet begins.
      ///
      void
      pageSize(
	       size_t
	       );

      /// @brief Creates the next packet and enques the last.
      Writer &
      operator ++();

      /// @brief Get to the Packet methods.
      Packet *
      operator->() const throw();

      /// @brief Get the packet
      Packet *
      packet() const throw()
      {
	return(operator->());
      }
    };

    /// @brief Logical stream reader.
    ///
    /// @ingroup derivables
    ///
    /// Do not derive from Reader.
    /// It is an iterator-like class for reading packets.
    class Reader
    {
    private:
      Reader(const Reader &);
      
      Reader &
      operator =(const Reader &);

    protected:
      Reader(){};

    public:
      /// @brief Stops reading this Logical stream.
      ///
      /// The user may request a new reader after this.
      ///
      ~Reader();

      /// @brief Deletes the last packet dequed, and deques the next.
      Reader &
      operator ++();

      /// @brief Seek to packet @c packets from current packet.
      ///
      /// Deques the packet after a successful seek.
      ///
      Reader &
      operator +=(PacketNo packets);

      /// @brief Seek to packet at or greater than granule @c position.
      ///
      /// Deques the packet after a successful seek.
      ///
      Reader &
      operator =(Position position);

      bool
      ending() const;	///< ending packet?

      Error
      transportError()	///< packet error of current packet
	const;

      Position
      granulePosition()///< last known granule position
	const;

      /// @brief Get to the Packet methods.
      Packet *
      operator->() const throw();

      /// @brief Get the packet
      Packet *
      packet() const throw()
      {
	return(operator->());
      }
    };

    /// @brief Logical stream Page writer
    /// @ingroup derivables
    ///
    /// Do not derive from PageWriter.
    /// It is designed for writing pages
    /// that have been routed from a PageReader.
    /// 
    class PageWriter
    {
    private:
      PageWriter(const PageWriter &);
      
      PageWriter &
      operator =(const PageWriter &);

    protected:
      PageWriter(){};

    public:
      /// @brief Ends writing this Logical stream.
      ///
      /// The user deletes the PageWriter instance,
      /// or the PageWriter instance falls out of scrope,
      /// which implicitly calls this destructor
      /// which closes the Logical stream.
      ///
      ~PageWriter();

      /// @brief Enques the given page.
      ///
      /// Return true if successful.
      /// Returns false if queue is congested.
      /// Note once a page has been put,
      /// it should not be referenced again,
      /// as the page is sunk after being muxed.
      /// When that happens on the muxing thread is unpredictable.
      bool
      put(Page &);

      /// @brief Enques the given page.
      ///
      void
      doPut(Page &);

    };

    /// @brief Logical stream Page reader.
    ///
    /// @ingroup derivables
    ///
    /// Do not derive from PageReader.
    /// It is designed for reading raw (un-decapsulated) pages
    /// either simultaneously with reading the decapsulated packets,
    /// or by themselves. This allows logical streams to be
    /// ripped and merged into Ogg streams without re-encapsulation.
    class PageReader
    {
    private:
      PageReader(const PageReader &);
      
      PageReader &
      operator =(const PageReader &);

    protected:
      PageReader(){};

    public:
      /// @brief Stops reading this Logical stream.
      ///
      /// The user may request a new pageReader after this.
      ///
      ~PageReader();

      /// @brief Deques the next page.
      ///
      /// Null if none available. This call does not block.
      /// Any gotten page should be sunk by a PageWriter.
      /// Otherwise the page's memory will not be released.
      /// This may lead to exhaustion of available memory.
      Page *
      get();

      /// @brief Deques the next page.
      ///
      /// Blocks if none available.
      Page &
      doGet();
      
    };

    /// @brief Get a Writer for this Logical.
    ///
    /// Only one Writer per Logical at one time is allowed.
    /// In addition, a first Packet is created.
    Writer &
    writer();

    /// @brief Get a PageWriter for this Logical.
    ///
    /// Only one PageWriter or one Writer, not both
    /// per Logical at one time is allowed.
    PageWriter &
    pageWriter();


    /// @brief Get a PageReader for this Logical based on selectCallback
    ///
    /// Only one PageReader per Logical at one time is allowed.
    ///
    /// A PageReader may be used by itself on a logical stream, in which case
    /// it serves simply to provide the pages for merging into an Ogg stream
    /// without re-encapsulation.
    ///
    /// A PageReader may be used together with a Reader on the same logical stream.
    /// In this case, the pageReader() must be requested prior to requesting reader().
    /// Otherwise, the request will result in a PageReaderTooLate Exception.
    /// The reader() request must also precede the first PageReader::get() call.
    /// Otherwise, the reader() request will result in a ReaderTooLate Exception.
    PageReader &
    pageReader();

    /// @brief Get a Reader for this Logical based on selectCallback
    ///
    /// Only one Reader per Logical at one time is allowed.
    /// A new Reader may be requested after destroying a previous Reader.
    Reader &
    reader();

  };
  //@} logical
};
#endif
