/*===========================================================================*/
/*
 * This file is part of libogg++ - a c++ library for the Ogg transport format
 *
 * Copyright (C) 2006, 2007  Elaine Tsiang YueLien
 *
 * libogg++ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/** @file Logical.H                                                          *
 *                                                                           *
 *  Ogg::Logical to application interface                                    *
 *                                                                           */
/*===========================================================================*/
#ifndef LogicalToApplicationInterface
#define	LogicalToApplicationInterface

#ifdef __GNUG__
#pragma interface
#endif


namespace	Ogg
{
  class Logical;
  class Packet;
}

#include	<Ogg/Ogg.H>
#include	<Ogg/Transport.H>
#include	<Ogg/Exception.H>

namespace	Ogg
{
  /// @defgroup logical		Logical Interface
  //@{

  /// @brief Logical Packet interface
  ///
  /// @ingroup derivables
  ///
  /// Do not derive from Packet.
  /// Cast Packet::data() to suit codec needs.
  class Packet
  {
  public:
    void *
    data();		///< data area of packet
    
    size_t
    size();		///< size of data area

    PacketNo		///< decapsulated packet no.
    packetNo();

  };

  /// @brief Logical interface
  ///
  /// @ingroup derivables
  ///
  /// Codecs may derive from Logical.
  class Logical
  {
  private:
    void *		impl_;

  protected:
    /// @brief Despite appearance, derived classes should NOT construct with this
    Logical(
	    bool	dummy
	    )
      : impl_(0)
    {}

  public:
    void *
    impl()
    {
      return(impl_);
    }

    /// @brief Add a writing Logical to an encapsulating Transport
    Logical(
	    Transport & transport
	    ,long	serialNo
	    );
    
    /// @brief Construct a Logical for reading
    Logical(
	    Transport & transport
	    );


    /// @brief Removes this Logical from Transport.
    ///
    /// @ingroup overrides
    virtual
    ~Logical();

    /// @brief Call back for logical stream selection based on first packet.
    ///
    /// @ingroup overrides
    ///
    /// Default: returns true for selection of whatever is available
    ///
    virtual
    bool
    selectCallback(
		   Packet &	firstPacket
		   )
    {
      return(true);
    }

    /// @brief Get the serialNo for this Logical
    long
    serialNo();

    /// @brief Get the Transport for this Logical
    Transport &
    transport();

    /// @brief Logical stream writer
    /// @ingroup derivables
    ///
    /// Do not derive from Writer.
    /// It is an iterator-like class for writing packets.
    /// 
    class Writer
    {
    public:
      /// @brief Ends writing this Logical stream.
      ///
      /// The user deletes the Writer instance,
      /// or the Writer instance falls out of scrope,
      /// which implicitly calls this destructor
      /// which closes the Logical stream.
      ///
      ~Writer();

      /// @brief Get the current size of the data payload.
      size_t
      dataSize();

      /// @brief Set the size of the next packet's data payload.
      void
      dataSize(size_t);

      /// @brief Flush previously enqued packets.
      void
      flush();
      
      /// @brief Set the current packet to begin data packets.
      ///
      /// Writer assumes header packets until beginData() is called.
      ///
      void
      beginData();

      /// @brief Set granule position of current packet.
      ///
      /// Default increments last granulePosition by 1.
      ///
      void
      granulePosition(
		      Position
		      );

      /// @brief Get granule position of the last enqued packet.
      Position
      granulePosition();

      /// @brief Get the current page size for transport.
      size_t
      pageSize();

      /// @brief Set size for transport pages 
      ///
      /// beginning with the page where the current packet begins.
      ///
      void
      pageSize(
	       size_t
	       );

      /// @brief Creates the next packet and enques the last.
      Writer &
      operator++();

      /// @brief Get to the Packet methods.
      Packet *
      operator->() const throw();
    };

    /// @brief Logical stream reader.
    ///
    /// @ingroup derivables
    ///
    /// Do not derive from Reader.
    /// It is an iterator-like class for reading packets.
    class Reader
    {
    public:
      /// @brief Stops reading this Logical stream.
      ///
      /// The user may request a new reader after this.
      ///
      ~Reader();

      /// @brief Deletes the last packet dequed, and deques the next.
      Reader &
      operator++();

      /// @brief Seek to packet @c packets from current packet.
      ///
      /// Deques the packet after a successful seek.
      ///
      Reader &
      operator+=(PacketNo packets);

      /// @brief Seek to packet at or greater than granule @c position.
      ///
      /// Deques the packet after a successful seek.
      ///
      Reader &
      operator=(Position position);

      bool
      ending() const;	///< ending packet?

      Error
      transportError()	///< packet error of current packet
	const;

      Position
      granulePosition()///< last known granule position
	const;

      /// @brief Get to the Packet methods.
      Packet *
      operator->() const throw();
    };

    /// @brief Get a Writer for this Logical.
    ///
    /// Only one Writer per Logical at one time is allowed.
    /// In addition, a first Packet is created.
    Writer &
    writer();


    /// @brief Get a Reader for this Logical based on selectCallback
    ///
    /// Only one Reader per Logical at one time is allowed.
    /// A new Reader may be requested after destroying a previous Reader.
    Reader &
    reader();

    /// @brief Exception thrown by Logical::Writer
    class NotForWriting : public Exception
    {
    public:
      NotForWriting(
		    const long	serialNo
		    ) throw()
	;

      NotForWriting(
		    const NotForWriting & ex
		    ) throw()
	: Exception(ex)
      {}

      ~NotForWriting() throw()
      {}
    }
    ;

    /// @brief Exception thrown by Logical::Reader
    class ReaderAlreadyExists : public Exception
    {
    public:
      ReaderAlreadyExists(
			  const long	serialNo
			  ) throw()
	;
      
      ReaderAlreadyExists(
			  const ReaderAlreadyExists & ex
			  ) throw()
	: Exception(ex)
      {}
      
      ~ReaderAlreadyExists() throw()
      {}
    }
    ;

    /// @brief Exception thrown by Logical::Reader
    class NotForReading : public Exception
    {
    public:
      NotForReading(
		    const long	serialNo
		    ) throw()
	;

      NotForReading(
		    const NotForReading & ex
		    ) throw()
	: Exception(ex)
      {}

      ~NotForReading() throw()
      {}
    }
    ;

    ///@brief Exception thrown by Logical::Logical
    class MuxTooLate : public Exception
    {
    public:
      MuxTooLate(
		 const long	serialNo
		 ) throw()
	;

      MuxTooLate(
		 const MuxTooLate & ex
		 ) throw()
	: Exception(ex)
      {}

      ~MuxTooLate() throw()
      {}
    }
    ;

    ///@brief Exception thrown by Logical::Logical
    class SerialNoAlreadyUsed : public Exception
    {
    public:
      SerialNoAlreadyUsed(
			  const long	serialNo
			  ) throw()
	;
      
      SerialNoAlreadyUsed(
			  const SerialNoAlreadyUsed & ex
			  ) throw()
	: Exception(ex)
      {}
      
      ~SerialNoAlreadyUsed() throw()
      {}
    }
    ;

    class StreamAlreadyClaimed : public Exception
    {
    public:
      StreamAlreadyClaimed(
			   const long	serialNo
			  ) throw()
	;
      
      StreamAlreadyClaimed(
			   const StreamAlreadyClaimed & ex
			  ) throw()
	: Exception(ex)
      {}
      
      ~StreamAlreadyClaimed() throw()
      {}
    }
    ;

    class NoSuchStream : public Exception
    {
    public:
      NoSuchStream(
		   const long	serialNo
		   ) throw()
	;
      
      NoSuchStream(
		   const NoSuchStream & ex
		   ) throw()
	: Exception(ex)
      {}
      
      ~NoSuchStream() throw()
      {}
    }
    ;

    class FailedToSelectStream : public Exception
    {
    public:
      FailedToSelectStream(
			   const long	serialNo
			   ) throw()
	;
      
      FailedToSelectStream(
			   const FailedToSelectStream & ex
			   ) throw()
	: Exception(ex)
      {}
      
      ~FailedToSelectStream() throw()
      {}
    }
    ;

    ///@brief Exception thrown by Logical::Reader:: seek operators
    class SeekInternalError : public Exception
    {
    public:
      SeekInternalError(
			const long	serialNo
			) throw()
	;
      
      SeekInternalError(
			const SeekInternalError & ex
			) throw()
	: Exception(ex)
      {}
      
      ~SeekInternalError() throw()
      {}
    }
    ;

  };
  //@} logical
};
#endif
