/*
* Copyright (C) 2011 Sunil Mohan Adapa <sunil@medhas.org>.
* Copyright (C) 2011 O S K Chaitanya <osk@medhas.org>.
*
* Author: Sunil Mohan Adapa <sunil@medhas.org>
*         O S K Chaitanya <osk@medhas.org>
*
* This file is part of GNOME Nonogram.
*
* GNOME Nonogram is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* GNOME Nonogram is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with GNOME Nonogram. If not, see <http://www.gnu.org/licenses/>.
*/

const Lang = imports.lang;
const GObject = imports.gi.GObject;
const Gtk = imports.gi.Gtk;

const Config = imports.config;
const NonogramIO = imports.nonogramio;

GameWidget = new GType(_GameWidget = {
    parent: Gtk.Table.type,
    name: "GameWidget",
    properties: [
        {
            name: "puzzleFile",
            type: GObject.TYPE_STRING,
            default_value: null,
            flags: GObject.ParamFlags.READABLE
                 | GObject.ParamFlags.WRITABLE
                 | GObject.ParamFlags.CONSTRUCT
        }
    ],
    signals: [
        {
            name: "undo_redo_available",
            parameters: [GObject.TYPE_BOOLEAN, GObject.TYPE_BOOLEAN]
        }
    ],

    _BACKGROUND_COLOR: 0xffffffff,
    _NO_COLOR: 0,
    _MIN_PIXEL_SIZE: 10,

    __puzzleFile: null,
    _solutionPuzzlePixels: null,
    _numRows: 0,
    _numColumns: 0,

    _rowClueTable: null,
    _columnClueTable: null,
    _pixelGrid: null,
    _palette: null,
    _correctSolutionDialog: null,
    _incorrectSolutionDialog: null,
    _incorrectSolutionDialogShown: false,

    class_init: function(klass, prototype) {
        prototype._BACKGROUND_COLOR = _GameWidget._BACKGROUND_COLOR;
        prototype._NO_COLOR = _GameWidget._NO_COLOR;
        prototype._MIN_PIXEL_SIZE = _GameWidget._MIN_PIXEL_SIZE;

        prototype.__puzzleFile = _GameWidget.__puzzleFile;
        prototype._solutionPuzzlePixels = _GameWidget._solutionPuzzlePixels;
        prototype._numRows = _GameWidget._numRows;
        prototype._numColumns = _GameWidget._numColumns;

        prototype._rowClueTable = _GameWidget._rowClueTable;
        prototype._columnClueTable = _GameWidget._columnClueTable;
        prototype._pixelGrid = _GameWidget._pixelGrid;
        prototype._palette = _GameWidget._palette;
        prototype._correctSolutionDialog = _GameWidget._correctSolutionDialog;
        prototype._incorrectSolutionDialog =
            _GameWidget._incorrectSolutionDialog;
        prototype._incorrectSolutionDialogShown =
            _GameWidget._incorrectSolutionDialogShown;

        prototype.__defineGetter__("_puzzleFile", _GameWidget._getPuzzleFile);
        prototype.__defineSetter__("_puzzleFile", _GameWidget._setPuzzleFile);
        prototype.__defineGetter__("puzzleFile", _GameWidget._getPuzzleFile);
        prototype.__defineSetter__("puzzleFile", _GameWidget._setPuzzleFile);

        prototype.connectSignalsAndChildren =
            _GameWidget.connectSignalsAndChildren;
        prototype.restart = _GameWidget.restart;
        prototype._acquireChildren = _GameWidget._acquireChildren;
        prototype._listColorIndices = _GameWidget._listColorIndices;
        prototype._updateClueTables = _GameWidget._updateClueTables;
        prototype._updatePixelGrid = _GameWidget._updatePixelGrid;
        prototype._updatePalette = _GameWidget._updatePalette;
        prototype._onPuzzleFull = _GameWidget._onPuzzleFull;
        prototype._isPuzzleSolved = _GameWidget._isPuzzleSolved;
        prototype._onPuzzleSolved = _GameWidget._onPuzzleSolved;
        prototype._onPuzzleIncorrectSolution =
            _GameWidget._onPuzzleIncorrectSolution;
        prototype._onClueTableAllCluesSatisfied =
            _GameWidget._onClueTableAllCluesSatisfied;
        prototype._setPixelGridMinimumSize =
            _GameWidget._setPixelGridMinimumSize;
        prototype.undo = _GameWidget.undo;
        prototype.redo = _GameWidget.redo;
        prototype._emitUndoRedoAvailability =
            _GameWidget._emitUndoRedoAvailability;
    },

    init: function() {
    },

    connectSignalsAndChildren: function(builder) {
        this._acquireChildren(builder);

        this._pixelGrid.signal.puzzle_changed.connect(
            Lang.bind(this._rowClueTable, this._rowClueTable.onPuzzleChanged));
        this._pixelGrid.signal.puzzle_changed.connect(
            Lang.bind(this._columnClueTable,
                      this._columnClueTable.onPuzzleChanged));
        this._pixelGrid.signal.highlight_changed.connect(
            Lang.bind(this._rowClueTable,
                      this._rowClueTable.onHighlightChanged));
        this._pixelGrid.signal.highlight_changed.connect(
            Lang.bind(this._columnClueTable,
                      this._columnClueTable.onHighlightChanged));
        this._pixelGrid.signal.puzzle_full.connect(
            Lang.bind(this, this._onPuzzleFull));
        this._pixelGrid.signal.undo_redo_available.connect(
            Lang.bind(this, this._emitUndoRedoAvailability));

        this._rowClueTable.signal.all_clues_satisfied.connect(
            Lang.bind(this, this._onClueTableAllCluesSatisfied));
        this._columnClueTable.signal.all_clues_satisfied.connect(
            Lang.bind(this, this._onClueTableAllCluesSatisfied));

        this._palette.connectSignalsAndChildren(builder);
        this._palette.signal.color_selected.connect(
            Lang.bind(this,
                function(palette, color) {
                    this._pixelGrid.drawingColor = color;
                }
            )
        );
    },

    restart: function() {
        this._pixelGrid.reset();
        this._pixelGrid.set_sensitive(true);
        this._rowClueTable.reset();
        this._columnClueTable.reset();
    },

    _acquireChildren: function(builder) {
        this._rowClueTable = builder.get_object("rowcluetable");
        this._columnClueTable = builder.get_object("columncluetable");
        this._pixelGrid = builder.get_object("pixelgrid");
        this._palette = builder.get_object("palette");

        builder.add_from_file(Config.uiDataDir + "/puzzle-end-dialog.ui");
        this._correctSolutionDialog =
            builder.get_object("correct_solution_dialog");
        this._correctSolutionDialog.transient_for = this.get_toplevel();
        // TODO: Clean up this mess
        var dashboardButton = this._correctSolutionDialog.add_button(
            "Go to Dashboard", 0);
        var dashboardAction = builder.get_object("switch_to_dashboard_action");
        dashboardButton.set_use_action_appearance(false);
        dashboardButton.set_related_action(dashboardAction);
        var dashboardStockId = dashboardAction.get_stock_id();
        var dashboardImage = new Gtk.Image.from_stock(dashboardStockId,
                                                      Gtk.IconSize.BUTTON);
        dashboardButton.set_image(dashboardImage);
        this._correctSolutionDialog.add_button("gtk-close",
                                               Gtk.ResponseType.CLOSE);

        this._incorrectSolutionDialog =
            builder.get_object("incorrect_solution_dialog");
        this._incorrectSolutionDialog.transient_for = this.get_toplevel();
    },

    _getPuzzleFile: function() {
        return this.__puzzleFile;
    },

    _setPuzzleFile: function(value) {
        try {
            this._solutionPuzzlePixels = NonogramIO.load_puzzle(value);
        } catch (e) {
            if (e.name == "PuzzleLoadFileError")
                print("File error while loading puzzle: " + e.message);
            else if (e.name == "PuzzleLoadImageError")
                print("Image error while loading puzzle: " + e.message);
            return;
        }

        this.__puzzleFile = value;
        this._numRows = this._solutionPuzzlePixels.length;
        this._numColumns = this._solutionPuzzlePixels[0].length;

        this._listColorIndices();

        this._updateClueTables();
        this._updatePixelGrid();
        this._updatePalette();

        this._incorrectSolutionDialogShown = false;
    },

    _listColorIndices: function() {
        this._colorIndices = {};
        for (var y = 0; y < this._numRows; ++y) {
            for (var x = 0; x < this._numColumns; ++x) {
                var color = this._solutionPuzzlePixels[y][x];
                if (this._colorIndices[color] === undefined)
                    this._colorIndices[color] = 1;
                else
                    ++this._colorIndices[color];
            }
        }
    },

    _updateClueTables: function() {
        this._rowClueTable.setSolutionPuzzlePixels(
            this._solutionPuzzlePixels);
        this._columnClueTable.setSolutionPuzzlePixels(
            this._solutionPuzzlePixels);
    },

    _updatePixelGrid: function() {
        this._pixelGrid.createBlank(this._numColumns, this._numRows);
        this._setPixelGridMinimumSize();
        this._pixelGrid.set_sensitive(true);
    },

    _updatePalette: function() {
        this._palette.populateFromColorIndices(this._colorIndices,
            this._BACKGROUND_COLOR);
    },

    _onPuzzleFull: function() {
        if (this._isPuzzleSolved() == false)
            this._onPuzzleIncorrectSolution();
    },

    _isPuzzleSolved: function() {
        if (this._rowClueTable.allClueLinesAreSatisfied &&
            this._columnClueTable.allClueLinesAreSatisfied) {
            return true;
        }

        return false;
    },

    _onPuzzleSolved: function() {
        this._pixelGrid.set_sensitive(false);
        this._pixelGrid.revealRemaining();
        this._rowClueTable.markAllCluesSatisfied();
        this._columnClueTable.markAllCluesSatisfied();

        this._correctSolutionDialog.run();
        this._correctSolutionDialog.hide();
    },

    _onPuzzleIncorrectSolution: function() {
        if (this._incorrectSolutionDialogShown)
            return;

        this._incorrectSolutionDialogShown = true;

        this._incorrectSolutionDialog.run();
        this._incorrectSolutionDialog.hide();
    },

    _onClueTableAllCluesSatisfied: function() {
        if (this._isPuzzleSolved())
            this._onPuzzleSolved();
    },

    _setPixelGridMinimumSize: function() {
        if (this._numRows == 0 || this._numColumns == 0) {
            return;
        }

        var rowClueTableMinimumHeight =
            this._rowClueTable.get_preferred_height().minimum_height;
        var columnClueTableMinimumWidth =
            this._columnClueTable.get_preferred_width().minimum_width;

        var columnClueLineWidth = columnClueTableMinimumWidth
                                  / this._numColumns;
        var rowClueLineHeight = rowClueTableMinimumHeight
                                / this._numRows;

        var pixelSize = Math.max(
            columnClueLineWidth, rowClueLineHeight, this._MIN_PIXEL_SIZE);

        this._pixelGrid.set_size_request(pixelSize * this._numColumns,
                                         pixelSize * this._numRows);
    },

    undo: function() {
        this._pixelGrid.undo();
    },

    redo: function() {
        this._pixelGrid.redo();
    },

    _emitUndoRedoAvailability: function(pixelGrid,
                                        undoAvailable,
                                        redoAvailable) {
        this.signal.undo_redo_available.emit(undoAvailable, redoAvailable);
    }
});
