/*
 * Fisoco : FInd, Select, Order and COnvert files
 * Copyright 2015-2016 (C) Felicien PILLOT <felicien.pillot@member.fsf.org>
 *
 * This file is part of Fisoco.
 *
 * Fisoco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Fisoco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Fisoco.  If not, see <http://www.gnu.org/licenses/>.
 *
 *//////////////////////////////////////
/***************
 *    cli.cc    *
 *    ~~~~~~    *
 ****************/

#include "../include/cli.h"
#include "../include/global.h"
#include "../include/search.h"

#include <sys/stat.h>
#include <fstream>
#include <time.h>

void
Cli::backup ()
{
  string name = currentDateTime ();
  backup (name);
}
 
void
Cli::backup (std::string filename)
{
  filename += ".bak.fis";
  ofstream backup_file;
  string path = "/var/tmp/fisoco";
  mkdir (path.c_str(), S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH);
  int i = 1;
  std::list<std::string> list_names = results.get_complete_names ();
  if (list_names.size () != 0)
    {
      backup_file.open (path + "/" + filename);
      backup_file << "# This is generated data" << std::endl
		  << "# Please do not modify" << std::endl;
      for (std::list<std::string>::iterator complete_name = list_names.begin ();
	   complete_name != list_names.end (); complete_name++, i++)
	{
	  backup_file << i << "\t- " << *complete_name << std::endl;
	}
      backup_file.close ();
      std::cout << "List of files saved as " << filename << "." << std::endl;
    }
  else
    std::cout << "Cannot backup: list is empty" << std::endl;
}
 
void
Cli::clean ()
{
  results = List ();
}

// Get current date/time, format is YYYY-MM-DD.HH:mm:ss
const std::string
Cli::currentDateTime() {
    time_t     now = time(0);
    struct tm  tstruct;
    char       buf[80];
    tstruct = *localtime(&now);
    // Visit http://en.cppreference.com/w/cpp/chrono/c/strftime
    // for more information about date/time format
    strftime(buf, sizeof(buf), "%Y-%m-%d.%X", &tstruct);
    return buf;
}

std::string
Cli::get_dir_term ()
{
  return dir_term;
}

std::string
Cli::get_search_term ()
{
  return search_term;
}

int
Cli::load ()
{
  std::cout << gettext ("Scanning backup files...") << std::endl;
  clean ();
  dir_term = "/var/tmp/fisoco";
  search_term = ".bak.fis";
  search ();
  // Give some time to search ()
  for (int i = 0; i <= 3; i++)
    {
      if (results.get_list_size () != 0)
	break;
      else
	sleep (1);
    }
  
  if (results.get_list_size () != 0)
    std::cout <<
      results.get_list_size () <<
      gettext (" backup file(s) found ! Type 'read' to see them,") <<
      gettext (" then type 'load x' where x is the number of the") <<
      gettext (" backup file you want to load.") << std::endl;
  else
    {
      std::cout << "No backup files found !" << std::endl;
      return 1;
    }
  return 0;
}

int
Cli::load (int number)
{
  std::vector<std::string> infos = results.get_infos (number);
  // Check if number is good
  if (infos.size () == 0)
    error ("Backup file unknown : check the list by typing 'load'");
  // Check if file is a backup
  else if (infos.at (0).find (".bak.fis") == std::string::npos)
    error ("This is not a Fisoco backup file !");
  else
    {
      debug ("Loading backup file.");
      // Clear the old list, to refill it with loaded file
      clean ();
      string name = infos.at(1) + "/" + infos.at(0); 
      ifstream backup_file (name);
      if (backup_file.is_open())
	{
	  string line;
	  while ( getline (backup_file, line) )
	    {
	      std::list<std::string> file = get_file_from_line (line);
	      if (file.size () == 2)
		// Add path and filename to the new list
		results.add_file (file.front (), file.back ());
	    }
	  backup_file.close();
	  if (results.get_list_size () == 0)
	    error ("Nothing loaded. Does backup was empty ? Corrupted ?");
	  else
	    std::cout <<
	      gettext ("Backup file ") << number <<
	      gettext ("# has been loaded !") <<
	      gettext (" Type 'read' to see the new list") << std::endl;
	}
      else
	{
	  error ("Unable to open this file");
	  error ("Have you forgotten to run 'load' before this command ?");
	}
    }
  return 0;
}

std::list<std::string>
Cli::get_file_from_line (std::string line)
{
  std::list<std::string> file;
  if (line.substr (0,1) != "#")
    {
      file.push_back (line.substr
		      (line.find ("/"),
		       (line.find_last_of ("/") - line.find ("/"))
		       )
		      );
      file.push_back (line.substr
		      (line.find_last_of ("/")+1)
		      );
    }
  return file;
}
