/*
 * Fisoco, a FInding SOrting and COnverting free software
 * Copyright 2015-2016 (C) Felicien PILLOT <felicien.pillot@member.fsf.org>
 *
 * This file is part of Fisoco.
 *
 * Fisoco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Fisoco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Fisoco.  If not, see <http://www.gnu.org/licenses/>.
 *
 *//////////////////////////////////////
/****************
 *   search.cc  *
 *    ~~~~~~    *
 ****************/

#include "../include/cli.h"
#include "../include/global.h"
#include "../include/gui.h"
#include "../include/search.h"

Search::Search () :
  search_shall_stop (false),
  search_has_stopped (false)
{
}

void
Search::stop_work ()
{
    std::cout << gettext ("Stopping process...") << std::endl;
    search_shall_stop = true;
}

bool
Search::has_stopped () const
{
  return search_has_stopped;
}

void
Search::do_work (Gui* caller, Glib::ustring search, std::string dir)
{
    if (search_shall_stop == false)
    {
        if (search.length () == 0)
            caller->
            error_dialog (gettext
            ("No search keywords given !"));
        std::list<std::string> results = getdir (search, dir);
        if (results.size () == 0)
            caller->
            error_dialog (gettext
            ("This directory doesn't exists or is unreadable..."));
        else
            recursive_search (results, search, dir, caller);
    }
    else
    {
        search_shall_stop = false;
        search_has_stopped = true;
    }
    caller->notify ();
}

void
Search::do_work_only (Cli* caller, Glib::ustring search, std::string dir)
{
    if (search_shall_stop == false)
    {
        if (search.length () == 0)
            error (gettext ("No search keywords given !"));
        std::list<std::string> results = getdir (search, dir);
        if (results.size () == 0)
            error (gettext ("This directory doesn't exists or is unreadable..."));
        else
            recursive_search_only (results, search, dir, caller);
    }
    else
    {
        search_shall_stop = false;
        search_has_stopped = true;
    }
}

void
Search::recursive_search (std::list<std::string> results,
                          Glib::ustring search, std::string dir, Gui* caller)
{
  std::list<std::string> files = extr_files (search, results, dir);
  if (files.size () != 0)
    {
      caller->notify (files, dir);
    }
  std::list<std::string> directories = extr_dirs (results, dir);
  if (directories.size() > 0 && !search_shall_stop)
    {
        for (std::list<std::string>::iterator d = directories.begin ();
             d != directories.end (); ++d)
             {
                 std::string path = (dir != "/" ) ? (dir + "/" + *d) : (dir + *d);
                 std::list<std::string> new_results  = getdir (search, path);
                 recursive_search (new_results, search, path, caller);
             }
    }
}

void
Search::recursive_search_only (std::list<std::string> results,
                               Glib::ustring search, std::string dir, Cli* caller)
{
    std::list<std::string> files = extr_files (search, results, dir);
    if (files.size () != 0)
    {
        caller->notify (files, dir);
    }
    std::list<std::string> directories = extr_dirs (results, dir);
    if (directories.size() > 0 && !search_shall_stop)
    {
        for (std::list<std::string>::iterator d = directories.begin ();
             d != directories.end (); ++d)
             {
                 std::string path = (dir != "/" ) ? (dir + "/" + *d) : (dir + *d);
                 std::list<std::string> new_results  = getdir (search, path);
                 recursive_search_only (new_results, search, path, caller);
             }
    }
}

std::list<std::string>
Search::getdir (Glib::ustring search, std::string dir)
{
    std::list<std::string> files;
    DIR *dp;
    struct dirent *dirp;
    dir = (dir == "") ? "/" : dir;
    if ((dp = opendir (dir.c_str ())) == NULL)
        //    std::cout << gettext ("Error opening ") << dir << std::endl;
        ;
    else
        while ((dirp = readdir (dp)) != NULL)
            files.push_back (dirp->d_name);
        closedir (dp);
    return files;
}

std::list<std::string>
Search::extr_files (std::string search, std::list<std::string> res, std::string dir)
{
    std::list<std::string> files;
    for (std::list<std::string>::iterator file = res.begin (); file != res.end (); ++file)
    {
        if (get_type(*file, dir) == "File" &&
            (std::search (file->begin (), file->end (), 
                          search.begin (), search.end () ) != file->end () ))
            files.push_back (*file);
    }
    return files;
}

std::list<std::string>
Search::extr_dirs (std::list<std::string> res, std::string pdir)
{
    std::list<std::string> dirs;
    for (std::list<std::string>::iterator dir = res.begin (); dir != res.end (); ++dir)
    {
        if ((dir->back() != '.') &&             // We need this for the moment, because
            get_type(*dir, pdir) == "Directory" && *dir != "proc") // '/proc/*' contains a lot of junk
            dirs.push_back (*dir);              // and we don't know how to manage it...
    }
    return dirs;
}

std::string
Search::get_type (std::string file, std::string dir)
{
    struct stat st;
    std::string path = dir + "/" + file;
    stat (path.c_str (), &st);
    if (S_ISDIR (st.st_mode))
        return "Directory";
    else if (S_ISREG (st.st_mode))
        return "File";
    else
        return "error";
}
