/* 
    Copyright (C) 2024, 2025, Johannes Merten <coldemail@posteo.net>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
    */

declare name "cptdronesynth";
declare description "drone synth";
declare author "Johannes Merten";
declare copyright "Johannes Merten (coldemail@posteo.net)";
declare version "0.16";
declare license "GPL 3.0 or later";


si = library("signals.lib");
ba = library("basics.lib");
ma = library("maths.lib");
en = library("envelopes.lib");
ef = library("misceffects.lib");
ho = library("hoa.lib");
os = library("oscillators.lib");
no = library("noises.lib");
fi = library("effect.lib");

constant = environment {
	     feedbackhighpassfactor = 0.9;
	   };

gate = button("[-1]gate");
gain=hslider("[00]gain",1,0,1,0.01);

oscGroup(x) = hgroup("[02]OSCs",x);
s = oscGroup(hslider("[0]vco 1[lv2:integer] Off,Sawtooth, Square,  Sparse-Noise, Noise",0,0,4,2));
t = oscGroup(hslider("[1]vco 2[lv2:integer] Off,  Sawtooth, Square,  Sparse-Noise, Noise",0,0,4,2));
l = oscGroup( hslider("[2]vco 3[lv2:integer] Off, Sawtooth, Square, Triangle, Noise",0,0,4,2));

balanceGroup(x) = hgroup("[03]Balance",x);
vcoabal = balanceGroup (hslider("[0][lv2:integer]vco 1 balance",0,-63,63,1));
vcobbal = balanceGroup (hslider("[1][lv2:integer]vco 2 balance",0,-63,63,1));
vcocbal = balanceGroup (hslider("[2][lv2:integer]vco 3 balance",0,-63,63,1));

mixGroup(x) = hgroup("[04]Mix",x);
vcomix = mixGroup (hslider("[0][lv2:integer]vco mix",50,0,100,1)/100);
attrel = mixGroup (hslider("[1][lv2:integer]vco attack/release",1,0,127,1));
moddepth    = mixGroup (hslider("[2]vco 3 modulation index",0,0,1000,0.1));

mixctFreq(x) = hgroup("[05]Cutoff freq",x);
vcoactfreq = mixctFreq (hslider("[0][unit:Hz][lv2:integer]vco 1 lowpass/highpass", 0, -9000, 9000, 1));
vcobctfreq = mixctFreq (hslider("[1][unit:Hz][lv2:integer]vco 2 lowpass/highpass", 0, -9000, 9000, 1));
vcocctfreq = mixctFreq (hslider("[2][unit:Hz][lv2:integer]vco 3 lowpass/highpass", 0, -9000, 9000, 1));

noteGroup(x) = hgroup("[06]Note",x);
freq  =  noteGroup (hslider("[0][lv2:integer]key",60,0,127,1)) :  si.smooth(ba.tau2pole(glissandosec) ) : ba.midikey2hz;
glissandosec = noteGroup (hslider("[1][lv2:integer]glissando",0,0,60,0.1));
pitch = noteGroup (hslider("[2][unit:Hz][lv2:integer]vco pitch",0,-63,+63,1));

vcocGroup(x) = hgroup("[07]vco 3",x);
bpm= vcocGroup (hslider("[0][lv2:integer]vco 3/feedback bpm",60,0,127,1));
vcocshiftslider = vcocGroup (hslider("[2][lv2:integer]vco 3 shift",0,-63,63,1));


effectsGroup(x) = hgroup("[08]Effects",x);
flanger =  effectsGroup (hslider("[1]flanger", 0, 0, 1, 0.01));
feedback = effectsGroup (hslider("[2][lv2:integer]feedback",0,0,100,1)/99);
feedbackshift = effectsGroup (hslider("[3][lv2:integer]feedback shift",0,-24,24,1));

//all vco's     
//vco1-4 for vco 1/2
//vco5-8 for vco 3
vco1 = os.sawtooth(freq+pitch+(vcoc * moddepth));
vco2 = os.square(freq+pitch+(vcoc * moddepth));
vco3 = no.sparse_noise(freq+pitch+(vcoc * moddepth));
vco4 =  no.noise;
vco5 =  os.sawtooth(freq+pitch);
vco6 = os.square(freq+pitch);
vco7 = os.triangle(freq+pitch);
vco8 =  no.noise;

//transpose vco via "shift" slider
vcocshift = ef.transpose(24,24,vcocshiftslider);

// trigger attack and release of vco
vcoexec(vco,trigger,triggernumber)=  en.asr(attrel,1,attrel,(trigger==triggernumber))* vco;

//vco 1
vcoa = vcoexec(vco1,s,1)
       + vcoexec(vco2,s,2)
       + vcoexec(vco3,s,3)
       + vcoexec(vco4,s,4)
       :>/(4)<:attach(_,abs : ba.linear2db : hbargraph("[09]vco 1 out",-60,0));

//vco 2
vcob = vcoexec(vco1,t,1)
       + vcoexec(vco2,t,2)
       + vcoexec(vco3,t,3)
       + vcoexec(vco4,t,4)
       :>/(4)<:attach(_,abs : ba.linear2db : hbargraph("[10]vco 2 out",-60,0));

//vco 3
vcoc = vcoexec(vco5,l,1) : vcocshift
       + vcoexec(vco6,l,2) : vcocshift
       + vcoexec(vco7,l,3) : vcocshift
       + vcoexec(vco8,l,4) : vcocshift
       :>/(4)<:attach(_,abs : ba.linear2db : hbargraph("[11]vco 3 out",-60,0));

//pulse sound for vco 3
pulse = en.arfe(atrt, atrt, (1-finallevel),time) 
with {
    finallevel= vcocGroup (hslider("[1]vco 3 level",1,0.1,1,0.01));
    atrt = ba.samp2sec(ma.SR)*60/bpm;
    time =  ba.pulsen(ma.SR*60/bpm,(ma.SR*60/bpm)*2);
};

//flanger
flangereffect = (_<: flanger * (_<:@(ma.SR*envelopephase)*(en.ar(atrt,atrt,pulse)),(_) :>_), (_*dry):>_)
with {
  phase = 0.011;
  shift = freq / 10000;
  dry = (1 - flanger);
  atrt =  ba.samp2sec(ma.SR) * phase;
  pulse = ba.pulsen(phase * ma.SR,2 * phase * ma.SR);
  envelopephase = en.ar(shift,shift,pulse);
};

//feedback
feedbackeffect = +~@(delay) * feedback : feedbackhighpass 
with {
  delay = (bpm + feedbackshift,1:max)  / 60 * ma.SR;
  feedbackhighpass = _ <: (fi.highpass(1,freq)) , (_) :> _
  with {
    freq = 10000 * constant.feedbackhighpassfactor;
  };
};

//balance calculation
balanceleft(vcobal) = ((vcobal/63) * (-1)) + 1 : (sqrt);
balanceright(vcobal) = ((vcobal / 63)) + 1 : (sqrt);

vcoaballeft = balanceleft(vcoabal);
vcoabalright = balanceright(vcoabal);

vcobballeft = balanceleft(vcobbal);
vcobbalright = balanceright(vcobbal);

vcocballeft = balanceleft(vcocbal);
vcocbalright = balanceright(vcocbal);

//high- and lowpass calculation
lowpass(freq) = fi.lowpass(1, (10000+freq));
highpass(freq) = fi.highpass(1, ( (freq >0) * freq)+1);

//process the vco's, pulse for vco 3, gain, balance, low- and highpass and feedback for channel 1/2

process =
//left channel
           vcoc  * pulse *(1-vcomix) * gain  * vcocballeft
           :lowpass(vcocctfreq):highpass(vcocctfreq)

	+ (vcoa * vcomix/2 * gain  * vcoaballeft
	   :lowpass(vcoactfreq):highpass(vcoactfreq))

        +(vcob * vcomix/2  * gain  * vcobballeft
	  :lowpass(vcobctfreq) :highpass(vcobctfreq))

	    //flanger left channel
	    :flangereffect
	     //feedback left channel
	    : feedbackeffect
	     //bar left channel
	     <:attach(_,abs : ba.linear2db : hbargraph("[12]total out left",-60,0))/2,
	
//right channel
	   (
	     vcoc  * pulse *(1-vcomix) *gain * vcocbalright
	     :lowpass(vcocctfreq) :highpass(vcocctfreq)

	  + (vcoa * vcomix/2 * gain *vcoabalright 
     	     :lowpass(vcoactfreq) :highpass(vcoactfreq))
	  
	  +(vcob * vcomix/2  * gain  *vcobbalright
	    :lowpass(vcobctfreq) :highpass(vcobctfreq))

	      //flanger right channel
	     :flangereffect	  
	       //feedback right channel
	     : feedbackeffect
	       //bar right channel
	       <:attach(_,abs : ba.linear2db : hbargraph("[13]total out right",-60,0))/2
);
