/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/***************************************************************************
 *            tabs.c
 *
 *  Thu Oct 14 23:58:52 2004
 *  Copyright  2004  Gergely POLONKAI
 *  polesz@techinfo.hu
 ***************************************************************************/
/***************************************************************************
 * Notebook tab handling routines
 ***************************************************************************/

#include "botcommander.h"

#include <vte/vte.h>
#include <gtk/gtk.h>
#include <string.h>

#include "typedefs.h"
#include "variables.h"
#include "functions.h"

tab_data_t *tab_list;

/*
 * tab_close_button_pressed()
 * Called whenever the close button of a tab was pressed
 */
static void
tab_close_button_pressed(GtkWidget *widget, gpointer data)
{
	if (!data)
		return;
	
	remove_tab((tab_data_t *)data);
}

/*
 * add_tab()
 * Adds a new tab. TODO: param where is not used yet
 */
gboolean
add_tab(void)
{
	tab_data_t *temp;
	int w, h;
	
	if (tab_list == NULL)
	{
		if ((tab_list = g_malloc(sizeof(tab_data_t))) == NULL)
			return FALSE;
		tab_list->next = NULL;
		temp = tab_list;
	}
	else
	{
		temp = tab_list;
		while (temp->next)
			temp = temp->next;
		if ((temp->next = g_malloc(sizeof(tab_data_t))) == NULL)
			return FALSE;
		temp = temp->next;
	}
	temp->next = NULL;
	temp->vbox = gtk_vbox_new(FALSE, 0);

	/* Create the virtual terminal widget and pack it into the vertical box. Braces are for me (anjuta code folding is great) */
  {
		temp->vte = (VteTerminal *)vte_terminal_new();
		vte_terminal_set_encoding(VTE_TERMINAL(temp->vte), "UTF-8");
		vte_terminal_set_allow_bold(VTE_TERMINAL(temp->vte), ALLOW_BOLD);
		vte_terminal_set_scroll_on_output(VTE_TERMINAL(temp->vte), config_data.scroll_on_output);
		vte_terminal_set_background_transparent(VTE_TERMINAL(temp->vte), config_data.transparent_background);
		vte_terminal_set_background_saturation(VTE_TERMINAL(temp->vte), config_data.background_saturation);
		vte_terminal_set_audible_bell(VTE_TERMINAL(temp->vte), AUDIBLE_BELL);
		vte_terminal_set_visible_bell(VTE_TERMINAL(temp->vte), VISIBLE_BELL);
		vte_terminal_set_font_from_string(VTE_TERMINAL(temp->vte), config_data.vte_font);
		gtk_signal_connect(GTK_OBJECT(temp->vte), "clicked", GTK_SIGNAL_FUNC(vte_clicked), NULL);
		GTK_WIDGET_UNSET_FLAGS(GTK_WIDGET(temp->vte), GTK_CAN_FOCUS);
		gtk_signal_connect(GTK_OBJECT(temp->vte), "focus-in-event", GTK_SIGNAL_FUNC(grab_focus), NULL);
		gtk_box_pack_start(GTK_BOX(temp->vbox), GTK_WIDGET(temp->vte), TRUE, TRUE, 0);
		gtk_widget_show(GTK_WIDGET(temp->vte));
	}
	
	/* Create the command line and pack it into the vertical box. */
	temp->commandline = gtk_entry_new();
	g_signal_connect(G_OBJECT(temp->commandline), "activate", G_CALLBACK(activate_event), temp);
	gtk_signal_connect(GTK_OBJECT(temp->commandline), "key_press_event", GTK_SIGNAL_FUNC(commandline_keypress), temp);
	gtk_box_pack_start(GTK_BOX(temp->vbox), temp->commandline, FALSE, TRUE, 0);
	gtk_widget_show(temp->commandline);
	
	gtk_widget_show(temp->vbox);
	
	/* The box where the button and the label goes */
	temp->hbox = gtk_hbox_new(FALSE, 0);

  /* The close button */
	gtk_icon_size_lookup(GTK_ICON_SIZE_MENU, &w, &h);

	/* Create the button, and get the image from the stock */
	temp->closebutton = gtk_button_new();
	temp->image = gtk_image_new_from_stock(GTK_STOCK_CLOSE, GTK_ICON_SIZE_MENU);

	/* Add the image to the button */
	gtk_container_add(GTK_CONTAINER(temp->closebutton), temp->image);

	/* Disable the frame of the button, it's so ugly */
	gtk_button_set_relief(GTK_BUTTON(temp->closebutton), GTK_RELIEF_NONE);
	/* Connect the signal to the button */
	gtk_signal_connect(GTK_OBJECT(temp->closebutton), "clicked", GTK_SIGNAL_FUNC(tab_close_button_pressed), temp);
	/* Resize the button */
	gtk_widget_set_size_request(temp->closebutton, w, h);
	
	/* Label of the tab */
	temp->label = gtk_label_new("New tab");
	gtk_box_pack_start(GTK_BOX(temp->hbox), temp->label, FALSE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(temp->hbox), temp->closebutton, FALSE, TRUE, 0);

	/* Show the button and the label */
	gtk_widget_show(temp->label);
	gtk_widget_show(temp->image);
	gtk_widget_show(temp->closebutton);
	
	/* Now append the ready made page to the notebook */
	temp->index = gtk_notebook_append_page(GTK_NOTEBOOK(static_data.main_book), temp->vbox, temp->hbox);
	
	/* Set the new tab's default mode */
	temp->current_mode = 'B';
	
	temp->connected = FALSE;
	temp->want_echo = TRUE;
	memset(&(temp->connect_data), 0, sizeof(temp->connect_data));
	temp->socket = -1;
	temp->socket_tag = -1;
	
	/* Since this is the active widget, set the focus of its command line */
	gtk_widget_grab_focus(temp->commandline);

	return TRUE;
}

/*
 * get_active_tab()
 * Returns a pointer to the active tab's data
 */
tab_data_t *
get_active_tab(void)
{
	tab_data_t *temp;
	gint active_index = gtk_notebook_get_current_page(GTK_NOTEBOOK(static_data.main_book));
	
	if (tab_list == NULL)
		return NULL;
	
	if (tab_list->index == active_index)
		return tab_list;
	
	temp = tab_list;
	
	while (temp->next)
	{
		if (temp->index == active_index)
			return temp;
		temp = temp->next;
	}
	
	return NULL;
}

/*
 * remove_tab()
 * Removes the given tab
 */
void
remove_tab(tab_data_t *tab)
{
	tab_data_t *temp;

	if (!tab)
		return;
	
	if (tab->connected)
	{
		feed_error_to_terminal(tab, TRUE, "Won't close a connected tab yet");
		return;
	}
	
	if (tab == tab_list)
	{
		tab_list = tab_list->next;
		gtk_notebook_remove_page(GTK_NOTEBOOK(static_data.main_book), tab->index);
		temp = tab_list;
		g_free(tab);
		while (temp)
		{
			temp->index = gtk_notebook_page_num(GTK_NOTEBOOK(static_data.main_book), temp->vbox);
			temp = temp->next;
		}
	}
	else
	{
		tab_data_t *prev = NULL;
		
		temp = tab_list;
		while (temp)
		{
			if (temp == tab)
			{
				if (prev)
					prev->next = tab->next;
				temp = temp->next;
				gtk_notebook_remove_page(GTK_NOTEBOOK(static_data.main_book), tab->index);
				g_free(tab);
				tab = NULL;
			}
			else
			{
				temp->index = gtk_notebook_page_num(GTK_NOTEBOOK(static_data.main_book), temp->vbox);
				prev = temp;
				temp = temp->next;
			}
		}
	}
}

/*
 * get_tab_by_socket_tag()
 * Gets the tab which socket tag belongs to
 */
tab_data_t *
get_tab_by_socket_tag(gint tag)
{
	tab_data_t *temp;
	
	temp = tab_list;
	
	while (temp)
	{
		if (temp->socket_tag == tag)
			return temp;
		
		temp = temp->next;
	}
	
	return NULL;
}

/*
 * change_terminals_set_scroll_on_output()
 * Changes all terminals' setting on scrolling when data comes from the bot
 */
void
change_terminals_set_scroll_on_output(gboolean activate)
{
	tab_data_t *temp;
	
	temp = tab_list;
	
	while (temp)
	{
		if (temp->vte)
			vte_terminal_set_scroll_on_output(VTE_TERMINAL(temp->vte), activate);
		temp = temp->next;
	}
}

/*
 * change_terminals_set_background_transparent()
 * Changes all terminals' setting on background's transparency
 */
void
change_terminals_set_background_transparent(gboolean activate)
{
	tab_data_t *temp;
	
	temp = tab_list;
	
	while (temp)
	{
		if (temp->vte)
			vte_terminal_set_background_transparent(VTE_TERMINAL(temp->vte), activate);
		temp = temp->next;
	}
}

/*
 * change_terminals_set_font_name()
 * Changes all terminals' font
 */
void
change_terminals_set_font_name(gchar *font_name)
{
	tab_data_t *temp;
	
	temp = tab_list;
	
	while (temp)
	{
		if (temp->vte)
			vte_terminal_set_font_from_string(VTE_TERMINAL(temp->vte), font_name);
		temp = temp->next;
	}
}

/*
 * change_terminals_set_background_saturation()
 * Changes all terminals' background saturation
 */
void
change_terminals_set_background_saturation(gfloat value)
{
	tab_data_t *temp;
	
	temp = tab_list;
	
	while (temp)
	{
		if (temp->vte)
			vte_terminal_set_background_saturation(VTE_TERMINAL(temp->vte), value);
		temp = temp->next;
	}
}
