/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/***************************************************************************
 *            main.c
 *
 *  Sat Aug 21 23:02:47 2004
 *  Copyright  2004  Gergely Polonkai
 *  polesz@techinfo.hu
 ***************************************************************************/
/***************************************************************************
 * Functions which doesn't fit in other categories like networking and such
 ***************************************************************************/

#include "botcommander.h"

#include <gtk/gtk.h>
#include <netdb.h>
#include <string.h>
#include <arpa/inet.h>
#include <errno.h>
#include <unistd.h>

#include "botcommander.h"
#include "variables.h"
#include "functions.h"

/* XML support is only planned yet, so we give the user an error message */
#if (CONFIG_BACKEND == CB_XML)
# error Sorry, XML is not coded yet...
#endif

static_data_t static_data;
gchar *package_version;
gchar *package_version_feed;

/*
 * exit_cleanly()
 * Exits cleanly, closes all connections, destroys widgets. If force is FALSE,
 * a dialog appears, asking the user if (s)he really wants to quit
 */
void
exit_cleanly(gboolean force)
{
	tab_data_t *temp = tab_list;
	gboolean has_connected_tab = FALSE;

	botlist_clear(FALSE);
	
	history_free();
	
	while (temp)
	{
		if (temp->connected)
			has_connected_tab = TRUE;
		temp = temp->next;
	}
	
	if (has_connected_tab && !force)
	{
		/* TODO: write a dialog here... */
		return;
	}
	gtk_main_quit();
}

/*
 * change_mode()
 * Changes the current mode, and does all the stuff this requires (changing menu
 * and toolbar items, updating status bar
 */
gboolean
change_mode(tab_data_t *tab, gboolean from_menu_callback, gchar mode)
{
	GtkWidget *item = NULL;
	
	if (tab == NULL)
		return TRUE;
	
	switch (mode)
	{
		case 'B':
			item = static_data.menu_mode_b;
			break;
		case 'M':
			item = static_data.menu_mode_m;
			break;
		case 'E':
			item = static_data.menu_mode_e;
			break;
		default:
			item = NULL;
			break;
	}
	
	if (item)
	{
		if (!from_menu_callback)
		{
			gtk_signal_emit_by_name(GTK_OBJECT(item), "activate");
			return TRUE;
		}
	
		char buf[8];

		if (config_data.debug_mode)
			feed_info_to_terminal(tab, TRUE, "Changed to %s mode.", (mode == 'B') ? "BotCommander command" : ((mode == 'E') ? "Eggdrop command" : "message"));

		tab->current_mode = mode;
		sprintf((char *)&buf, "Mode: %c", mode);
		gtk_label_set_text(GTK_LABEL(static_data.main_statuslabel_mode), buf);
	}
	else
		return FALSE;
	
	return TRUE;
}

/*
 * connect_to_host()
 * Connects to the given host. Will be moved to tabs.c, and it will be tab specific
 */
gboolean
connect_to_host(tab_data_t *tab, gchar *hostname, short port)
{
	struct hostent *h;
	
	if ((h = gethostbyname(hostname)) == NULL)
	{
		feed_error_to_terminal(tab, TRUE, "Cannot resolve %s", hostname);
		return FALSE;
	}
	
	tab->connect_data.sin_family = AF_INET;
	tab->connect_data.sin_addr = *((struct in_addr *)h->h_addr);
	tab->connect_data.sin_port = htons(port);
	memset(&(tab->connect_data.sin_zero), 0, 8);
	
	feed_info_to_terminal(tab, TRUE, "Connecting to %s:%d...", inet_ntoa(tab->connect_data.sin_addr), ntohs(tab->connect_data.sin_port));
	
	if ((tab->socket = socket(PF_INET, SOCK_STREAM, 0)) < 0)
	{
		feed_error_to_terminal(tab, TRUE, "Unable to create socket (%s).", strerror(errno));
		return FALSE;
	}
	
	if (connect(tab->socket, (struct sockaddr *)&(tab->connect_data), sizeof(struct sockaddr)) < 0)
	{
		feed_error_to_terminal(tab, TRUE, "Unable to connect to the remote host (%s).", strerror(errno));
		return FALSE;
	}
	
	if (config_data.debug_mode)
		feed_info_to_terminal(tab, TRUE, "Connected.");
	
	if (config_data.change_to_message)
		change_mode(tab, FALSE, 'M');
	
	tab->connected = TRUE;

	tab->socket_tag = gdk_input_add(tab->socket, GDK_INPUT_READ, data_ready, tab);
	
	set_title(tab, "%s - %s:%d", package_version, inet_ntoa(tab->connect_data.sin_addr), ntohs(tab->connect_data.sin_port));
	tab->want_echo = TRUE;
	
	/* TODO: Change window title bar to the connected bot's data (host:port or botname) */
	
	return TRUE;
}

/*
 * data_ready()
 * Called when data is ready on a network socket
 */
void
data_ready(gpointer data, gint source, GdkInputCondition cond)
{
	gchar buf[3000];
	gint numbytes;
	tab_data_t *tab = data;
	char *converted_text;
	
	if (!tab)
		return;
	
	if ((numbytes = recv(tab->socket, buf, 2048, 0)) == -1)
	{
		feed_error_to_terminal(tab, TRUE, strerror(errno));
		return;
	}
	
	buf[numbytes] = '\0';
	
	if (numbytes == 0)
	{
		feed_info_to_terminal(tab, TRUE, "Disconnected.");
		set_title(tab, "%s", package_version);
		gtk_entry_set_visibility(GTK_ENTRY(tab->commandline), TRUE);
		tab->want_echo = TRUE;
		close(tab->socket);
		gdk_input_remove(tab->socket_tag);
		change_mode(tab, FALSE, 'B');
		tab->connected = FALSE;
		return;
	}
	
	if ((strstr((char *)&buf, TLN_WILLECHO)))
	{
		char *twe = strstr((char *)&buf, TLN_WILLECHO);
		*twe = ' ';
		*(twe + 1) = ' ';
		*(twe + 2) = ' ';
		tab->want_echo = FALSE;
		gtk_entry_set_visibility(GTK_ENTRY(tab->commandline), FALSE);
		if (config_data.debug_mode)
			feed_info_to_terminal(tab, TRUE, "Requested not to echo");
	}
	if ((strstr((char *)&buf, TLN_WONTECHO)))
	{
		char *twe = strstr((char *)&buf, TLN_WONTECHO);
		*twe = ' ';
		*(twe + 1) = ' ';
		*(twe + 2) = ' ';
		gtk_entry_set_visibility(GTK_ENTRY(tab->commandline), TRUE);
		tab->want_echo = TRUE;
		if (config_data.debug_mode)
			feed_info_to_terminal(tab, TRUE, "Want echo");
	}
	
	if ((converted_text = iconvert((gchar *)&buf, TRUE)))
	{
		feed_message_to_terminal(tab, TRUE, converted_text);
		g_free(converted_text);
	}
}

/*
 * main()
 * Do I really have to tell you what this function for?
 */
int
main(int argc, char **argv)
{
	gtk_init(&argc, &argv);
	
	package_version = create_version_string(FALSE);
	package_version_feed = create_version_string(TRUE);
	static_data.prefs_window = NULL;
	static_data.botlist_window = NULL;
	tab_list = NULL;
	bot_list = NULL;
	
	if (!config_init())
	{
		fprintf(stderr, "config_init() returned FALSE, something is wrong...\n");
		exit_cleanly(TRUE);
		return 1;
	}
	
	if (!history_init(TRUE))
	{
		fprintf(stderr, "history_init() returned FALSE, something is wrong...\n");
		exit_cleanly(TRUE);
		return 1;
	}

	if (!create_main_window())
	{
		GtkWidget *dialog;
		
		dialog = gtk_message_dialog_new(NULL,
		                                0,
		                                GTK_MESSAGE_ERROR,
		                                GTK_BUTTONS_OK,
		                                "There was an error while creating the main window.");
		gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
		gtk_dialog_run(GTK_DIALOG(dialog));

		return 1;
	}
	
	add_tab();
	botlist_read_from_config();
	
	gtk_main();

	return (0);
}
