/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/***************************************************************************
 *            commands.c
 *
 *  Wed Aug 25 22:17:41 2004
 *  Copyright  2004  Gergely POLONKAI
 *  polesz@techinfo.hu
 ***************************************************************************/
/***************************************************************************
 * Functions that handle BotCommander commands
 ***************************************************************************/

#define _GNU_SOURCE
#include <ctype.h>
#include <netdb.h>
#include <stdlib.h>
#include <netinet/in.h>
#include <string.h>
#include <arpa/inet.h>
#include <stdio.h>
#include <unistd.h>

#include "typedefs.h"
#include "functions.h"
#include "defines.h"
#include "variables.h"

char **command_history;
gint history_position = -1;

/*
 * I have to prototype these, but I don't want to define them yet, because it's
 * more beautiful this way.
 */
BC_COMMAND(command_exit);
BC_COMMAND(command_mode);
BC_COMMAND(command_connect);
BC_COMMAND(command_close);

/*
 * command[]
 * Command definitions
 */
command_t command[] = {
	{"QUIT",    CT_INTERNAL, command_exit},
	{"EXIT",    CT_INTERNAL, command_exit},
	{"MODE",    CT_INTERNAL, command_mode},
	{"CONNECT", CT_INTERNAL, command_connect},
	{"CLOSE",   CT_INTERNAL, command_close},
};

#define num_commands (sizeof(command) / sizeof(command_t))

/*
 * BC_COMMAND(*)
 * Command callbacks.
 */
BC_COMMAND(command_exit)
{
	exit_cleanly(FALSE);
	
	/* We should never reach here, but if we do, it is bad... */
	return FALSE;
}

BC_COMMAND(command_mode)
{
	if (paramnum == 0)
	{
		feed_error_to_terminal(tab, TRUE, "Usage: MODE (M|E|B)");
		return FALSE;
	}
	
	switch (toupper(*param[0]))
	{
		case 'B':
			change_mode(tab, FALSE, 'B');
			break;
		case 'M':
			change_mode(tab, FALSE, 'M');
			break;
		case 'E':
			change_mode(tab, FALSE, 'E');
			break;
		default:
			feed_error_to_terminal(tab, TRUE, "Usage: MODE (M|E|B)");
  		return FALSE;
			break;
	}

	return TRUE;
}

BC_COMMAND(command_connect)
{
	gchar *hostname;
	unsigned short port;

	if (paramnum < 1)
	{
		feed_error_to_terminal(tab, TRUE, "Usage: CONNECT botname or CONNECT host port [user [password]]");
		return FALSE;
	}
	if (paramnum < 2)
	{
		botrecord_t *record;

		if ((record = find_bot(param[0])) == NULL)
		{
			feed_error_to_terminal(tab, TRUE, "No such bot (%s)", param[0]);
			return FALSE;
		}
		hostname = g_strdup(record->hostname);
		port = record->port;
	}
	else
	{
		hostname = g_strdup(param[0]);
		if (!is_numeric(param[1]))
		{
      struct protoent *prot;
      struct servent *serv;
			
      if ((prot = getprotobyname("tcp")) == NULL)
      {
        feed_error_to_terminal(tab, TRUE, "Unable to get TCP protocol!");
        return FALSE;
      }
			
      if ((serv = getservbyname(param[1], prot->p_name)) == NULL)
      {
        feed_error_to_terminal(tab, TRUE, "Unknown port %s", param[1]);
        return FALSE;
      }
			
      port = ntohs(serv->s_port);
		}
		else
			port = atoi(param[1]);
	}
	
	if ((hostname == NULL) || (port == 0))
		return FALSE;
	
	return connect_to_host(tab, hostname, port);
}

BC_COMMAND(command_close)
{
	/* TODO: If the tab is connected, we should ask if the user really wants to
	 * close the tab. If so, we HAVE TO close its connection. Now let's just close
   * the connection.	*/
	if (tab->connected)
	{
		close(tab->socket);
		gdk_input_remove(tab->socket_tag);
	}
	remove_tab(tab);
	
	return TRUE;
}

/*
 * process_line()
 * Processes the full command line.
 */
void
process_line(tab_data_t *tab, gchar *text)
{
	gchar this_mode;
	
	this_mode = tab->current_mode;
	switch (*text)
	{
		case '/':
			this_mode = 'B';
			break;
		case '.':
			this_mode = 'E';
			break;
		case '@':
			this_mode = 'M';
			break;
	}

	switch (this_mode)
	{
		case 'E':
			dprintf(tab->socket, ".%s\n", (*text == '.') ? (text + 1) : text);
			break;
		case 'B':
			{
				char **words;
				int num, i;
				gboolean found = FALSE;
				
				if ((num = wrap_string(text, " \t", &words)) == -1)
					return;
				
				if (*words[0] == '/')
					words[0] = (words[0] + 1);
				
				for (i = 0; i < num_commands; i++)
				{
					if (str_eq(command[i].command, words[0], FALSE))
					{
						found = TRUE;
						if (!(command[i].func)(tab, num - 1, words + 1))
							if (config_data.debug_mode)
								feed_error_to_terminal(tab, TRUE, "Error while processing command.");
					}
				}
				if (!found)
					feed_error_to_terminal(tab, TRUE, "Unknown command.");
			}
			break;
		case 'M':
		default:
			{
				char *converted_text;
				
				if ((converted_text = iconvert(text, FALSE)))
				{
					dprintf(tab->socket, "%s%s\n", (*text == '.') ? " " : "", (*text == '@') ? (converted_text + 1) : converted_text);
					g_free(converted_text);
				}
				break;
			}
	}
}

/*
 * history_init()
 * Initialize the command line history
 */
gboolean
history_init(gboolean first_time)
{
	int i;
	
	history_position = -1;

	if (first_time)
	{
		if ((command_history = g_malloc(config_data.commandline_history_length * sizeof(char *))) == NULL)
			return FALSE;
		for (i = 0; i < config_data.commandline_history_length; i++)
			command_history[i] = NULL;
		return TRUE;
	}

	if (command_history == NULL)
	{
		if ((command_history = g_malloc(config_data.commandline_history_length * sizeof(char *))) == NULL)
		{
			return FALSE;
		}
	}
	for (i = 0; i < config_data.commandline_history_length; i++)
	{
		if (command_history[i])
		{
			g_free(command_history[i]);
			command_history[i] = NULL;
		}
	}
	return TRUE;
}

/*
 * history_add()
 * Add a command to the end of history
 */
void
history_add(gchar *command)
{
	int i;
	
	history_position = -1;

	if (command_history[0] == NULL)
	{
		command_history[0] = g_strdup(command);
		return;
	}
	
	if (command_history[config_data.commandline_history_length - 1] != NULL)
	{
		for (i = 1; i < config_data.commandline_history_length; i++)
		{
			command_history[i - 1] = command_history[i];
		}
		command_history[config_data.commandline_history_length - 1] = g_strdup(command);
		return;
	}
	
	for (i = 0; i < config_data.commandline_history_length; i++)
	{
		if (command_history[i] == NULL)
		{
			command_history[i] = g_strdup(command);
			return;
		}
	}
}

/*
 * history_get_next()
 * Get the next command from the history
 */
gchar *
history_get_next(void)
{
	/* If the first item is NULL, we have an empty history */
	if (command_history[0] == NULL)
		return NULL;
	
	if ((history_position == -1) || (history_position == config_data.commandline_history_length) || (command_history[history_position + 1] == NULL))
	{
		history_position = 0;
		return command_history[0];
	}
	
	history_position++;
	return command_history[history_position];
}

/*
 * history_get_previous()
 * Get the previous command from the history
 */
gchar *
history_get_previous(void)
{
	int i;
	
	/* It the first item is NULL, we have an empty history */
	if (command_history[0] == NULL)
		return NULL;
	
	if ((history_position == -1) || (history_position == 0))
	{
		for (i = 0; i < config_data.commandline_history_length; i++)
		{
			if (command_history[i + 1] == NULL)
			{
				history_position = i;
				return command_history[i];
			}
		}
		history_position = config_data.commandline_history_length - 1;
		return command_history[history_position];
	}
	
	history_position--;
	return command_history[history_position];
}

/*
 * history_free()
 * Free the command history
 */
void
history_free(void)
{
	int i;

	if (command_history)
	{
		for (i = 0; i < config_data.commandline_history_length; i++)
		{
			if (command_history[i])
			{
				g_free(command_history[i]);
			}
		}
		g_free(command_history);
	}
}
