/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
/***************************************************************************
 *            botlist.c
 *
 *  Tue Oct  5 16:36:47 2004
 *  Copyright  2004  User
 *  Email
 ***************************************************************************/
/***************************************************************************
 * Functions which deal with the bot list
 ***************************************************************************/

#include "botcommander.h"

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <string.h>

#include "typedefs.h"
#include "functions.h"
#include "variables.h"

botrecord_t *bot_list;
static GtkListStore *botlist_store;
static GtkWidget *treeview_bots;

/*
 * botlist_clear()
 * Clears the internal bot list. If init is TRUE, it just set bot_list to NULL.
 */
void
botlist_clear(gboolean init)
{
	if (init)
	{
		bot_list = NULL;
	}
	else
	{
		if (bot_list != NULL)
		{
			botrecord_t *temp;
			
			do
			{
				temp = bot_list->next;
				if (bot_list->name)
					g_free(bot_list->name);
				if (bot_list->hostname)
					g_free(bot_list->hostname);
				if (bot_list->user)
					g_free(bot_list->user);
				g_free(bot_list);
				bot_list = temp;
			}
			while (temp != NULL);
		}
	}
}

/*
 * add_bot()
 * Adds a bot to the end of the internal bot list
 */
gboolean
add_bot(char *botname)
{
	botrecord_t *temp;
	
	if (bot_list == NULL)
	{
		if ((bot_list = g_malloc(sizeof(botrecord_t))) == NULL)
			return FALSE;
		
		temp = bot_list;
	}
	else
	{
		temp = bot_list;
		while (temp->next)
			temp = temp->next;
		
		if ((temp->next = g_malloc(sizeof(botrecord_t))) == NULL)
			return FALSE;
		
		temp = temp->next;
	}
	/* At this point temp is a new record, and not NULL */
	
	memset(temp, 0, sizeof(botrecord_t));
	temp->next = NULL;
	temp->name = g_strdup(botname);
	temp->hostname = NULL;
	temp->port = 0;
	temp->user = NULL;
	return TRUE;
}

/*
 * find_bot()
 * Finds a bot in the botlist by its name. Case sensitive match!
 * Returns a pointer to the record in the bot list. Return value should NEVER
 * BE free()'D, IT WOULD MESS UP THE WHOLE LIST!
 */
botrecord_t *
find_bot(gchar *name)
{
	botrecord_t *temp = bot_list;
	
	while (temp)
	{
		if (str_eq(temp->name, name, 1))
		{
			return temp;
		}
		temp = temp->next;
	}
	
	return NULL;
}

/*
 * delete_bot()
 * Deletes a bot from the internal bot list.
 */
void
delete_bot(botrecord_t *record)
{
	botrecord_t *temp;
	
	if ((temp = bot_list) == NULL)
		return;
	
	if (bot_list == record)
	{
		bot_list = bot_list->next;
		if (record->name != NULL)
			g_free(record->name);
		if (record->hostname != NULL)
			g_free(record->hostname);
		if (record->user != NULL)
			g_free(record->user);
		g_free(record);
	}
	else
	{
		botrecord_t *prev = NULL;
		
		temp = bot_list;
		while (temp)
		{
			if (temp == record)
			{
				if (prev)
					prev->next = record->next;
				temp = temp->next;
				if (record->name != NULL)
					g_free(record->name);
				if (record->hostname != NULL)
					g_free(record->hostname);
				g_free(record);
				record = NULL;
			}
			else
			{
				prev = temp;
				temp = temp->next;
			}
		}
	}
}

/*
 * set_bot_hostname()
 * Sets the given bot's hostname
 */
void
set_bot_hostname(botrecord_t *record, gchar *newhost)
{
	if (record)
	{
		if (record->hostname)
			g_free(record->hostname);
		record->hostname = g_strdup(newhost);
	}
}

/*
 * set_bot_portnum()
 * Sets the given bot's port number
 */
void
set_bot_portnum(botrecord_t *record, gint portnum)
{
	if (record)
		record->port = portnum;
}

/*
 * set_bot_username()
 * Sets the given bot's username
 */
void
set_bot_username(botrecord_t *record, gchar *username)
{
	if (record)
	{
		if (record->user)
			g_free(record->user);
		record->user = g_strdup(username);
	}
}

#if (CONFIG_BACKEND == CB_GCONF2)
/*
 * add_bot_from_gslist()
 * Adds a bot to the list from the given GSList data. Called from
 * botlist_read_from_config() only, you shouldn't call from anywhere else.
 */
void
add_bot_from_gslist(gpointer data, gpointer user_data)
{
	char *key;
	GError *error;
	gchar *rethost = NULL, *retuser = NULL;
	gint retport = 0;
	botrecord_t *record;
	
	key = g_malloc(strlen((char *)data) + 6);
	memset(key, 0, strlen((char *)data) + 6);
	
	sprintf(key, "%s/host", (char *)data);
	error = NULL;
	rethost = gconf_client_get_string(gconf_client, key, &error);
	
	sprintf(key, "%s/port", (char *)data);
	error = NULL;
	retport = gconf_client_get_int(gconf_client, key, &error);

	sprintf(key, "%s/user", (char *)data);
	error = NULL;
	retuser = gconf_client_get_string(gconf_client, key, &error);
	
	if (config_data.debug_mode)
		printf("Adding bot...\nName: %s\nHost: %s\nPort: %d\nUser: %s\n\n", (char *)data + strlen(GCONF_CONFIG_ROOT) + 5, rethost, retport, retuser);
	add_bot((char *)data + strlen(GCONF_CONFIG_ROOT) + 5);
	if ((record = find_bot((char *)data + strlen(GCONF_CONFIG_ROOT) + 5)))
	{
		if (rethost)
			set_bot_hostname(record, rethost);
		if (retport)
			set_bot_portnum(record, retport);
		if (retuser)
			set_bot_username(record, retuser);
	}
	g_free(key);
}
#endif

/*
 * botlist_read_from_config()
 * Read botlist from the actual config backend
 */
void
botlist_read_from_config(void)
{
	botrecord_t *temp;
	botlist_clear(FALSE);
#if (CONFIG_BACKEND == CB_GCONF2)
	GSList *all_botnames;
	GError *error;
#endif
#if (CONFIG_BACKEND != CB_GCONF2)
# warning "Botlist reading from other backend than GConf2 is not supported yet."
#endif
	
#if (CONFIG_BACKEND == CB_GCONF2)
	all_botnames = gconf_client_all_dirs(gconf_client, GCONF_CONFIG_ROOT "bots", &error);
	g_slist_foreach(all_botnames, add_bot_from_gslist, NULL);
#endif
	if (config_data.debug_mode)
	{
		printf("Listing bots:\n");
		temp = bot_list;
		while (temp)
		{
			printf(":%s:%s:%d:%s:\n", temp->name, temp->hostname, temp->port, temp->user);
			temp = temp->next;
		}
	}
}

/*
 * botlist_write_to_config()
 * Writes botlist to the actual config backend
 */
void
botlist_write_to_config(void)
{
}

/*
 * botlist_response()
 * Called when a response signal occurs
 */
static void
botlist_response(GtkDialog *botlist, int id, gpointer data)
{
	gtk_signal_emit_by_name(GTK_OBJECT(static_data.botlist_window), "destroy", NULL);
	static_data.botlist_window = NULL;
}

/*
 * botlist_row_activated()
 * Called when a row of the treeview double-clicked
 */
void
botlist_row_activated(GtkTreeView *treeview, GtkTreePath *treepath, GtkTreeViewColumn *column, gpointer user_data)
{
	printf("Row activated\n");
}

/*
 * botlist_connect()
 * Connects to the selected bot
 */
static void
botlist_connect(GtkWidget *widget, gpointer data)
{
  GtkTreeSelection *sel;
  GtkTreeIter iter;
  GtkTreeModel *model = GTK_TREE_MODEL(botlist_store);
  int num_sel = 0;
  char *botname = NULL;

  sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview_bots));
  if (gtk_tree_model_get_iter_first(model, &iter))
  {
    do
    {
      if (gtk_tree_selection_iter_is_selected(sel, &iter))
        num_sel++;
    }
    while (gtk_tree_model_iter_next(model, &iter));
  }

  if (num_sel == 1)
  {
    gtk_tree_model_get_iter_first(model, &iter);
    do
    {
      if (gtk_tree_selection_iter_is_selected(sel, &iter))
        gtk_tree_model_get(model, &iter, 0, &botname, -1);
    }
    while (gtk_tree_model_iter_next(model, &iter));
  }

  if (botname)
  {
    char *cmdline;
    tab_data_t *tab;

    if ((tab = get_active_tab()) == NULL)
      return;

    cmdline = g_malloc(10 + strlen(botname));
    sprintf(cmdline, "/CONNECT %s", botname);
    process_line(tab, cmdline);
    g_free(cmdline);
    g_free(botname);
    gtk_signal_emit_by_name(GTK_OBJECT(static_data.botlist_window), "destroy");
  }
}

/*
 * open_botlist()
 * Opens the botlist window
 */
void
open_botlist(void)
{
	GtkWidget *vbox,
	          *table,
            *button_connect,
	          *button_add,
	          *button_edit,
	          *button_remove;
	GtkTreeModel *model;
	GtkCellRenderer *renderer;
  botrecord_t *temp;

	if (static_data.botlist_window != NULL)
	{
		gtk_window_present(GTK_WINDOW(static_data.botlist_window));
		return;
	}
	
	static_data.botlist_window = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(static_data.botlist_window), "BotCommander's bot list");
	gtk_window_set_policy(GTK_WINDOW(static_data.botlist_window), FALSE, FALSE, FALSE);
	gtk_window_set_modal(GTK_WINDOW(static_data.botlist_window), TRUE);
	gtk_signal_connect(GTK_OBJECT(static_data.botlist_window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &(static_data.botlist_window));
	gtk_signal_connect(GTK_OBJECT(static_data.botlist_window), "response", GTK_SIGNAL_FUNC(botlist_response), NULL);
	
	vbox = GTK_DIALOG(static_data.botlist_window)->vbox;
	
	table = gtk_table_new(3, 2, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 0);
	gtk_widget_show(table);
	
	botlist_store = gtk_list_store_new(4, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT, G_TYPE_STRING);
	model = GTK_TREE_MODEL(botlist_store);
	
	treeview_bots = gtk_tree_view_new_with_model(model);
	g_object_unref(model);
	gtk_table_attach_defaults(GTK_TABLE(table), treeview_bots, 0, 1, 0, 4);
	gtk_widget_show(treeview_bots);
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview_bots), TRUE);
	gtk_signal_connect(GTK_OBJECT(treeview_bots), "row-activated", GTK_SIGNAL_FUNC(botlist_row_activated), NULL);
	
	renderer = gtk_cell_renderer_text_new();
	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(treeview_bots), 0, "Bot name", renderer, "text", 0, NULL);
	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(treeview_bots), 1, "Host", renderer, "text", 1, NULL);
	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(treeview_bots), 2, "Port", renderer, "text", 2, NULL);
	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(treeview_bots), 3, "Username", renderer, "text", 3, NULL);
  /* TODO: What signal do I have to catch if I want double-click on a list item to "press" the connect button? */
	
  button_connect = gtk_button_new_with_label("Connect");
  gtk_table_attach_defaults(GTK_TABLE(table), button_connect, 1, 2, 0, 1);
  gtk_signal_connect(GTK_OBJECT(button_connect), "clicked", GTK_SIGNAL_FUNC(botlist_connect), NULL);
  gtk_widget_show(button_connect);

	button_add = gtk_button_new_with_label("Add...");
	gtk_table_attach_defaults(GTK_TABLE(table), button_add, 1, 2, 1, 2);
	gtk_widget_show(button_add);
	
	button_edit = gtk_button_new_with_label("Modify...");
	gtk_table_attach_defaults(GTK_TABLE(table), button_edit, 1, 2, 2, 3);
	gtk_widget_show(button_edit);
	
	button_remove = gtk_button_new_with_label("Remove");
	gtk_table_attach_defaults(GTK_TABLE(table), button_remove, 1, 2, 3, 4);
	gtk_widget_show(button_remove);
	
	gtk_dialog_set_has_separator(GTK_DIALOG(static_data.botlist_window), TRUE);
	gtk_dialog_add_buttons(GTK_DIALOG(static_data.botlist_window), GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
	
	gtk_widget_show(static_data.botlist_window);
	
  temp = bot_list;
  while (temp)
  {
    GtkTreeIter iter;
    
    gtk_list_store_append(botlist_store, &iter);
    gtk_list_store_set(botlist_store, &iter, 0, (temp->name == NULL) ? "" : temp->name, 1, (temp->hostname == NULL) ? "" : temp->hostname, 2, temp->port, 3, (temp->user == NULL) ? "" : temp->user, -1);

    temp = temp->next;
  }
}
