<?php
# Job editor.
#
# Copyright (C) 1999, 2000 The SourceForge Crew
# Copyright (C) 2000-2006 Mathieu Roy
# Copyright (C) 2014, 2016, 2017 Assaf Gordon
# Copyright (C) 2001-2011, 2013, 2017 Sylvain Beucler
# Copyright (C) 2013, 2014, 2017-2024 Ineiev
#
# This file is part of Savane.
#
# Code written before 2008-03-30 (commit 8b757b2565ff) is distributed
# under the terms of the GNU General Public license version 3 or (at your
# option) any later version; further contributions are covered by
# the GNU Affero General Public license version 3 or (at your option)
# any later version.  The license notices for the AGPL and the GPL follow.
#
# Savane is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# Savane is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# Savane is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as
# published by the Free Software Foundation, either version 3 of the
# License, or (at your option) any later version.
#
# Savane is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

require_once ('../include/init.php');
require_once ('../include/sane.php');
require_once ('../include/form.php');
require_once ('../include/people/general.php');

extract (sane_import ('request', ['digits' => 'job_id']));
$submits = ['add_job', 'update_job', 'add_to_job_inventory',
  'update_job_inventory', 'delete_from_job_inventory',
];
extract (sane_import ('post',
  [
    'true' => $submits,
    'digits' =>
      [
        'status_id', 'category_id', 'job_inventory_id', 'skill_id',
        'skill_level_id', 'skill_year_id',
      ],
    'specialchars' => 'title',
    'pass' => 'description',
  ]
));
form_check ($submits);

if (!$group_id)
  exit_no_group ();

if (!user_ismember ($group_id, 'A'))
  exit_permission_denied ();

if ($add_job)
  {
    # Create a new job.
    if (!$title || !$description || $category_id == 100)
      exit_error (_("error - missing info"), _("Fill in all required fields"));
    $result = db_autoexecute ('people_job',
      [ 'group_id' => $group_id, 'created_by' => user_getid (),
        'title' => $title, 'description' => $description, 'date' => time (),
        'status_id' => 1, 'category_id' => $category_id,
      ], DB_AUTOQUERY_INSERT
    );
    if (!$result || db_affected_rows ($result) < 1)
      {
        fb (_("JOB insert FAILED"));
        print db_error ();
      }
    else
      {
        $job_id = db_insertid ($result);
        fb (_("JOB inserted successfully"));
      }
  }
elseif ($update_job)
  {
    # Update the job's description, status, etc.
    if (!$title || !$description || $category_id == 100 || $status_id == 100
        || !$job_id
    )
      exit_error (_("error - missing info"), _("Fill in all required fields"));
    $result = db_autoexecute ('people_job',
      [
        'title' => $title, 'description' => $description,
        'status_id' => $status_id, 'category_id' => $category_id,
      ], DB_AUTOQUERY_UPDATE,
      "job_id = ? AND group_id = ?", [$job_id, $group_id]
    );
    if (!$result || db_affected_rows ($result) < 1)
      {
        fb (_("JOB update FAILED"));
        print db_error ();
      }
    else
      fb (_("JOB updated successfully"));
  }
elseif ($add_to_job_inventory)
  {
    # Add item to job inventory.
    if ($skill_id == 100 || $skill_level_id == 100 || $skill_year_id == 100
        || !$job_id)
      exit_error (_("error - missing info"), _("Fill in all required fields"));

    if (people_verify_job_group ($job_id, $group_id))
      {
        people_add_to_job_inventory (
          $job_id, $skill_id, $skill_level_id, $skill_year_id
        );
        fb (_("JOB updated successfully"));
      }
    else
      fb (_("JOB update failed - wrong project_id"));
  }
elseif ($update_job_inventory)
  {
    # Change Skill level, experience etc.
    if ($skill_level_id == 100 || $skill_year_id == 100  || !$job_id
        || !$job_inventory_id)
      exit_error (_("error - missing info"), _("Fill in all required fields"));

    if (people_verify_job_group ($job_id, $group_id))
      {
        $result = db_autoexecute ('people_job_inventory',
          [
            'skill_level_id' => $skill_level_id,
            'skill_year_id' => $skill_year_id,
          ], DB_AUTOQUERY_UPDATE,
          "job_id = ? AND job_inventory_id = ?", [$job_id, $job_inventory_id]
        );
        if (!$result || db_affected_rows($result) < 1)
          {
            fb(_("JOB skill update FAILED"));
            print db_error();
          }
        else
          fb(_("JOB skill updated successfully"));
      }
    else
      fb (_("JOB skill update failed - wrong project_id"));
  }
elseif ($delete_from_job_inventory)
  {
    # Remove this skill from this job.
    if (!$job_id)
      exit_error (_("error - missing info"), _("Fill in all required fields"));
    if (people_verify_job_group ($job_id, $group_id))
      {
        $result = db_execute("
          DELETE FROM people_job_inventory
          WHERE job_id = ? AND job_inventory_id = ?",
          [$job_id, $job_inventory_id]
        );
        if (!$result || db_affected_rows ($result) < 1)
          {
            fb (_("JOB skill delete FAILED"));
            print db_error ();
          }
        else
          fb (_("JOB skill deleted successfully"));
      }
    else
      fb (_("JOB skill delete failed - wrong project_id"));
  }

# Fill in the info to create a job.  Only if we have a job id specified;
# if not, it means that we are looking for a project to edit.
if ($job_id)
  {
    site_project_header (
      [ 'title' => _("Edit a job for your project"),
        'group' => $group_id, 'context' => 'ahome']
    );
    # For security, include group_id.
    $result = db_execute (
      "SELECT * FROM people_job WHERE job_id = ? AND group_id = ?",
      [$job_id, $group_id]
    );
    if (db_numrows ($result) < 1)
      {
        print db_error ();
        fb (_("POSTING fetch FAILED"));
      }
    else
      {
        $description = utils_specialchars (
          db_result ($result, 0, 'description')
        );
        utils_get_content ("people/editjob");
        print form_tag ()
          . form_hidden (['group_id' => $group_id, 'job_id' => $job_id])
          . "<strong>" . _("Category:") . "</strong><br />\n"
          . people_job_category_box (
              'category_id', db_result ($result, 0, 'category_id')
            )
          . "\n<p><strong>" . _("Status") . ":</strong><br />\n"
          . people_job_status_box (
              'status_id', db_result ($result, 0, 'status_id')
            )
          . "</p>\n<p><strong>"
          . html_label ('title', _("Short Description:"))
          . "</strong><br />\n"
          . "<input type='text' id='title' name='title' value=\""
          . db_result ($result, 0, 'title')
          . "\" size='40' maxlength='60' /></p>\n<p><strong>"
          . html_label ('description', _("Long Description:"))
          . "</strong><br />\n"
          . "<textarea name='description' id='description' rows='10' "
          . "cols='60' wrap='soft'>$description</textarea>\n</p>\n"
          . '<p>'
          . form_submit (_("Update Descriptions"), "update_job")
          . "\n</form>\n";
        print '<p>' . people_edit_job_inventory ($job_id, $group_id)
          . "</p>\n<p>[<a href='/people'>" . _("Back to jobs listing")
          . "</a>]</p>\n";
      }
  }
else # ! $job_id
  {
    site_project_header (
      [ 'title' => _("Looking for a job to Edit"),
        'group' => $group_id,'context' => 'ahome']
    );
    print '<p>'
      . _("Here is a list of positions available for this project, choose "
          . "the\none you want to modify.")
      . "</p>\n";
    print people_show_project_jobs ($group_id, 1);
  }
site_project_footer ();
?>
