/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.security.authtoken.jwt;

import com.nimbusds.jose.Algorithm;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.JWSHeader;
import com.nimbusds.jose.JWSSigner;
import com.nimbusds.jose.KeyLengthException;
import com.nimbusds.jose.crypto.MACSigner;
import com.nimbusds.jose.jwk.JWK;
import com.nimbusds.jose.jwk.KeyUse;
import com.nimbusds.jose.jwk.OctetSequenceKey;
import com.nimbusds.jwt.JWTClaimsSet;
import com.nimbusds.jwt.SignedJWT;
import java.security.AccessController;
import java.security.PrivilegedAction;
import java.text.ParseException;
import java.util.Base64;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.function.LongSupplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.opensearch.OpenSearchException;
import org.opensearch.common.collect.Tuple;
import org.opensearch.common.settings.Settings;
import org.opensearch.security.authtoken.jwt.EncryptionDecryptionUtil;
import org.opensearch.security.authtoken.jwt.ExpiringBearerAuthToken;
import org.opensearch.security.util.AuthTokenUtils;

public class JwtVendor {
    private static final Logger logger = LogManager.getLogger(JwtVendor.class);
    private final JWK signingKey;
    private final JWSSigner signer;
    private final LongSupplier timeProvider;
    private final EncryptionDecryptionUtil encryptionDecryptionUtil;
    private static final Integer MAX_EXPIRY_SECONDS = 600;

    public JwtVendor(Settings settings, Optional<LongSupplier> timeProvider) {
        Tuple<JWK, JWSSigner> tuple = JwtVendor.createJwkFromSettings(settings);
        this.signingKey = (JWK)tuple.v1();
        this.signer = (JWSSigner)tuple.v2();
        if (AuthTokenUtils.isKeyNull(settings, "encryption_key").booleanValue()) {
            throw new IllegalArgumentException("encryption_key cannot be null");
        }
        this.encryptionDecryptionUtil = new EncryptionDecryptionUtil(settings.get("encryption_key"));
        this.timeProvider = timeProvider.orElse(System::currentTimeMillis);
    }

    static Tuple<JWK, JWSSigner> createJwkFromSettings(Settings settings) {
        OctetSequenceKey key;
        if (!AuthTokenUtils.isKeyNull(settings, "signing_key").booleanValue()) {
            String signingKey = settings.get("signing_key");
            key = new OctetSequenceKey.Builder(Base64.getDecoder().decode(signingKey)).algorithm((Algorithm)JWSAlgorithm.HS512).keyUse(KeyUse.SIGNATURE).build();
        } else {
            Settings jwkSettings = settings.getAsSettings("jwt").getAsSettings("key");
            if (jwkSettings.isEmpty()) {
                throw new OpenSearchException("Settings for signing key is missing. Please specify at least the option signing_key with a shared secret.", new Object[0]);
            }
            String signingKey = jwkSettings.get("k");
            key = new OctetSequenceKey.Builder(Base64.getDecoder().decode(signingKey)).algorithm((Algorithm)JWSAlgorithm.HS512).keyUse(KeyUse.SIGNATURE).build();
        }
        try {
            return new Tuple((Object)key, (Object)new MACSigner(key));
        }
        catch (KeyLengthException kle) {
            throw new OpenSearchException((Throwable)kle);
        }
    }

    public ExpiringBearerAuthToken createJwt(final String issuer, final String subject, final String audience, final long requestedExpirySeconds, final List<String> roles, final List<String> backendRoles, final boolean includeBackendRoles) {
        return AccessController.doPrivileged(new PrivilegedAction<ExpiringBearerAuthToken>(){

            @Override
            public ExpiringBearerAuthToken run() {
                try {
                    long currentTimeMs = JwtVendor.this.timeProvider.getAsLong();
                    Date now = new Date(currentTimeMs);
                    JWTClaimsSet.Builder claimsBuilder = new JWTClaimsSet.Builder();
                    claimsBuilder.issuer(issuer);
                    claimsBuilder.issueTime(now);
                    claimsBuilder.subject(subject);
                    claimsBuilder.audience(audience);
                    claimsBuilder.notBeforeTime(now);
                    long expirySeconds = Math.min(requestedExpirySeconds, (long)MAX_EXPIRY_SECONDS.intValue());
                    if (expirySeconds <= 0L) {
                        throw new IllegalArgumentException("The expiration time should be a positive integer");
                    }
                    Date expiryTime = new Date(currentTimeMs + expirySeconds * 1000L);
                    claimsBuilder.expirationTime(expiryTime);
                    if (roles == null) {
                        throw new IllegalArgumentException("Roles cannot be null");
                    }
                    String listOfRoles = String.join((CharSequence)",", roles);
                    claimsBuilder.claim("er", (Object)JwtVendor.this.encryptionDecryptionUtil.encrypt(listOfRoles));
                    if (includeBackendRoles && backendRoles != null) {
                        String listOfBackendRoles = String.join((CharSequence)",", backendRoles);
                        claimsBuilder.claim("br", (Object)listOfBackendRoles);
                    }
                    JWSHeader header = new JWSHeader.Builder(JWSAlgorithm.parse((String)JwtVendor.this.signingKey.getAlgorithm().getName())).build();
                    SignedJWT signedJwt = new SignedJWT(header, claimsBuilder.build());
                    signedJwt.sign(JwtVendor.this.signer);
                    if (logger.isDebugEnabled()) {
                        logger.debug("Created JWT: " + signedJwt.serialize() + "\n" + String.valueOf(signedJwt.getHeader().toJSONObject()) + "\n" + String.valueOf(signedJwt.getJWTClaimsSet().toJSONObject()));
                    }
                    return new ExpiringBearerAuthToken(signedJwt.serialize(), subject, expiryTime, expirySeconds);
                }
                catch (JOSEException | ParseException e) {
                    logger.error("Error while creating JWT token", e);
                    throw new OpenSearchException("Error while creating JWT token", e, new Object[0]);
                }
            }
        });
    }
}

