"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var stringUtils_1 = require("./utils/stringUtils");
var CommentChar_1 = require("./CommentChar");
var CodeBlockWriter = /** @class */ (function () {
    function CodeBlockWriter(opts) {
        if (opts === void 0) { opts = {}; }
        this._currentIndentation = 0;
        this._text = "";
        this._newLineOnNextWrite = false;
        /** @internal */
        this._currentCommentChar = undefined;
        this._stringCharStack = [];
        this._isInRegEx = false;
        this._isOnFirstLineOfBlock = true;
        this._newLine = opts.newLine || "\n";
        this._useTabs = opts.useTabs || false;
        this._indentNumberOfSpaces = opts.indentNumberOfSpaces || 4;
        this._indentationText = getIndentationText(this._useTabs, this._indentNumberOfSpaces);
        this._quoteChar = opts.useSingleQuote ? "'" : "\"";
    }
    /**
     * Gets the options.
     */
    CodeBlockWriter.prototype.getOptions = function () {
        return {
            indentNumberOfSpaces: this._indentNumberOfSpaces,
            newLine: this._newLine,
            useTabs: this._useTabs,
            useSingleQuote: this._quoteChar === "'"
        };
    };
    CodeBlockWriter.prototype.queueIndentationLevel = function (countOrText) {
        this._queuedIndentation = this._getIndentationLevelFromArg(countOrText);
        return this;
    };
    CodeBlockWriter.prototype.setIndentationLevel = function (countOrText) {
        this._currentIndentation = this._getIndentationLevelFromArg(countOrText);
        return this;
    };
    /**
     * Gets the current indentation level.
     */
    CodeBlockWriter.prototype.getIndentationLevel = function () {
        return this._currentIndentation;
    };
    /**
     * Writes a block using braces.
     * @param block - Write using the writer within this block.
     */
    CodeBlockWriter.prototype.block = function (block) {
        this._newLineIfNewLineOnNextWrite();
        if (this.getLength() > 0 && !this.isLastNewLine())
            this.spaceIfLastNot();
        this.inlineBlock(block);
        this._newLineOnNextWrite = true;
        return this;
    };
    /**
     * Writes an inline block with braces.
     * @param block - Write using the writer within this block.
     */
    CodeBlockWriter.prototype.inlineBlock = function (block) {
        this._newLineIfNewLineOnNextWrite();
        this.write("{");
        this._indentBlockInternal(block);
        this.newLineIfLastNot().write("}");
        return this;
    };
    /**
     * Indents a block of code.
     * @param block - Block to indent.
     */
    CodeBlockWriter.prototype.indentBlock = function (block) {
        this._indentBlockInternal(block);
        if (!this.isLastNewLine())
            this._newLineOnNextWrite = true;
        return this;
    };
    CodeBlockWriter.prototype._indentBlockInternal = function (block) {
        this._currentIndentation++;
        if (this.getLastChar() != null)
            this.newLineIfLastNot();
        this._isOnFirstLineOfBlock = true;
        if (block != null)
            block();
        this._isOnFirstLineOfBlock = false;
        this._currentIndentation = Math.max(0, this._currentIndentation - 1);
    };
    CodeBlockWriter.prototype.conditionalWriteLine = function (condition, strOrFunc) {
        if (condition)
            this.writeLine(stringUtils_1.getStringFromStrOrFunc(strOrFunc));
        return this;
    };
    /**
     * Writes a line of text.
     * @param text - String to write.
     */
    CodeBlockWriter.prototype.writeLine = function (text) {
        this._newLineIfNewLineOnNextWrite();
        if (this._text.length > 0)
            this.newLineIfLastNot();
        this._writeIndentingNewLines(text);
        this.newLine();
        return this;
    };
    /**
     * Writes a newline if the last line was not a newline.
     */
    CodeBlockWriter.prototype.newLineIfLastNot = function () {
        this._newLineIfNewLineOnNextWrite();
        if (!this.isLastNewLine())
            this.newLine();
        return this;
    };
    /**
     * Writes a blank line if the last written text was not a blank line.
     */
    CodeBlockWriter.prototype.blankLineIfLastNot = function () {
        if (!this.isLastBlankLine())
            this.blankLine();
        return this;
    };
    /**
     * Writes a blank line if the condition is true.
     * @param condition - Condition to evaluate.
     */
    CodeBlockWriter.prototype.conditionalBlankLine = function (condition) {
        if (condition)
            this.blankLine();
        return this;
    };
    /**
     * Writes a blank line.
     */
    CodeBlockWriter.prototype.blankLine = function () {
        return this.newLineIfLastNot().newLine();
    };
    /**
     * Indents the code one level for the current line.
     */
    CodeBlockWriter.prototype.indent = function () {
        this._newLineIfNewLineOnNextWrite();
        return this.write(this._indentationText);
    };
    /**
     * Writes a newline if the condition is true.
     * @param condition - Condition to evaluate.
     */
    CodeBlockWriter.prototype.conditionalNewLine = function (condition) {
        if (condition)
            this.newLine();
        return this;
    };
    /**
     * Writes a newline.
     */
    CodeBlockWriter.prototype.newLine = function () {
        this._newLineOnNextWrite = false;
        this._baseWriteNewline();
        return this;
    };
    CodeBlockWriter.prototype.quote = function (text) {
        this._newLineIfNewLineOnNextWrite();
        this._writeIndentingNewLines(text == null ? this._quoteChar : this._quoteChar + stringUtils_1.escapeForWithinString(text, this._quoteChar) + this._quoteChar);
        return this;
    };
    /**
     * Writes a space if the last character was not a space.
     */
    CodeBlockWriter.prototype.spaceIfLastNot = function () {
        this._newLineIfNewLineOnNextWrite();
        if (!this.isLastSpace())
            this._writeIndentingNewLines(" ");
        return this;
    };
    /**
     * Writes a space.
     * @param times - Number of times to write a space.
     */
    CodeBlockWriter.prototype.space = function (times) {
        if (times === void 0) { times = 1; }
        this._newLineIfNewLineOnNextWrite();
        this._writeIndentingNewLines(stringUtils_1.stringRepeat(" ", times));
        return this;
    };
    /**
     * Writes a tab if the last character was not a tab.
     */
    CodeBlockWriter.prototype.tabIfLastNot = function () {
        this._newLineIfNewLineOnNextWrite();
        if (!this.isLastTab())
            this._writeIndentingNewLines("\t");
        return this;
    };
    /**
     * Writes a tab.
     * @param times - Number of times to write a tab.
     */
    CodeBlockWriter.prototype.tab = function (times) {
        if (times === void 0) { times = 1; }
        this._newLineIfNewLineOnNextWrite();
        this._writeIndentingNewLines(stringUtils_1.stringRepeat("\t", times));
        return this;
    };
    CodeBlockWriter.prototype.conditionalWrite = function (condition, textOrFunc) {
        if (condition)
            this.write(stringUtils_1.getStringFromStrOrFunc(textOrFunc));
        return this;
    };
    /**
     * Writes the provided text.
     * @param text - Text to write.
     */
    CodeBlockWriter.prototype.write = function (text) {
        this._newLineIfNewLineOnNextWrite();
        this._writeIndentingNewLines(text);
        return this;
    };
    /**
     * Writes text to exit a comment if in a comment.
     */
    CodeBlockWriter.prototype.closeComment = function () {
        var commentChar = this._currentCommentChar;
        switch (commentChar) {
            case CommentChar_1.CommentChar.Line:
                this.newLine();
                break;
            case CommentChar_1.CommentChar.Star:
                if (!this.isLastNewLine())
                    this.spaceIfLastNot();
                this.write("*/");
                break;
            default:
                var assertUndefined = commentChar;
                break;
        }
        return this;
    };
    /**
     * Gets the length of the string in the writer.
     */
    CodeBlockWriter.prototype.getLength = function () {
        return this._text.length;
    };
    /**
     * Gets if the writer is currently in a comment.
     */
    CodeBlockWriter.prototype.isInComment = function () {
        return this._currentCommentChar !== undefined;
    };
    /**
     * Gets if the writer is currently at the start of the first line of the text, block, or indentation block.
     */
    CodeBlockWriter.prototype.isAtStartOfFirstLineOfBlock = function () {
        return this.isOnFirstLineOfBlock() && (this.isLastNewLine() || this.getLastChar() == null);
    };
    /**
     * Gets if the writer is currently on the first line of the text, block, or indentation block.
     */
    CodeBlockWriter.prototype.isOnFirstLineOfBlock = function () {
        return this._isOnFirstLineOfBlock;
    };
    /**
     * Gets if the writer is currently in a string.
     */
    CodeBlockWriter.prototype.isInString = function () {
        return this._stringCharStack.length > 0 && this._stringCharStack[this._stringCharStack.length - 1] !== "{";
    };
    /**
     * Gets if the last chars written were for a newline.
     */
    CodeBlockWriter.prototype.isLastNewLine = function () {
        return this._text.indexOf(this._newLine, this._text.length - this._newLine.length) !== -1 || this._text[this._text.length - 1] === "\n";
    };
    /**
     * Gets if the last chars written were for a blank line.
     */
    CodeBlockWriter.prototype.isLastBlankLine = function () {
        var foundCount = 0;
        for (var i = this._text.length - 1; i >= 0; i--) {
            var currentChar = this._text[i];
            if (currentChar === "\n") {
                foundCount++;
                if (foundCount === 2)
                    return true;
            }
            else if (currentChar !== "\r")
                return false;
        }
        return false;
    };
    /**
     * Gets if the last char written was a space.
     */
    CodeBlockWriter.prototype.isLastSpace = function () {
        return this.getLastChar() === " ";
    };
    /**
     * Gets if the last char written was a tab.
     */
    CodeBlockWriter.prototype.isLastTab = function () {
        return this.getLastChar() === "\t";
    };
    /**
     * Gets the last char written.
     */
    CodeBlockWriter.prototype.getLastChar = function () {
        if (this._text.length === 0)
            return undefined;
        return this._text[this._text.length - 1];
    };
    /**
     * Gets the writer's text.
     */
    CodeBlockWriter.prototype.toString = function () {
        return this._text;
    };
    CodeBlockWriter.prototype._writeIndentingNewLines = function (text) {
        var _this = this;
        text = text || "";
        if (text.length === 0) {
            writeIndividual.call(this, "");
            return;
        }
        var items = text.split(CodeBlockWriter._newLineRegEx);
        items.forEach(function (s, i) {
            if (i > 0)
                _this._baseWriteNewline();
            if (s.length === 0)
                return;
            writeIndividual.call(_this, s);
        });
        function writeIndividual(s) {
            if (!this.isInString()) {
                var isAtStartOfLine = this.isLastNewLine() || this._text.length === 0;
                if (isAtStartOfLine)
                    this._writeIndentation();
            }
            this._updateInternalState(s);
            this._text += s;
            this.dequeueQueuedIndentation();
        }
    };
    CodeBlockWriter.prototype._baseWriteNewline = function () {
        if (this._currentCommentChar === CommentChar_1.CommentChar.Line)
            this._currentCommentChar = undefined;
        this._text += this._newLine;
        this._isOnFirstLineOfBlock = false;
        this.dequeueQueuedIndentation();
    };
    CodeBlockWriter.prototype.dequeueQueuedIndentation = function () {
        if (this._queuedIndentation == null)
            return;
        this._currentIndentation = this._queuedIndentation;
        this._queuedIndentation = undefined;
    };
    CodeBlockWriter.prototype._updateInternalState = function (str) {
        for (var i = 0; i < str.length; i++) {
            var currentChar = str[i];
            var pastChar = i === 0 ? this._text[this._text.length - 1] : str[i - 1];
            var pastPastChar = i === 0 ? this._text[this._text.length - 2] : str[i - 2];
            // handle regex
            if (this._isInRegEx) {
                if (pastChar === "/" && pastPastChar !== "\\" || pastChar === "\n")
                    this._isInRegEx = false;
                else
                    continue;
            }
            else if (!this.isInString() && !this.isInComment() && isRegExStart(currentChar, pastChar, pastPastChar)) {
                this._isInRegEx = true;
                continue;
            }
            // handle comments
            if (this._currentCommentChar == null && pastChar === "/" && currentChar === "/")
                this._currentCommentChar = CommentChar_1.CommentChar.Line;
            else if (this._currentCommentChar == null && pastChar === "/" && currentChar === "*")
                this._currentCommentChar = CommentChar_1.CommentChar.Star;
            else if (this._currentCommentChar === CommentChar_1.CommentChar.Star && pastChar === "*" && currentChar === "/")
                this._currentCommentChar = undefined;
            if (this.isInComment())
                continue;
            // handle strings
            var lastStringCharOnStack = this._stringCharStack.length === 0 ? undefined : this._stringCharStack[this._stringCharStack.length - 1];
            if (pastChar !== "\\" && (currentChar === "\"" || currentChar === "'" || currentChar === "`")) {
                if (lastStringCharOnStack === currentChar)
                    this._stringCharStack.pop();
                else if (lastStringCharOnStack === "{" || lastStringCharOnStack === undefined)
                    this._stringCharStack.push(currentChar);
            }
            else if (pastPastChar !== "\\" && pastChar === "$" && currentChar === "{" && lastStringCharOnStack === "`")
                this._stringCharStack.push(currentChar);
            else if (currentChar === "}" && lastStringCharOnStack === "{")
                this._stringCharStack.pop();
        }
        function isRegExStart(currentChar, pastChar, pastPastChar) {
            return pastChar === "/"
                && currentChar !== "/"
                && currentChar !== "*"
                && pastPastChar !== "*"
                && pastPastChar !== "/";
        }
    };
    CodeBlockWriter.prototype._writeIndentation = function () {
        var flooredIndentation = Math.floor(this._currentIndentation);
        for (var i = 0; i < flooredIndentation; i++)
            this._text += this._indentationText;
        var overflow = this._currentIndentation - flooredIndentation;
        if (this._useTabs) {
            if (overflow > 0.5)
                this._text += this._indentationText;
        }
        else {
            var portion = Math.round(this._indentationText.length * overflow);
            for (var i = 0; i < portion; i++)
                this._text += this._indentationText[i];
        }
    };
    CodeBlockWriter.prototype._newLineIfNewLineOnNextWrite = function () {
        if (!this._newLineOnNextWrite)
            return;
        this._newLineOnNextWrite = false;
        this.newLine();
    };
    CodeBlockWriter.prototype._getIndentationLevelFromArg = function (countOrText) {
        if (typeof countOrText === "number") {
            if (countOrText < 0)
                throw new Error("Passed in indentation level should be greater than or equal to 0.");
            return countOrText;
        }
        else if (typeof countOrText === "string") {
            if (!CodeBlockWriter._spacesOrTabsRegEx.test(countOrText))
                throw new Error("Provided string must be empty or only contain spaces or tabs.");
            var _a = getSpacesAndTabsCount(countOrText), spacesCount = _a.spacesCount, tabsCount = _a.tabsCount;
            return tabsCount + spacesCount / this._indentNumberOfSpaces;
        }
        else
            throw new Error("Argument provided must be a string or number.");
    };
    CodeBlockWriter._newLineRegEx = /\r?\n/;
    CodeBlockWriter._spacesOrTabsRegEx = /^[ \t]*$/;
    return CodeBlockWriter;
}());
exports.default = CodeBlockWriter;
function getIndentationText(useTabs, numberSpaces) {
    if (useTabs)
        return "\t";
    return Array(numberSpaces + 1).join(" ");
}
function getSpacesAndTabsCount(str) {
    var spacesCount = 0;
    var tabsCount = 0;
    for (var _i = 0, str_1 = str; _i < str_1.length; _i++) {
        var char = str_1[_i];
        if (char === "\t")
            tabsCount++;
        else if (char === " ")
            spacesCount++;
    }
    return { spacesCount: spacesCount, tabsCount: tabsCount };
}
//# sourceMappingURL=code-block-writer.js.map