/**
 * A collection of runtime utility functions
 *
 * @remarks
 * This package should be marked as a dependency for any package that publishes the output of {@link @messageformat/core#compileModule},
 * as it may be included in its ES module source output as a dependency.
 *
 * For applications that bundle their output using e.g. Webpack this is not necessary.
 *
 * The `Messages` accessor class is a completely optional addition.
 * See also {@link @messageformat/react# | @messageformat/react} for a React-specific solution.
 *
 * @packageDocumentation
 */
/**
 * A message function, as generated by {@link @messageformat/core#MessageFormat.compile}
 *
 * @public
 */
export declare type MessageFunction = (param?: Record<string, unknown>) => string | unknown[];
/**
 * Hierarchical message object
 *
 * @public
 */
export interface MessageData {
    [key: string]: MessageData | MessageFunction | string;
}
/**
 * Accessor class for compiled message functions generated by
 * {@link @messageformat/core#compileModule}
 *
 * @public
 * @remarks
 * ```js
 * import Messages from '@messageformat/runtime/messages'
 * ```
 *
 * @example
 * ```js
 * // build.js
 * import { writeFileSync } from 'fs';
 * import MessageFormat from '@messageformat/core';
 * import compileModule from '@messageformat/core/compile-module'
 *
 * const mf = new MessageFormat(['en', 'fi']);
 * const msgSet = {
 *   en: {
 *     a: 'A {TYPE} example.',
 *     b: 'This has {COUNT, plural, one{one user} other{# users}}.',
 *     c: {
 *       d: 'We have {P, number, percent} code coverage.'
 *     }
 *   },
 *   fi: {
 *     b: 'Tällä on {COUNT, plural, one{yksi käyttäjä} other{# käyttäjää}}.',
 *     e: 'Minä puhun vain suomea.'
 *   }
 * };
 * writeFileSync('messages.js', compileModule(mf, msgSet));
 * ```
 *
 * ```js
 * // runtime.js
 * import Messages from '@messageformat/runtime/messages';
 * import msgData from './messages';
 *
 * const messages = new Messages(msgData, 'en');
 *
 * messages.hasMessage('a')                // true
 * messages.hasObject('c')                 // true
 * messages.get('b', { COUNT: 3 })         // 'This has 3 users.'
 * messages.get(['c', 'd'], { P: 0.314 })  // 'We have 31% code coverage.'
 *
 * messages.get('e')                       // 'e'
 * messages.setFallback('en', ['foo', 'fi'])
 * messages.get('e')                       // 'Minä puhun vain suomea.'
 *
 * messages.locale = 'fi'
 * messages.hasMessage('a')                // false
 * messages.hasMessage('a', 'en')          // true
 * messages.hasMessage('a', null, true)    // true
 * messages.hasObject('c')                 // false
 * messages.get('b', { COUNT: 3 })         // 'Tällä on 3 käyttäjää.'
 * messages.get('c').d({ P: 0.628 })       // 'We have 63% code coverage.'
 * ```
 */
export default class Messages {
    /** @internal */
    _data: {
        [key: string]: MessageData;
    };
    /** @internal */
    _fallback: {
        [key: string]: string[] | null;
    };
    /** @internal */
    _defaultLocale: string | null;
    /** @internal */
    _locale: string | null;
    /**
     * @param msgData - A map of locale codes to their function objects
     * @param defaultLocale - If not defined, default and initial locale is the first key of `msgData`
     */
    constructor(msgData: {
        [key: string]: MessageData;
    }, defaultLocale?: string);
    /** Read-only list of available locales */
    get availableLocales(): string[];
    /**
     * Current locale
     *
     * @remarks
     * One of {@link Messages.availableLocales} or `null`.
     * Partial matches of language tags are supported, so e.g. with an `en` locale defined, it will be selected by `messages.locale = 'en-US'` and vice versa.
     */
    get locale(): string | null;
    set locale(locale: string | null);
    /**
     * Default fallback locale
     *
     * @remarks
     * One of {@link Messages.availableLocales} or `null`.
     * Partial matches of language tags are supported, so e.g. with an `en` locale defined, it will be selected by `messages.defaultLocale = 'en-US'` and vice versa.
     */
    get defaultLocale(): string | null;
    set defaultLocale(locale: string | null);
    /**
     * Add new messages to the accessor; useful if loading data dynamically
     *
     * @remarks
     * The locale code `lc` should be an exact match for the locale being updated, or empty to default to the current locale.
     * Use {@link Messages.resolveLocale} for resolving partial locale strings.
     *
     * If `keypath` is empty, adds or sets the complete message object for the corresponding locale.
     * If any keys in `keypath` do not exist, a new object will be created at that key.
     *
     * @param data - Hierarchical map of keys to functions, or a single message function
     * @param locale - If empty or undefined, defaults to `this.locale`
     * @param keypath - The keypath being added
     */
    addMessages(data: MessageData | MessageFunction, locale?: string, keypath?: string[]): this;
    /**
     * Resolve `lc` to the key of an available locale or `null`, allowing for partial matches.
     *
     * @remarks
     * For example, with an `en` locale defined, it will be selected by `messages.defaultLocale = 'en-US'` and vice versa.
     */
    resolveLocale(locale: string | null): string | null;
    /**
     * Get the list of fallback locales
     *
     * @param locale - If empty or undefined, defaults to `this.locale`
     */
    getFallback(locale?: string | null): string[];
    /**
     * Set the fallback locale or locales for `lc`
     *
     * @remarks
     * To disable fallback for the locale, use `setFallback(lc, [])`.
     * To use the default fallback, use `setFallback(lc, null)`.
     */
    setFallback(lc: string, fallback: string[] | null): this;
    /**
     * Check if `key` is a message function for the locale
     *
     * @remarks
     * `key` may be a `string` for functions at the root level, or `string[]` for
     * accessing hierarchical objects. If an exact match is not found and
     * `fallback` is true, the fallback locales are checked for the first match.
     *
     * @param key - The key or keypath being sought
     * @param locale - If empty or undefined, defaults to `this.locale`
     * @param fallback - If true, also checks fallback locales
     */
    hasMessage(key: string | string[], locale?: string, fallback?: boolean): boolean;
    /**
     * Check if `key` is a message object for the locale
     *
     * @remarks
     * `key` may be a `string` for functions at the root level, or `string[]` for
     * accessing hierarchical objects. If an exact match is not found and
     * `fallback` is true, the fallback locales are checked for the first match.
     *
     * @param key - The key or keypath being sought
     * @param locale - If empty or undefined, defaults to `this.locale`
     * @param fallback - If true, also checks fallback locales
     */
    hasObject(key: string | string[], locale?: string, fallback?: boolean): boolean;
    /**
     * Get the message or object corresponding to `key`
     *
     * @remarks
     * `key` may be a `string` for functions at the root level, or `string[]` for accessing hierarchical objects.
     * If an exact match is not found, the fallback locales are checked for the first match.
     *
     * If `key` maps to a message function, the returned value will be the result of calling it with `props`.
     * If it maps to an object, the object is returned directly.
     * If nothing is found, `key` is returned.
     *
     * @param key - The key or keypath being sought
     * @param props - Optional properties passed to the function
     * @param lc - If empty or undefined, defaults to `this.locale`
     */
    get(key: string | string[], props?: Record<string, unknown>, locale?: string): string | unknown[] | MessageData;
}
