"""Calendar is a dictionary like Python object that can render itself as VCAL
files according to RFC 5545.

These are the defined components.
"""
from __future__ import annotations

import os
from datetime import date, datetime, timedelta
from typing import List, Optional, Tuple

import dateutil.rrule
import dateutil.tz
from icalendar.caselessdict import CaselessDict
from icalendar.parser import Contentline, Contentlines, Parameters, q_join, q_split
from icalendar.parser_tools import DEFAULT_ENCODING
from icalendar.prop import TypesFactory, vDDDLists, vDDDTypes, vText, vDuration
from icalendar.timezone import tzp


def get_example(component_directory: str, example_name: str) -> bytes:
    """Return an example and raise an error if it is absent."""
    here = os.path.dirname(__file__)
    examples = os.path.join(here, "tests", component_directory)
    if not example_name.endswith(".ics"):
        example_name = example_name + ".ics"
    example_file = os.path.join(examples, example_name)
    if not os.path.isfile(example_file):
        raise ValueError(f"Example {example_name} for {component_directory} not found. You can use one of {', '.join(os.listdir(examples))}")
    with open(example_file, "rb") as f:
        return f.read()


######################################
# The component factory

class ComponentFactory(CaselessDict):
    """All components defined in RFC 5545 are registered in this factory class.
    To get a component you can use it like this.
    """

    def __init__(self, *args, **kwargs):
        """Set keys to upper for initial dict.
        """
        super().__init__(*args, **kwargs)
        self['VEVENT'] = Event
        self['VTODO'] = Todo
        self['VJOURNAL'] = Journal
        self['VFREEBUSY'] = FreeBusy
        self['VTIMEZONE'] = Timezone
        self['STANDARD'] = TimezoneStandard
        self['DAYLIGHT'] = TimezoneDaylight
        self['VALARM'] = Alarm
        self['VCALENDAR'] = Calendar


# These Properties have multiple property values inlined in one propertyline
# seperated by comma. Use CaselessDict as simple caseless set.
INLINE = CaselessDict({
    'CATEGORIES': 1,
    'RESOURCES': 1,
    'FREEBUSY': 1,
})

_marker = []

class InvalidCalendar(ValueError):
    """The calendar given is not valid.

    This calendar does not conform with RFC 5545 or breaks other RFCs.
    """

class IncompleteComponent(ValueError):
    """The component is missing attributes.

    The attributes are not required, otherwise this would be
    an InvalidCalendar. But in order to perform calculations,
    this attribute is required.
    """



class Component(CaselessDict):
    """Component is the base object for calendar, Event and the other
    components defined in RFC 5545. Normally you will not use this class
    directly, but rather one of the subclasses.
    """

    name = None         # should be defined in each component
    required = ()       # These properties are required
    singletons = ()     # These properties must only appear once
    multiple = ()       # may occur more than once
    exclusive = ()      # These properties are mutually exclusive
    inclusive = ()      # if any occurs the other(s) MUST occur
                        # ('duration', 'repeat')
    ignore_exceptions = False   # if True, and we cannot parse this
                                # component, we will silently ignore
                                # it, rather than let the exception
                                # propagate upwards
    # not_compliant = ['']  # List of non-compliant properties.

    def __init__(self, *args, **kwargs):
        """Set keys to upper for initial dict.
        """
        super().__init__(*args, **kwargs)
        # set parameters here for properties that use non-default values
        self.subcomponents = []  # Components can be nested.
        self.errors = []  # If we ignored exception(s) while
                          # parsing a property, contains error strings

    # def is_compliant(self, name):
    #    """Returns True is the given property name is compliant with the
    #    icalendar implementation.
    #
    #    If the parser is too strict it might prevent parsing erroneous but
    #    otherwise compliant properties. So the parser is pretty lax, but it is
    #    possible to test for non-compliance by calling this method.
    #    """
    #    return name in not_compliant

    def __bool__(self):
        """Returns True, CaselessDict would return False if it had no items.
        """
        return True

    # python 2 compatibility
    __nonzero__ = __bool__

    def is_empty(self):
        """Returns True if Component has no items or subcomponents, else False.
        """
        return True if not (list(self.values()) + self.subcomponents) else False  # noqa

    #############################
    # handling of property values

    @staticmethod
    def _encode(name, value, parameters=None, encode=1):
        """Encode values to icalendar property values.

        :param name: Name of the property.
        :type name: string

        :param value: Value of the property. Either of a basic Python type of
                      any of the icalendar's own property types.
        :type value: Python native type or icalendar property type.

        :param parameters: Property parameter dictionary for the value. Only
                           available, if encode is set to True.
        :type parameters: Dictionary

        :param encode: True, if the value should be encoded to one of
                       icalendar's own property types (Fallback is "vText")
                       or False, if not.
        :type encode: Boolean

        :returns: icalendar property value
        """
        if not encode:
            return value
        if isinstance(value, types_factory.all_types):
            # Don't encode already encoded values.
            obj = value
        else:
            klass = types_factory.for_property(name)
            obj = klass(value)
        if parameters:
            if not hasattr(obj, "params"):
                obj.params = Parameters()
            for key, item in parameters.items():
                if item is None:
                    if key in obj.params:
                        del obj.params[key]
                else:
                    obj.params[key] = item
        return obj

    def add(self, name, value, parameters=None, encode=1):
        """Add a property.

        :param name: Name of the property.
        :type name: string

        :param value: Value of the property. Either of a basic Python type of
                      any of the icalendar's own property types.
        :type value: Python native type or icalendar property type.

        :param parameters: Property parameter dictionary for the value. Only
                           available, if encode is set to True.
        :type parameters: Dictionary

        :param encode: True, if the value should be encoded to one of
                       icalendar's own property types (Fallback is "vText")
                       or False, if not.
        :type encode: Boolean

        :returns: None
        """
        if isinstance(value, datetime) and\
                name.lower() in ('dtstamp', 'created', 'last-modified'):
            # RFC expects UTC for those... force value conversion.
            value = tzp.localize_utc(value)

        # encode value
        if encode and isinstance(value, list) \
                and name.lower() not in ['rdate', 'exdate', 'categories']:
            # Individually convert each value to an ical type except rdate and
            # exdate, where lists of dates might be passed to vDDDLists.
            value = [self._encode(name, v, parameters, encode) for v in value]
        else:
            value = self._encode(name, value, parameters, encode)

        # set value
        if name in self:
            # If property already exists, append it.
            oldval = self[name]
            if isinstance(oldval, list):
                if isinstance(value, list):
                    value = oldval + value
                else:
                    oldval.append(value)
                    value = oldval
            else:
                value = [oldval, value]
        self[name] = value

    def _decode(self, name, value):
        """Internal for decoding property values.
        """

        # TODO: Currently the decoded method calls the icalendar.prop instances
        # from_ical. We probably want to decode properties into Python native
        # types here. But when parsing from an ical string with from_ical, we
        # want to encode the string into a real icalendar.prop property.
        if isinstance(value, vDDDLists):
            # TODO: Workaround unfinished decoding
            return value
        decoded = types_factory.from_ical(name, value)
        # TODO: remove when proper decoded is implemented in every prop.* class
        # Workaround to decode vText properly
        if isinstance(decoded, vText):
            decoded = decoded.encode(DEFAULT_ENCODING)
        return decoded

    def decoded(self, name, default=_marker):
        """Returns decoded value of property.
        """
        # XXX: fail. what's this function supposed to do in the end?
        # -rnix

        if name in self:
            value = self[name]
            if isinstance(value, list):
                return [self._decode(name, v) for v in value]
            return self._decode(name, value)
        else:
            if default is _marker:
                raise KeyError(name)
            else:
                return default

    ########################################################################
    # Inline values. A few properties have multiple values inlined in in one
    # property line. These methods are used for splitting and joining these.

    def get_inline(self, name, decode=1):
        """Returns a list of values (split on comma).
        """
        vals = [v.strip('" ') for v in q_split(self[name])]
        if decode:
            return [self._decode(name, val) for val in vals]
        return vals

    def set_inline(self, name, values, encode=1):
        """Converts a list of values into comma separated string and sets value
        to that.
        """
        if encode:
            values = [self._encode(name, value, encode=1) for value in values]
        self[name] = types_factory['inline'](q_join(values))

    #########################
    # Handling of components

    def add_component(self, component):
        """Add a subcomponent to this component.
        """
        self.subcomponents.append(component)

    def _walk(self, name, select):
        """Walk to given component.
        """
        result = []
        if (name is None or self.name == name) and select(self):
            result.append(self)
        for subcomponent in self.subcomponents:
            result += subcomponent._walk(name, select)
        return result

    def walk(self, name=None, select=lambda c: True):
        """Recursively traverses component and subcomponents. Returns sequence
        of same. If name is passed, only components with name will be returned.

        :param name: The name of the component or None such as ``VEVENT``.
        :param select: A function that takes the component as first argument
          and returns True/False.
        :returns: A list of components that match.
        :rtype: list[Component]
        """
        if name is not None:
            name = name.upper()
        return self._walk(name, select)

    #####################
    # Generation

    def property_items(self, recursive=True, sorted=True):
        """Returns properties in this component and subcomponents as:
        [(name, value), ...]
        """
        vText = types_factory['text']
        properties = [('BEGIN', vText(self.name).to_ical())]
        if sorted:
            property_names = self.sorted_keys()
        else:
            property_names = self.keys()

        for name in property_names:
            values = self[name]
            if isinstance(values, list):
                # normally one property is one line
                for value in values:
                    properties.append((name, value))
            else:
                properties.append((name, values))
        if recursive:
            # recursion is fun!
            for subcomponent in self.subcomponents:
                properties += subcomponent.property_items(sorted=sorted)
        properties.append(('END', vText(self.name).to_ical()))
        return properties

    @classmethod
    def from_ical(cls, st, multiple=False):
        """Populates the component recursively from a string.
        """
        stack = []  # a stack of components
        comps = []
        for line in Contentlines.from_ical(st):  # raw parsing
            if not line:
                continue

            try:
                name, params, vals = line.parts()
            except ValueError as e:
                # if unable to parse a line within a component
                # that ignores exceptions, mark the component
                # as broken and skip the line. otherwise raise.
                component = stack[-1] if stack else None
                if not component or not component.ignore_exceptions:
                    raise
                component.errors.append((None, str(e)))
                continue

            uname = name.upper()
            # check for start of component
            if uname == 'BEGIN':
                # try and create one of the components defined in the spec,
                # otherwise get a general Components for robustness.
                c_name = vals.upper()
                c_class = component_factory.get(c_name, Component)
                # If component factory cannot resolve ``c_name``, the generic
                # ``Component`` class is used which does not have the name set.
                # That's opposed to the usage of ``cls``, which represents a
                # more concrete subclass with a name set (e.g. VCALENDAR).
                component = c_class()
                if not getattr(component, 'name', ''):  # undefined components
                    component.name = c_name
                stack.append(component)
            # check for end of event
            elif uname == 'END':
                # we are done adding properties to this component
                # so pop it from the stack and add it to the new top.
                if not stack:
                    # The stack is currently empty, the input must be invalid
                    raise ValueError('END encountered without an accompanying BEGIN!')

                component = stack.pop()
                if not stack:  # we are at the end
                    comps.append(component)
                else:
                    stack[-1].add_component(component)
                if vals == 'VTIMEZONE' and 'TZID' in component:
                    tzp.cache_timezone_component(component)
            # we are adding properties to the current top of the stack
            else:
                factory = types_factory.for_property(name)
                component = stack[-1] if stack else None
                if not component:
                    # only accept X-COMMENT at the end of the .ics file
                    # ignore these components in parsing
                    if uname == 'X-COMMENT':
                        break
                    else:
                        raise ValueError(f'Property "{name}" does not have a parent component.')
                datetime_names = ('DTSTART', 'DTEND', 'RECURRENCE-ID', 'DUE',
                                  'RDATE', 'EXDATE')
                try:
                    if name == 'FREEBUSY':
                        vals = vals.split(',')
                        if 'TZID' in params:
                            parsed_components = [factory(factory.from_ical(val, params['TZID'])) for val in vals]
                        else:
                            parsed_components = [factory(factory.from_ical(val)) for val in vals]
                    elif name in datetime_names and 'TZID' in params:
                        parsed_components = [factory(factory.from_ical(vals, params['TZID']))]
                    else:
                        parsed_components = [factory(factory.from_ical(vals))]
                except ValueError as e:
                    if not component.ignore_exceptions:
                        raise
                    component.errors.append((uname, str(e)))
                else:
                    for parsed_component in parsed_components:
                        parsed_component.params = params
                        component.add(name, parsed_component, encode=0)

        if multiple:
            return comps
        if len(comps) > 1:
            raise ValueError(cls._format_error(
                'Found multiple components where only one is allowed', st))
        if len(comps) < 1:
            raise ValueError(cls._format_error(
                'Found no components where exactly one is required', st))
        return comps[0]

    def _format_error(error_description, bad_input, elipsis='[...]'):
        # there's three character more in the error, ie. ' ' x2 and a ':'
        max_error_length = 100 - 3
        if len(error_description) + len(bad_input) + len(elipsis) > max_error_length:
            truncate_to = max_error_length - len(error_description) - len(elipsis)
            return f'{error_description}: {bad_input[:truncate_to]} {elipsis}'
        else:
            return f'{error_description}: {bad_input}'

    def content_line(self, name, value, sorted=True):
        """Returns property as content line.
        """
        params = getattr(value, 'params', Parameters())
        return Contentline.from_parts(name, params, value, sorted=sorted)

    def content_lines(self, sorted=True):
        """Converts the Component and subcomponents into content lines.
        """
        contentlines = Contentlines()
        for name, value in self.property_items(sorted=sorted):
            cl = self.content_line(name, value, sorted=sorted)
            contentlines.append(cl)
        contentlines.append('')  # remember the empty string in the end
        return contentlines

    def to_ical(self, sorted=True):
        '''
        :param sorted: Whether parameters and properties should be
                       lexicographically sorted.
        '''

        content_lines = self.content_lines(sorted=sorted)
        return content_lines.to_ical()

    def __repr__(self):
        """String representation of class with all of it's subcomponents.
        """
        subs = ', '.join(str(it) for it in self.subcomponents)
        return f"{self.name or type(self).__name__}({dict(self)}{', ' + subs if subs else ''})"

    def __eq__(self, other):
        if len(self.subcomponents) != len(other.subcomponents):
            return False

        properties_equal = super().__eq__(other)
        if not properties_equal:
            return False

        # The subcomponents might not be in the same order,
        # neither there's a natural key we can sort the subcomponents by nor
        # are the subcomponent types hashable, so  we cant put them in a set to
        # check for set equivalence. We have to iterate over the subcomponents
        # and look for each of them in the list.
        for subcomponent in self.subcomponents:
            if subcomponent not in other.subcomponents:
                return False

        return True

#######################################
# components defined in RFC 5545

def create_single_property(prop:str, value_attr:str, value_type:tuple[type], type_def:type, doc:str):
    """Create a single property getter and setter."""

    def p_get(self : Component):
        default = object()
        result = self.get(prop, default)
        if result is default:
            return None
        if isinstance(result, list):
            raise InvalidCalendar(f"Multiple {prop} defined.")
        value = getattr(result, value_attr, result)
        if not isinstance(value, value_type):
            raise InvalidCalendar(f"{prop} must be either a date or a datetime, not {value}.")
        return value

    def p_set(self:Component, value) -> None:
        if value is None:
            p_del(self)
            return
        if not isinstance(value, value_type):
            raise TypeError(f"Use {' or '.join(t.__name__ for t in value_type)}, not {type(value).__name__}.")
        self[prop] = vDDDTypes(value)
        if prop in self.exclusive:
            for other_prop in self.exclusive:
                if other_prop != prop:
                    self.pop(other_prop, None)
    p_set.__annotations__["value"] = p_get.__annotations__["return"] = Optional[type_def]

    def p_del(self:Component):
        self.pop(prop)

    p_doc = f"""The {prop} property.

    {doc}

    Accepted values: {', '.join(t.__name__ for t in value_type)}.
    If the attribute has invalid values, we raise InvalidCalendar.
    If the value is absent, we return None.
    You can also delete the value with del or by setting it to None.
    """
    return property(p_get, p_set, p_del, p_doc)


def is_date(dt: date) -> bool:
    """Whether this is a date and not a datetime."""
    return isinstance(dt, date) and not isinstance(dt, datetime)

def is_datetime(dt: date) -> bool:
    """Whether this is a date and not a datetime."""
    return isinstance(dt, datetime)

class Event(Component):

    name = 'VEVENT'

    canonical_order = (
        'SUMMARY', 'DTSTART', 'DTEND', 'DURATION', 'DTSTAMP',
        'UID', 'RECURRENCE-ID', 'SEQUENCE', 'RRULE', 'RDATE',
        'EXDATE',
    )

    required = ('UID', 'DTSTAMP',)
    singletons = (
        'CLASS', 'CREATED', 'DESCRIPTION', 'DTSTART', 'GEO', 'LAST-MODIFIED',
        'LOCATION', 'ORGANIZER', 'PRIORITY', 'DTSTAMP', 'SEQUENCE', 'STATUS',
        'SUMMARY', 'TRANSP', 'URL', 'RECURRENCE-ID', 'DTEND', 'DURATION',
        'UID', 'CATEGORIES',
    )
    exclusive = ('DTEND', 'DURATION',)
    multiple = (
        'ATTACH', 'ATTENDEE', 'COMMENT', 'CONTACT', 'EXDATE',
        'RSTATUS', 'RELATED', 'RESOURCES', 'RDATE', 'RRULE'
    )
    ignore_exceptions = True

    @classmethod
    def example(cls, name:str) -> Event:
        """Return the calendar example with the given name."""
        return cls.from_ical(get_example("events", name))

    DTSTART = create_single_property("DTSTART", "dt", (datetime, date), date, 'The "DTSTART" property for a "VEVENT" specifies the inclusive start of the event.')
    DTEND = create_single_property("DTEND", "dt", (datetime, date), date, 'The "DTEND" property for a "VEVENT" calendar component specifies the non-inclusive end of the event.')

    def _get_start_end_duration(self):
        """Verify the calendar validity and return the right attributes."""
        start = self.DTSTART
        end = self.DTEND
        duration = self.DURATION
        if duration is not None and end is not None:
            raise InvalidCalendar("Only one of DTEND and DURATION may be in a VEVENT, not both.")
        if isinstance(start, date) and not isinstance(start, datetime) and duration is not None and duration.seconds != 0:
            raise InvalidCalendar("When DTSTART is a date, DURATION must be of days or weeks.")
        if start is not None and end is not None and is_date(start) != is_date(end):
            raise InvalidCalendar("DTSTART and DTEND must be of the same type, either date or datetime.")
        return start, end, duration

    @property
    def DURATION(self) -> Optional[timedelta]:  # noqa: N802
        """The DURATION of the component.

        The "DTSTART" property for a "VEVENT" specifies the inclusive start of the event.
        The "DURATION" property in conjunction with the DTSTART property
        for a "VEVENT" calendar component specifies the non-inclusive end
        of the event.

        If you would like to calculate the duration of an event do not use this.
        Instead use the difference between DTSTART and DTEND.
        """
        default = object()
        duration = self.get("duration", default)
        if isinstance(duration, vDDDTypes):
            return duration.dt
        if isinstance(duration, vDuration):
            return duration.td
        if duration is not default and not isinstance(duration, timedelta):
            raise InvalidCalendar(f"DURATION must be a timedelta, not {type(duration).__name__}.")
        return None
    
    @DURATION.setter
    def DURATION(self, value: Optional[timedelta]):  # noqa: N802
        if value is None:
            self.pop("duration", None)
            return
        if not isinstance(value, timedelta):
            raise TypeError(f"Use timedelta, not {type(value).__name__}.")
        self["duration"] = vDuration(value)
        del self.DTEND

    @property
    def duration(self) -> timedelta:
        """The duration of the component.

        This duration is calculated from the start and end of the event.
        You cannot set the duration as it is unclear what happens to start and end.
        """
        return self.end - self.start

    @property
    def start(self) -> date | datetime:
        """The start of the component.

        Invalid values raise an InvalidCalendar.
        If there is no start, we also raise an IncompleteComponent error.

        You can get the start, end and duration of an event as follows:

        >>> from datetime import datetime
        >>> from icalendar import Event
        >>> event = Event()
        >>> event.start = datetime(2021, 1, 1, 12)
        >>> event.end = datetime(2021, 1, 1, 12, 30) # 30 minutes
        >>> event.end - event.start  # 1800 seconds == 30 minutes
        datetime.timedelta(seconds=1800)
        >>> print(event.to_ical())
        BEGIN:VEVENT
        DTSTART:20210101T120000
        DTEND:20210101T123000
        END:VEVENT
        """
        start = self._get_start_end_duration()[0]
        if start is None:
            raise IncompleteComponent("No DTSTART given.")
        return start

    @start.setter
    def start(self, start: Optional[date | datetime]):
        """Set the start."""
        self.DTSTART = start

    @property
    def end(self) -> date | datetime:
        """The end of the component.

        Invalid values raise an InvalidCalendar error.
        If there is no end, we also raise an IncompleteComponent error.
        """
        start, end, duration = self._get_start_end_duration()
        if end is None and duration is None:
            if start is None:
                raise IncompleteComponent("No DTEND or DURATION+DTSTART given.")
            if is_date(start):
                return start + timedelta(days=1)
            return start
        if duration is not None:
            if start is not None:
                return start + duration
            raise IncompleteComponent("No DTEND or DURATION+DTSTART given.")
        return end

    @end.setter
    def end(self, end: date | datetime | None):
        """Set the end."""
        self.DTEND = end


class Todo(Component):

    name = 'VTODO'

    required = ('UID', 'DTSTAMP',)
    singletons = (
        'CLASS', 'COMPLETED', 'CREATED', 'DESCRIPTION', 'DTSTAMP', 'DTSTART',
        'GEO', 'LAST-MODIFIED', 'LOCATION', 'ORGANIZER', 'PERCENT-COMPLETE',
        'PRIORITY', 'RECURRENCE-ID', 'SEQUENCE', 'STATUS', 'SUMMARY', 'UID',
        'URL', 'DUE', 'DURATION',
    )
    exclusive = ('DUE', 'DURATION',)
    multiple = (
        'ATTACH', 'ATTENDEE', 'CATEGORIES', 'COMMENT', 'CONTACT', 'EXDATE',
        'RSTATUS', 'RELATED', 'RESOURCES', 'RDATE', 'RRULE'
    )


class Journal(Component):
    """A descriptive text at a certain time or associated with a component.

    A "VJOURNAL" calendar component is a grouping of
    component properties that represent one or more descriptive text
    notes associated with a particular calendar date.  The "DTSTART"
    property is used to specify the calendar date with which the
    journal entry is associated.  Generally, it will have a DATE value
    data type, but it can also be used to specify a DATE-TIME value
    data type.  Examples of a journal entry include a daily record of
    a legislative body or a journal entry of individual telephone
    contacts for the day or an ordered list of accomplishments for the
    day.
    """

    name = 'VJOURNAL'

    required = ('UID', 'DTSTAMP',)
    singletons = (
        'CLASS', 'CREATED', 'DTSTART', 'DTSTAMP', 'LAST-MODIFIED', 'ORGANIZER',
        'RECURRENCE-ID', 'SEQUENCE', 'STATUS', 'SUMMARY', 'UID', 'URL',
    )
    multiple = (
        'ATTACH', 'ATTENDEE', 'CATEGORIES', 'COMMENT', 'CONTACT', 'EXDATE',
        'RELATED', 'RDATE', 'RRULE', 'RSTATUS', 'DESCRIPTION',
    )

    DTSTART = create_single_property(
        "DTSTART", "dt", (datetime, date), date,
        'The "DTSTART" property for a "VJOURNAL" that specifies the exact date at which the journal entry was made.')

    @property
    def start(self) -> date:
        """The start of the Journal.
        
        The "DTSTART"
        property is used to specify the calendar date with which the
        journal entry is associated.
        """
        start = self.DTSTART
        if start is None:
            raise IncompleteComponent("No DTSTART given.")
        return start
    
    @start.setter
    def start(self, value: datetime|date) -> None:
        """Set the start of the journal."""
        self.DTSTART = value

    end = start
    
    @property
    def duration(self) -> timedelta:
        """The journal has no duration."""
        return timedelta(0)

class FreeBusy(Component):

    name = 'VFREEBUSY'

    required = ('UID', 'DTSTAMP',)
    singletons = (
        'CONTACT', 'DTSTART', 'DTEND', 'DTSTAMP', 'ORGANIZER',
        'UID', 'URL',
    )
    multiple = ('ATTENDEE', 'COMMENT', 'FREEBUSY', 'RSTATUS',)


class Timezone(Component):
    name = 'VTIMEZONE'
    canonical_order = ('TZID',)
    required = ('TZID',) # it also requires one of components DAYLIGHT and STANDARD
    singletons = ('TZID', 'LAST-MODIFIED', 'TZURL',)

    @classmethod
    def example(cls, name: str) -> Calendar:
        """Return the calendar example with the given name."""
        return cls.from_ical(get_example("timezones", name))

    @staticmethod
    def _extract_offsets(component, tzname):
        """extract offsets and transition times from a VTIMEZONE component
        :param component: a STANDARD or DAYLIGHT component
        :param tzname: the name of the zone
        """
        offsetfrom = component['TZOFFSETFROM'].td
        offsetto = component['TZOFFSETTO'].td
        dtstart = component['DTSTART'].dt

        # offsets need to be rounded to the next minute, we might loose up
        # to 30 seconds accuracy, but it can't be helped (datetime
        # supposedly cannot handle smaller offsets)
        offsetto_s = int((offsetto.seconds + 30) / 60) * 60
        offsetto = timedelta(days=offsetto.days, seconds=offsetto_s)
        offsetfrom_s = int((offsetfrom.seconds + 30) / 60) * 60
        offsetfrom = timedelta(days=offsetfrom.days, seconds=offsetfrom_s)

        # expand recurrences
        if 'RRULE' in component:
            # to be paranoid about correct weekdays
            # evaluate the rrule with the current offset
            tzi = dateutil.tz.tzoffset ("(offsetfrom)", offsetfrom)
            rrstart = dtstart.replace (tzinfo=tzi)

            rrulestr = component['RRULE'].to_ical().decode('utf-8')
            rrule = dateutil.rrule.rrulestr(rrulestr, dtstart=rrstart)
            tzp.fix_rrule_until(rrule, component['RRULE'])

            # constructing the timezone requires UTC transition times.
            # here we construct local times without tzinfo, the offset to UTC
            # gets subtracted in to_tz().
            transtimes = [dt.replace (tzinfo=None) for dt in rrule]

        # or rdates
        elif 'RDATE' in component:
            if not isinstance(component['RDATE'], list):
                rdates = [component['RDATE']]
            else:
                rdates = component['RDATE']
            transtimes = [dtstart] + [leaf.dt for tree in rdates for
                                      leaf in tree.dts]
        else:
            transtimes = [dtstart]

        transitions = [(transtime, offsetfrom, offsetto, tzname) for
                       transtime in set(transtimes)]

        if component.name == 'STANDARD':
            is_dst = 0
        elif component.name == 'DAYLIGHT':
            is_dst = 1
        return is_dst, transitions

    @staticmethod
    def _make_unique_tzname(tzname, tznames):
        """
        :param tzname: Candidate tzname
        :param tznames: Other tznames
        """
        # TODO better way of making sure tznames are unique
        while tzname in tznames:
            tzname += '_1'
        tznames.add(tzname)
        return tzname

    def to_tz(self, tzp=tzp):
        """convert this VTIMEZONE component to a timezone object
        """
        return tzp.create_timezone(self)

    @property
    def tz_name(self) -> str:
        """Return the name of the timezone component.

        Please note that the names of the timezone are different from this name
        and may change with winter/summer time.
        """
        try:
            return str(self['TZID'])
        except UnicodeEncodeError:
            return self['TZID'].encode('ascii', 'replace')

    def get_transitions(self) -> Tuple[List[datetime], List[Tuple[timedelta, timedelta, str]]]:
        """Return a tuple of (transition_times, transition_info)

        - transition_times = [datetime, ...]
        - transition_info = [(TZOFFSETTO, dts_offset, tzname)]

        """
        zone = self.tz_name
        transitions = []
        dst = {}
        tznames = set()
        for component in self.walk():
            if type(component) == Timezone:
                continue
            assert isinstance(component['DTSTART'].dt, datetime), (
                "VTIMEZONEs sub-components' DTSTART must be of type datetime, not date"
            )
            try:
                tzname = str(component['TZNAME'])
            except UnicodeEncodeError:
                tzname = component['TZNAME'].encode('ascii', 'replace')
                tzname = self._make_unique_tzname(tzname, tznames)
            except KeyError:
                # for whatever reason this is str/unicode
                tzname = f"{zone}_{component['DTSTART'].to_ical().decode('utf-8')}_" + \
                         f"{component['TZOFFSETFROM'].to_ical()}_" + \
                         f"{component['TZOFFSETTO'].to_ical()}"
                tzname = self._make_unique_tzname(tzname, tznames)

            dst[tzname], component_transitions = self._extract_offsets(
                component, tzname
            )
            transitions.extend(component_transitions)

        transitions.sort()
        transition_times = [
            transtime - osfrom for transtime, osfrom, _, _ in transitions
        ]

        # transition_info is a list with tuples in the format
        # (utcoffset, dstoffset, name)
        # dstoffset = 0, if current transition is to standard time
        #           = this_utcoffset - prev_standard_utcoffset, otherwise
        transition_info = []
        for num, (transtime, osfrom, osto, name) in enumerate(transitions):
            dst_offset = False
            if not dst[name]:
                dst_offset = timedelta(seconds=0)
            else:
                # go back in time until we find a transition to dst
                for index in range(num - 1, -1, -1):
                    if not dst[transitions[index][3]]:  # [3] is the name
                        dst_offset = osto - transitions[index][2]  # [2] is osto  # noqa
                        break
                # when the first transition is to dst, we didn't find anything
                # in the past, so we have to look into the future
                if not dst_offset:
                    for index in range(num, len(transitions)):
                        if not dst[transitions[index][3]]:  # [3] is the name
                            dst_offset = osto - transitions[index][2]  # [2] is osto  # noqa
                            break
            assert dst_offset is not False
            transition_info.append((osto, dst_offset, name))
        return transition_times, transition_info


class TimezoneStandard(Component):
    name = 'STANDARD'
    required = ('DTSTART', 'TZOFFSETTO', 'TZOFFSETFROM')
    singletons = ('DTSTART', 'TZOFFSETTO', 'TZOFFSETFROM',)
    multiple = ('COMMENT', 'RDATE', 'TZNAME', 'RRULE', 'EXDATE')


class TimezoneDaylight(Component):
    name = 'DAYLIGHT'
    required = TimezoneStandard.required
    singletons = TimezoneStandard.singletons
    multiple = TimezoneStandard.multiple


class Alarm(Component):

    name = 'VALARM'
    # some properties MAY/MUST/MUST NOT appear depending on ACTION value
    required = ('ACTION', 'TRIGGER',)
    singletons = (
            'ATTACH', 'ACTION', 'DESCRIPTION', 'SUMMARY', 'TRIGGER',
            'DURATION', 'REPEAT',
            )
    inclusive = (('DURATION', 'REPEAT',), ('SUMMARY', 'ATTENDEE',))
    multiple = ('ATTENDEE', 'ATTACH')


class Calendar(Component):
    """This is the base object for an iCalendar file.
    """
    name = 'VCALENDAR'
    canonical_order = ('VERSION', 'PRODID', 'CALSCALE', 'METHOD',)
    required = ('PRODID', 'VERSION', )
    singletons = ('PRODID', 'VERSION', 'CALSCALE', 'METHOD')

    @classmethod
    def example(cls, name: str) -> Calendar:
        """Return the calendar example with the given name."""
        return cls.from_ical(get_example("calendars", name))

# These are read only singleton, so one instance is enough for the module
types_factory = TypesFactory()
component_factory = ComponentFactory()

__all__ = ["Alarm", "Calendar", "Component", "ComponentFactory", "Event",
           "FreeBusy", "INLINE", "Journal", "Timezone", "TimezoneDaylight",
           "TimezoneStandard", "Todo", "component_factory", "get_example",
           "IncompleteComponent", "InvalidCalendar"]
