#!/usr/bin/env python

"""
User profile management.

Copyright (C) 2015, 2016, 2017 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from imiptools.config import settings
from imiptools.dates import get_default_timezone
from imiptools.filesys import fix_permissions, FileBase
from os.path import exists, isdir
from os import listdir, makedirs
import codecs
import pytz

# Fake gettext method for strings to be translated later.

_ = lambda s: s

def identity_dict(l):
    return dict([(i, i) for i in l])

class Preferences(FileBase):

    "A simple preferences file manager."

    # See: docs/wiki/Preferences

    known_keys = {
        "CN"                    : "",
        "TZID"                  : get_default_timezone(),
        "LANG"                  : settings["LANG"],
        "add_method_response"   : settings["ADD_RESPONSE_DEFAULT"],
        "event_refreshing"      : settings["REFRESHING_DEFAULT"],
        "freebusy_bundling"     : settings["BUNDLING_DEFAULT"],
        "freebusy_messages"     : settings["NOTIFYING_DEFAULT"],
        "freebusy_offers"       : settings["FREEBUSY_OFFER_DEFAULT"],
        "freebusy_publishing"   : settings["PUBLISHING_DEFAULT"],
        "freebusy_sharing"      : settings["SHARING_DEFAULT"],
        "incoming"              : settings["INCOMING_DEFAULT"],
        "organiser_replacement" : settings["ORGANISER_REPLACEMENT_DEFAULT"],
        "participating"         : settings["PARTICIPATING_DEFAULT"],
        "permitted_times"       : None,
        }

    known_key_choices = {
        "TZID"                  : identity_dict(pytz.all_timezones),
        "add_method_response"   : {
            "add"                   : _("Add events"),
            "ignore"                : _("Ignore requests"),
            "refresh"               : _("Ask for refreshed event details"),
                                  },
        "event_refreshing"      : {
            "never"                 : _("Do not respond"),
            "always"                : _("Always respond"),
                                  },
        "freebusy_bundling"     : {
            "never"                 : _("Never"),
            "always"                : _("Always"),
                                  },
        "freebusy_messages"     : {
            "none"                  : _("Do not notify"),
            "notify"                : _("Notify"),
                                  },
        "freebusy_publishing"   : {
            "publish"               : _("Publish"),
            "no"                    : _("Do not publish"),
                                  },
        "freebusy_sharing"      : {
            "share"                 : _("Share"),
            "no"                    : _("Do not share"),
                                  },
        "incoming"              : {
            "message-only"          : _("Original message only"),
            "message-then-summary"  : _("Original message followed by a separate summary message"),
            "summary-then-message"  : _("Summary message followed by the original message"),
            "summary-only"          : _("Summary message only"),
            "summary-wraps-message" : _("Summary message wrapping the original message"),
                                  },
        "organiser_replacement" : {
            "any"                   : _("Anyone"),
            "attendee"              : _("Existing attendees only"),
            "never"                 : _("Never allow organiser replacement"),
                                  },
        "participating"         : {
            "participate"           : _("Participate"),
            "no"                    : _("Do not participate"),
                                  }
        }

    def __init__(self, user, store_dir=None):
        FileBase.__init__(self, store_dir or settings["PREFERENCES_DIR"])
        self.user = user

    def get(self, name, default=None, config_default=False):

        """
        Return the value for 'name', with absent entries providing a default of
        None or any indicated 'default' or, if 'config_default' is set to a true
        value, the default value from the config module.
        """

        try:
            return self[name]
        except KeyError:
            if config_default:
                return self.known_keys.get(name, default)
            else:
                return default

    def get_all(self, names):

        """
        Return a dictionary containing values for entries having the given
        'names'. Absent entries for names are omitted without error.
        """

        d = {}
        for name in names:
            value = self.get(name)
            if value is not None:
                d[name] = value
        return d

    def has_key(self, name):

        "Return whether an entry exists for 'name'."

        try:
            self[name]
            return True
        except KeyError:
            return False

    def keys(self):

        "Return all entry names in the preferences."

        filename = self.get_object_in_store(self.user)
        if not filename or not isdir(filename):
            return []

        return listdir(filename)

    def items(self, all_known=False, default=None, config_default=False):

        """
        Return all entries in the preferences or all known entries if
        'all_known' is set to a true value, with absent entries providing a
        default of None or any indicated 'default' or, if 'config_default' is
        set to a true value, the default value from the config module.

        Each entry will have the form (key, value).
        """

        l = []
        for key in (all_known and self.known_keys or self).keys():
            l.append((key, self.get(key, default, config_default)))
        return l

    def choices(self, all_known=False, default=None, config_default=False):

        """
        Return all entries in the preferences or all known entries if
        'all_known' is set to a true value, with absent entries providing a
        default of None or any indicated 'default' or, if 'config_default' is
        set to a true value, the default value from the config module.

        Each entry will have the form (key, value, choices).
        """

        l = []
        for key, value in self.items(all_known, default, config_default):
            l.append((key, value, self.known_key_choices.get(key)))
        return l

    def __getitem__(self, name):

        "Return the value for 'name', raising a KeyError if absent."

        filename = self.get_object_in_store(self.user, name)
        if not filename or not exists(filename):
            raise KeyError, name

        f = codecs.open(filename, encoding="utf-8")
        try:
            return f.read().strip()
        finally:
            f.close()

    def __setitem__(self, name, value):

        "Set for 'name' the given 'value'."

        filename = self.get_object_in_store(self.user, name)
        if not filename:
            return False

        f = codecs.open(filename, "w", encoding="utf-8")
        try:
            f.write(value)
        finally:
            f.close()
            fix_permissions(filename)

        return True

# vim: tabstop=4 expandtab shiftwidth=4
