#!/usr/bin/env python

"""
Quota-related scheduling functionality.

Copyright (C) 2016 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from imiptools.dates import get_duration, to_utc_datetime
from imiptools.data import get_uri, uri_dict, Object
from imiptools.handlers.scheduling.common import get_scheduling_conflicts, \
                                                 standard_responses
from imiptools.period import Endless
from datetime import timedelta

# Quota maintenance.

def check_quota(handler, args):

    """
    Check the current object of the given 'handler' against the applicable
    quota.
    """

    _ = handler.get_translator()

    quota, group = _get_quota_and_group(handler, args)

    # Obtain the journal entries and check the balance.

    journal = handler.get_journal()
    entries = journal.get_entries(quota, group)
    limits = journal.get_limits(quota)

    # Obtain a limit for the group or any general limit.
    # Decline invitations if no limit has been set.

    limit = limits.get(group) or limits.get("*")
    if not limit:
        return "DECLINED", _("You have no quota allocation for the recipient.")

    # Where the quota is unlimited, accept the invitation.

    if limit == "*":
        return "ACCEPTED", _("The recipient has scheduled the requested period.")

    # Decline endless events for limited quotas.

    total = _get_duration(handler)

    if total == Endless():
        return "DECLINED", _("The event period exceeds your quota allocation for the recipient.")

    # Decline events whose durations exceed the balance.

    balance = get_duration(limit) - _get_usage(entries)

    if total > balance:
        return "DECLINED", _("The event period exceeds your quota allocation for the recipient.")
    else:
        return "ACCEPTED", _("The recipient has scheduled the requested period.")

def add_to_quota(handler, args):

    """
    Record details of the current object of the given 'handler' in the
    applicable quota.
    """

    quota, group = _get_quota_and_group(handler, args)
    _add_to_quota(handler, quota, group, handler.user, False)

def remove_from_quota(handler, args):

    """
    Remove details of the current object of the given 'handler' from the
    applicable quota.
    """

    quota, group = _get_quota_and_group(handler, args)
    _remove_from_quota(handler, quota, group, handler.user)

def update_event(handler, args):

    "Update a stored version of the current object of the given 'handler'."

    quota, group = _get_quota_and_group(handler, args)
    journal = handler.get_journal()

    # Where an existing version of the object exists, merge the recipient's
    # attendance information.

    fragment = journal.get_event(quota, handler.uid, handler.recurrenceid)
    obj = fragment and Object(fragment)
    if not obj:
        obj = handler.obj

    # Set attendance.

    attendee_map = uri_dict(obj.get_value_map("ATTENDEE"))
    attendee_map[handler.user]["PARTSTAT"] = "ACCEPTED"
    obj["ATTENDEE"] = attendee_map.items()

    # Record the object so that recurrences can be generated.

    journal.set_event(quota, handler.uid, handler.recurrenceid, obj.to_node())

def remove_event(handler, args):

    "Remove a stored version of the current object of the given 'handler'."

    quota, group = _get_quota_and_group(handler, args)
    journal = handler.get_journal()

    # Where an existing version of the object exists, remove the recipient's
    # attendance information.

    fragment = journal.get_event(quota, handler.uid, handler.recurrenceid)
    obj = fragment and Object(fragment)
    if not obj:
        return

    attendee_map = uri_dict(obj.get_value_map("ATTENDEE"))
    delegates = journal.get_delegates(quota)

    # Determine whether any of the delegates are still involved.

    attendees = set(delegates).intersection(attendee_map.keys())
    if handler.user in attendees:
        attendees.remove(handler.user)

    # Remove event details where no delegates will be involved.

    if not attendees:
        journal.remove_event(quota, handler.uid, handler.recurrenceid)
        return

    del attendee_map[handler.user]
    obj["ATTENDEE"] = attendee_map.items()

    # Record the object so that recurrences can be generated.

    journal.set_event(quota, handler.uid, handler.recurrenceid, obj.to_node())

def _get_quota_and_group(handler, args):

    """
    Combine information about the current object from the 'handler' with the
    given 'args' to return a tuple containing the quota group and the user
    identity or group involved.
    """

    quota = args and args[0] or handler.user

    # Obtain the identity to whom the quota will apply.

    organiser = get_uri(handler.obj.get_value("ORGANIZER"))

    # Obtain any user group to which the quota will apply instead.

    journal = handler.get_journal()
    groups = journal.get_groups(quota)

    return quota, groups.get(organiser) or groups.get("*") or organiser

def _get_duration(handler):

    "Return the duration of the current object provided by the 'handler'."

    # Reject indefinitely recurring events.

    if handler.obj.possibly_recurring_indefinitely():
        return Endless()

    # Otherwise, return a sum of the period durations.

    total = timedelta(0)

    for period in handler.get_periods(handler.obj):
        duration = period.get_duration()

        # Decline events whose period durations are endless.

        if duration == Endless():
            return duration
        else:
            total += duration

    return total

def _get_expiry_time(handler):

    """
    Return the expiry time for quota purposes of the current object provided by
    the 'handler'.
    """

    # Reject indefinitely recurring events.

    if handler.obj.possibly_recurring_indefinitely():
        return None

    periods = handler.get_periods(handler.obj)
    return periods and to_utc_datetime(periods[-1].get_end_point()) or None

def _get_usage(entries):

    "Return the usage total according to the given 'entries'."

    total = timedelta(0)
    for period in entries:
        total += period.get_duration()
    return total

def _add_to_quota(handler, quota, user, participant, is_organiser):

    """
    Record details of the current object of the 'handler' in the applicable
    free/busy resource.
    """

    journal = handler.get_journal()
    freebusy = journal.get_entries_for_update(quota, user)
    handler.update_freebusy(freebusy, participant, is_organiser)

    # Remove original recurrence details replaced by additional
    # recurrences, as well as obsolete additional recurrences.

    handler.remove_freebusy_for_recurrences(freebusy, journal.get_recurrences(quota, handler.uid))

    # Update free/busy provider information if the event may recur indefinitely.

    if handler.possibly_recurring_indefinitely():
        journal.append_freebusy_provider(quota, handler.obj)

    journal.set_entries(quota, user, freebusy)

def _remove_from_quota(handler, quota, user, participant):

    """
    Remove details of the current object of the 'handler' from the applicable
    free/busy resource.
    """

    journal = handler.get_journal()
    freebusy = journal.get_entries_for_update(quota, user)

    # Remove only the entries associated with this recipient.

    removed = handler.remove_from_freebusy(freebusy)
    for p in removed:
        if p.attendee != participant:
            freebusy.insert_period(p)

    # Update free/busy provider information if the event may recur indefinitely.

    if handler.possibly_recurring_indefinitely():
        journal.remove_freebusy_provider(quota, handler.obj)

    journal.set_entries(quota, user, freebusy)

# Collective free/busy maintenance.

def schedule_across_quota(handler, args):

    """
    Check the current object of the given 'handler' against the individual
    schedules managed by the quota. The consolidated schedules are not tested,
    nor are the quotas themselves.
    """

    quota, organiser = _get_quota_and_identity(handler, args)

    # Check the event periods against the quota's consolidated record of the
    # organiser's reservations.

    periods = handler.get_periods(handler.obj)
    freebusy = handler.get_journal().get_entries(quota, organiser)
    scheduled = handler.can_schedule(freebusy, periods)

    return standard_responses(handler, scheduled and "ACCEPTED" or "DECLINED")

def add_to_quota_freebusy(handler, args):

    """
    Record details of the current object of the 'handler' in the applicable
    free/busy resource.
    """

    quota, organiser = _get_quota_and_identity(handler, args)
    _add_to_quota(handler, quota, organiser, organiser, True)

def remove_from_quota_freebusy(handler, args):

    """
    Remove details of the current object of the 'handler' from the applicable
    free/busy resource.
    """

    quota, organiser = _get_quota_and_identity(handler, args)
    _remove_from_quota(handler, quota, organiser, organiser)

def _get_quota_and_identity(handler, args):

    """
    Combine information about the current object from the 'handler' with the
    given 'args' to return a tuple containing the quota group and the user
    identity involved.
    """

    quota = args and args[0] or handler.user

    # Obtain the identity for whom the scheduling will apply.

    organiser = get_uri(handler.obj.get_value("ORGANIZER"))

    return quota, organiser

# Delegation of reservations.

def schedule_for_delegate(handler, args):

    """
    Check the current object of the given 'handler' against the schedules
    managed by the quota, delegating to a specific recipient according to the
    given policies.
    """

    # First check the quota and decline any request that would exceed the quota.

    scheduled = check_quota(handler, args)
    response, description = scheduled or ("DECLINED", None)

    if response == "DECLINED":
        return response, description

    # Obtain the quota and organiser group details to evaluate delegation.

    quota, group = _get_quota_and_group(handler, args)
    policies = args and args[1:] or ["available"]

    # Determine the status of the recipient.

    attendee_map = uri_dict(handler.obj.get_value_map("ATTENDEE"))
    attendee_attr = attendee_map[handler.user]

    # Prevent delegation by a delegate.

    if attendee_attr.get("DELEGATED-FROM"):
        delegates = set([handler.user])

    # Obtain the delegate pool for the quota.

    else:
        delegates = handler.get_journal().get_delegates(quota)

        # Obtain the remaining delegates not already involved in the event.

        delegates = set(delegates).difference(attendee_map)
        delegates.add(handler.user)

    # Get the quota's schedule for the requested periods and identify
    # unavailable delegates.

    entries = handler.get_journal().get_entries(quota, group)
    conflicts = get_scheduling_conflicts(handler, entries, delegates, attendee=True)

    # Get the delegates in order of increasing unavailability (or decreasing
    # availability).

    unavailability = conflicts.items()

    # Apply the policies to choose a suitable delegate.

    if "most-available" in policies:
        unavailability.sort(key=lambda t: t[1])
        available = [delegate for (delegate, commitments) in unavailability]
        delegate = available and available[0]

    # The default is to select completely available delegates.

    else:
        available = [delegate for (delegate, commitments) in unavailability if not commitments]
        delegate = available and (handler.user in available and handler.user or available[0])

    # Only accept or delegate if a suitably available delegate is found.

    if delegate:

        # Add attendee for delegate, obtaining the original attendee dictionary.
        # Modify this user's status to refer to the delegate.

        if delegate != handler.user:
            attendee_map = handler.obj.get_value_map("ATTENDEE")
            attendee_map[delegate] = {"DELEGATED-FROM" : [handler.user]}
            attendee_attr["DELEGATED-TO"] = [delegate]
            handler.obj["ATTENDEE"] = attendee_map.items()

            response = "DELEGATED"
        else:
            response = "ACCEPTED"
    else:
        response = "DECLINED"

    return standard_responses(handler, response)

# Locking and unlocking.

def lock_journal(handler, args):

    "Using the 'handler' and 'args', lock the journal for the quota."

    handler.get_journal().acquire_lock(_get_quota(handler, args))

def unlock_journal(handler, args):

    "Using the 'handler' and 'args', unlock the journal for the quota."

    handler.get_journal().release_lock(_get_quota(handler, args))

def _get_quota(handler, args):

    "Return the quota using the 'handler' and 'args'."

    return args and args[0] or handler.user

# Registry of scheduling functions.

scheduling_functions = {
    "check_quota" : [check_quota],
    "schedule_across_quota" : [schedule_across_quota],
    "schedule_for_delegate" : [schedule_for_delegate],
    }

# Registries of locking and unlocking functions.

locking_functions = {
    "check_quota" : [lock_journal],
    "schedule_across_quota" : [lock_journal],
    "schedule_for_delegate" : [lock_journal],
    }

unlocking_functions = {
    "check_quota" : [unlock_journal],
    "schedule_across_quota" : [unlock_journal],
    "schedule_for_delegate" : [unlock_journal],
    }

# Registries of listener functions.

confirmation_functions = {
    "check_quota" : [add_to_quota, update_event],
    "schedule_across_quota" : [add_to_quota_freebusy, update_event],
    "schedule_for_delegate" : [add_to_quota, update_event],
    }

retraction_functions = {
    "check_quota" : [remove_from_quota, remove_event],
    "schedule_across_quota" : [remove_from_quota_freebusy, remove_event],
    "schedule_for_delegate" : [remove_from_quota, remove_event],
    }

# vim: tabstop=4 expandtab shiftwidth=4
