/*
 *   Copyright (C) 2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "FFTPhase.h"

#include "common/Exception.h"
#include "common/Hamming.h"
#include "common/Inline.h"
#include "common/SFFT.h"
#include "common/NCO.h"

#include "fsk441/FSK441Defs.h"

int main(int argc, char *argv[])
{
	if (argc < 2) {
		::fprintf(stderr, "Usage: FFTPhase <freq offset>\n");
		return 1;
	}

	double  offset = ::atof(argv[1]);

	try {
		CFFTPhase phase(offset);
		phase.run();
	}
	catch (CException& ex) {
		::fprintf(stderr, "FFTPhase: error: %s\n", ex.getMessage().c_str());
		return 1;
	}
	catch (...) {
		::fprintf(stderr, "FFTPhase: an error has occured\n");
		return 1;
	}

	return 0;
}

CFFTPhase::CFFTPhase(double offset) :
m_offset(offset)
{
}

CFFTPhase::~CFFTPhase()
{
}

void CFFTPhase::run()
{
	double f = double(FSK441_SAMPLE_RATE) / double(FSK441_FFT_LENGTH);

	CNCO nco(FSK441_SAMPLE_RATE, FSK441_SEND_VOLUME);
	CSFFT sfft(FSK441_FFT_LENGTH, FSK441_BIN0, FSK441_BIN3 + 1);

	complex<double> bins[FSK441_FFT_LENGTH];

	nco.setPhase(1.67546);

	double storePhase[FSK441_BIN3 + 1];
	storePhase[FSK441_BIN0] = 0.0;
	storePhase[FSK441_BIN1] = 0.0;
	storePhase[FSK441_BIN2] = 0.0;
	storePhase[FSK441_BIN3] = 0.0;

	for (int n = 0; n < 125; n++) {
		int l = FSK441_BIN2;
		double sample = 0.0;
/*
		if (n < 25) {
			l = 0;
		} else if (n < 50) {
			l = FSK441_BIN2;
		} else if (n < 75) {
			l = FSK441_BIN1;
		} else if (n < 100) {
			l = FSK441_BIN3;
		} else {
			l = FSK441_BIN0;
		}
*/
		double frequency = m_offset + double(l) * f;

		nco.generate(frequency, &sample, 1);

		sfft.process(sample, bins);

		if (l != 0 && n >= 25) {
			::fprintf(stderr, "%03d %d:  ", n, l - FSK441_BIN0);

			for (int i = FSK441_BIN0; i <= FSK441_BIN3; i++) {
				double ampl  = ::CABS(bins[i]);
				double phase = ::CPHASE(bins[i]);

				double diff = phase - storePhase[i];

				if (diff < 0.0)
					diff += 2.0 * M_PI;

				::fprintf(stderr, "%.1f %.1f %.1f\t",
					ampl, phase, diff);

				storePhase[i] = phase;
			}

			::fprintf(stderr, "\n");
		}
	}
}
