/*
 *   Copyright (C) 2002-2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "FSK441Receive.h"
#include "FSK441App.h"

#include <cmath>
using namespace std;

#include <wx/datetime.h>
#include <wx/debug.h>
#include <wx/log.h>

const int BUFFER_OFFSET = (FSK441_FFT_LENGTH - FSK441_SYMBOL_LENGTH) / 2;

enum {
	FSK441_SEEKING,
	FSK441_LOCKED
};

CFSK441Receive::CFSK441Receive(const wxString& name, EWho who) :
CReceive(name, who),
m_id(wxEmptyString),
m_audioSamples(NULL),
m_samplesCount(0),
m_fft(FSK441_FFT_LENGTH),
m_DF(0),
m_noise(),
m_burstData(),
m_minLength(0),
m_minStrength(0),
m_maxOffset(0)
{
	m_audioSamples = new double[FSK441_MAX_AUDIO_DATA];

	m_noise.addValue(0.5);
}

CFSK441Receive::~CFSK441Receive()
{
	delete[] m_audioSamples;
}

void CFSK441Receive::reset(bool firstTime)
{
	m_samplesCount = 0;
	m_DF = 0;

	m_noise.clear();
	m_noise.addValue(0.5);

	m_burstData.clear();
}

void CFSK441Receive::run()
{
	m_id = createId();

	::wxGetApp().getProtoSettings(m_minLength, m_minStrength, m_maxOffset);

	bool end = false;

	double audio[FSK441_SOUNDBUF_LENGTH];
	double in[FSK441_FFT_LENGTH];
	double bins[FSK441_FFT_LENGTH];

	int state = FSK441_SEEKING;

	openSoundDevice();

	while (!isStopped() && !end && m_samplesCount < (FSK441_MAX_AUDIO_DATA - FSK441_SOUNDBUF_LENGTH)) {
		int len = FSK441_SOUNDBUF_LENGTH;
		getSoundDevice()->read(audio, len);

		if (len <= 0) break;

		::wxGetApp().showAudio(audio[100], getWho());

		for (int i = 0; i < len; i++) {
			m_audioSamples[m_samplesCount++] = audio[i];

			if (m_samplesCount >= FSK441_SYMBOL_LENGTH && (m_samplesCount % 5) == 0) {
				for (int j = 0; j < FSK441_FFT_LENGTH; j++)
					in[j] = 0.0;

				for (int j = 0; j < FSK441_SYMBOL_LENGTH; j++)
					in[j + BUFFER_OFFSET] = m_audioSamples[m_samplesCount - FSK441_SYMBOL_LENGTH + j];

				m_fft.process(in, bins);

				switch (state) {
					case FSK441_SEEKING:
						state = seekBurst(bins);
						break;
					case FSK441_LOCKED:
						state = storeBurst(bins);
						break;
					default:
						::wxLogError(wxT("Unknown state in FSK441Receive"));
						state = FSK441_SEEKING;
						break;
				}
			}
		}

		end = getEndTime();
	}

	closeSoundDevice();

	// End of the data and still receiving a burst, display the collected data
	if (state == FSK441_LOCKED)
		decodeBurst();

	recordAudio(m_id, m_audioSamples, m_samplesCount);

	getLevels();
}

int CFSK441Receive::seekBurst(double* bins)
{
	wxASSERT(bins != NULL);

	int maxBins = m_maxOffset / FSK441_BIN_WIDTH;

	for (int i = FSK441_TONE0_BIN - maxBins; i <= FSK441_TONE3_BIN + maxBins; i++)
		m_noise.addValue(bins[i]);

	double max = -999.0;
	m_DF       = -999;

	for (int i = 0; i < 4; i++) {
		int midBin = FSK441_TONE0_BIN + i * FSK441_TONE_SPACING;

		int botBin = midBin - maxBins;
		int topBin = midBin + maxBins;

		for (int j = botBin; j <= topBin; j++) {
			if (bins[j] > max) {
				max  = bins[j];
				m_DF = j - midBin;
			}
		}
	}

	double tone0 = bins[FSK441_TONE0_BIN + m_DF];
	double tone1 = bins[FSK441_TONE1_BIN + m_DF];
	double tone2 = bins[FSK441_TONE2_BIN + m_DF];
	double tone3 = bins[FSK441_TONE3_BIN + m_DF];

	m_noise.subtractValue(tone0);
	m_noise.subtractValue(tone1);
	m_noise.subtractValue(tone2);
	m_noise.subtractValue(tone3);

	if ((max / m_noise.getAverage()) >= FSK441_MIN_RATIO) {
		m_burstData.setStartBurst();
		m_burstData.setDF(m_DF);

		m_burstData.setData(m_samplesCount, tone0, tone1, tone2, tone3);

		return FSK441_LOCKED;
	}

	return FSK441_SEEKING;
}

int CFSK441Receive::storeBurst(double* bins)
{
	wxASSERT(bins != NULL);

	int maxBins = m_maxOffset / FSK441_BIN_WIDTH;

	int botTone0Bin = FSK441_TONE0_BIN - maxBins;
	int topTone3Bin = FSK441_TONE3_BIN + maxBins;

	for (int i = botTone0Bin; i <= topTone3Bin; i++)
		m_noise.addValue(bins[i]);

	double tone0 = bins[FSK441_TONE0_BIN + m_DF];
	double tone1 = bins[FSK441_TONE1_BIN + m_DF];
	double tone2 = bins[FSK441_TONE2_BIN + m_DF];
	double tone3 = bins[FSK441_TONE3_BIN + m_DF];

	double max = tone0;
	if (tone1 > max) max = tone1;
	if (tone2 > max) max = tone2;
	if (tone3 > max) max = tone3;

	m_noise.subtractValue(tone0);
	m_noise.subtractValue(tone1);
	m_noise.subtractValue(tone2);
	m_noise.subtractValue(tone3);

	if ((max / m_noise.getAverage()) >= FSK441_MIN_RATIO) {
		m_burstData.setData(m_samplesCount, tone0, tone1, tone2, tone3);
		return FSK441_LOCKED;
	}

	decodeBurst();

	return FSK441_SEEKING;
}

void CFSK441Receive::decodeBurst()
{
	int length   = 1000 * m_burstData.getLength() / FSK441_SAMPLE_RATE;
	int strength = int(m_burstData.getStrength(m_noise.getAverage()) + 0.5);
	int offset   = m_burstData.getDF() * FSK441_BIN_WIDTH;

	if (length < m_minLength || strength < m_minStrength)
		return;

	bool ret = m_burstData.processBurst();

	if (!ret)
		return;

	wxString    text = m_burstData.getMessage();
	double startTime = double(m_burstData.getStartTime()) / double(FSK441_SAMPLE_RATE);

	CFSK441Message* message = new CFSK441Message(m_id, startTime, length, strength, offset, text);

	::wxGetApp().receiveMessage(message, getWho());
}

void CFSK441Receive::getLevels() const
{
	CFSK441Levels* levels = new CFSK441Levels();

	levels->setAudioData(m_audioSamples, m_samplesCount);

	::wxGetApp().showLevels(levels, getWho());
}
