/*
 *   Copyright (C) 2002-2004 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "SoundFile.h"
#include "Exception.h"

#include <wx/debug.h>
#include <wx/log.h>

#ifdef __WINDOWS__

typedef unsigned char uint8;
typedef signed short  sint16;

CSoundFile::CSoundFile(const wxString& fileName, int sampleRate, int sampleWidth) :
m_fileName(fileName),
m_sampleRate(sampleRate),
m_sampleWidth(sampleWidth),
m_direction(SoundFile_Read),
m_handle(NULL),
m_parent(),
m_child()
{
	wxASSERT(sampleRate > 0);
	wxASSERT(sampleWidth == 8 || sampleWidth == 16);
}

CSoundFile::~CSoundFile()
{
}

void CSoundFile::openRead()
{
	wxASSERT(m_handle == NULL);

	m_handle = ::mmioOpen((CHAR *)m_fileName.c_str(), 0, MMIO_READ | MMIO_ALLOCBUF);

	if (m_handle == NULL)
		throw CException(wxT("Cannot open the WAV file for reading"));

	MMCKINFO parent;
	parent.fccType = mmioFOURCC('W', 'A', 'V', 'E');

	MMRESULT res = ::mmioDescend(m_handle, &parent, 0, MMIO_FINDRIFF);

	if (res != MMSYSERR_NOERROR)
		throw CException(wxT("File is not a valid WAV file"));

	MMCKINFO child;
	child.ckid = mmioFOURCC('f', 'm', 't', ' ');

	res = ::mmioDescend(m_handle, &child, &parent, MMIO_FINDCHUNK);

	if (res != MMSYSERR_NOERROR)
		throw CException(wxT("File is not a valid WAV file"));

	WAVEFORMATEX format;

	LONG len = ::mmioRead(m_handle, (char *)&format, child.cksize);

	if (len != LONG(child.cksize))
		throw CException(wxT("File is not a valid WAV file"));

	if (format.wFormatTag != WAVE_FORMAT_PCM)
		throw CException(wxT("File is not a standard WAV file"));

	if (int(format.nSamplesPerSec) != m_sampleRate) {
		wxString text;
		text.Printf(wxT("Invalid sample rate in the WAV file, found %d samples/s"), int(format.nSamplesPerSec));
		throw CException(text);
	}

	if (format.nChannels != 1)
		throw CException(wxT("More than one channel in the WAV file"));

	m_sampleWidth = format.wBitsPerSample;

	if (m_sampleWidth != 8 && m_sampleWidth != 16)
		throw CException(wxT("None standard data format in the WAV file"));

	res = ::mmioAscend(m_handle, &child, 0);

	if (res != MMSYSERR_NOERROR)
		throw CException(wxT("File is not a valid WAV file"));

	child.ckid = mmioFOURCC('d', 'a', 't', 'a');

	res = ::mmioDescend(m_handle, &child, &parent, MMIO_FINDCHUNK);

	if (res != MMSYSERR_NOERROR)
		throw CException(wxT("File is not a valid WAV file"));

	m_direction = SoundFile_Read;
}

void CSoundFile::openWrite()
{
	wxASSERT(m_handle == NULL);

	m_handle = ::mmioOpen((CHAR *)m_fileName.c_str(), 0, MMIO_WRITE | MMIO_CREATE | MMIO_ALLOCBUF);

	if (m_handle == NULL)
		throw CException(wxT("Cannot open the WAV file for writing"));

	m_parent.fccType = mmioFOURCC('W', 'A', 'V', 'E');
	m_parent.cksize  = 0;

	MMRESULT res = ::mmioCreateChunk(m_handle, &m_parent, MMIO_CREATERIFF);

	if (res != MMSYSERR_NOERROR)
		throw CException(wxT("Cannot write to the WAV file"));

	m_child.ckid   = mmioFOURCC('f', 'm', 't', ' ');
	m_child.cksize = sizeof(WAVEFORMATEX);

	res = ::mmioCreateChunk(m_handle, &m_child, 0);

	if (res != MMSYSERR_NOERROR)
		throw CException(wxT("Cannot write to the WAV file"));

	WAVEFORMATEX format;
	format.wBitsPerSample  = m_sampleWidth;
	format.wFormatTag      = WAVE_FORMAT_PCM;
	format.nChannels       = 1;
	format.nSamplesPerSec  = m_sampleRate;
	format.nAvgBytesPerSec = m_sampleRate * 1 * m_sampleWidth / 8;
	format.nBlockAlign     = 1 * m_sampleWidth / 8;
	format.cbSize          = 0;

	LONG n = ::mmioWrite(m_handle, (CHAR *)&format, sizeof(WAVEFORMATEX));

	if (n != sizeof(WAVEFORMATEX))
		throw CException(wxT("Cannot write to the WAV file"));

	::mmioAscend(m_handle, &m_child, 0);

	m_child.ckid   = mmioFOURCC('d', 'a', 't', 'a');
	m_child.cksize = 0;

	res = ::mmioCreateChunk(m_handle, &m_child, 0);

	if (res != MMSYSERR_NOERROR)
		throw CException(wxT("Cannot write to the WAV file"));

	m_direction = SoundFile_Write;
}

void CSoundFile::read(double* sample, int& len)
{
	wxASSERT(m_handle != NULL);
	wxASSERT(sample != NULL);
	wxASSERT(len > 0);

	if (m_sampleWidth == 8) {
		uint8* data = new uint8[len];

		LONG n = ::mmioRead(m_handle, (char *)data, len);

		if (n <= 0) {
			delete[] data;
			len = n;
			return;
		}

		for (int i = 0; i < n; i++)
			sample[i] = (double(data[i]) - 127.0) / 128.0;

		delete[] data;

		len = n;
	} else {
		sint16* data = new sint16[len];

		LONG n = ::mmioRead(m_handle, (char *)data, len * 2);

		if (n <= 0) {
			delete[] data;
			len = n;
			return;
		}

		n /= 2;

		for (int i = 0; i < n; i++)
			sample[i] = double(data[i]) / 32768.0;

		delete[] data;

		len = n;
	}
}

void CSoundFile::write(double* sample, int len)
{
	wxASSERT(m_handle != NULL);
	wxASSERT(sample != NULL);
	wxASSERT(len > 0);

	if (m_sampleWidth == 8) {
		uint8* data = new uint8[len];

		for (int i = 0; i < len; i++)
			data[i] = uint8(sample[i] * 128.0 + 127.0);

		LONG n = ::mmioWrite(m_handle, (char *)data, len * sizeof(uint8));

		delete[] data;

		if (n != LONG(len * sizeof(uint8)))
			throw CException(wxT("Cannot write to the WAV file"));
	} else {
		sint16* data = new sint16[len];

		for (int i = 0; i < len; i++)
			data[i] = sint16(sample[i] * 32768.0);

		LONG n = ::mmioWrite(m_handle, (char *)data, len * sizeof(sint16));

		delete[] data;

		if (n != LONG(len * sizeof(sint16)))
			throw CException(wxT("Cannot write to the WAV file"));
	}
}

void CSoundFile::close()
{
	wxASSERT(m_handle != NULL);

	if (m_direction == SoundFile_Write) {
		::mmioAscend(m_handle, &m_child, 0);
		::mmioAscend(m_handle, &m_parent, 0);
	}

	::mmioClose(m_handle, 0);
	m_handle = NULL;
}

#else

CSoundFile::CSoundFile(const wxString& fileName, int sampleRate, int sampleWidth) :
m_fileName(fileName),
m_sampleRate(sampleRate),
m_sampleWidth(sampleWidth),
m_direction(SoundFile_Read),
m_file(NULL)
{
	wxASSERT(sampleRate > 0);
	wxASSERT(sampleWidth == 8 || sampleWidth == 16);
}

CSoundFile::~CSoundFile()
{
}

void CSoundFile::openRead()
{
	wxASSERT(m_file == NULL);

	SF_INFO info;
	info.format = 0;
	m_file = ::sf_open(m_fileName.mb_str(), SFM_READ, &info);

	if (m_file == NULL)
		throw CException(wxT("Cannot open the WAV file for reading"));

	if (info.samplerate != m_sampleRate) {
		wxString text;
		text.Printf(wxT("Invalid sample rate in the WAV file, found %d samples/s"), info.samplerate);
		throw CException(text);
	}

	if (info.channels != 1)
		throw CException(wxT("More than one channel in the WAV file"));

	m_direction = SoundFile_Read;
}

void CSoundFile::openWrite()
{
	wxASSERT(m_file == NULL);

	SF_INFO info;
	info.samplerate = m_sampleRate;
	info.channels   = 1;
	info.format     = SF_FORMAT_WAV;

	if (m_sampleWidth == 8)
		info.format |= SF_FORMAT_PCM_U8;
	else
		info.format |= SF_FORMAT_PCM_16;

	int ret = ::sf_format_check(&info);

	if (!ret)
		throw CException(wxT("Mistake in setting up SF_INFO for write"));

	m_file = ::sf_open(m_fileName.mb_str(), SFM_WRITE, &info);

	if (m_file == NULL)
		throw CException(wxT("Cannot open the WAV file for writing"));

	m_direction = SoundFile_Write;
}

void CSoundFile::read(double* sample, int& len)
{
	wxASSERT(m_file != NULL);
	wxASSERT(sample != NULL);
	wxASSERT(len > 0);

	len = ::sf_read_double(m_file, sample, len);
}

void CSoundFile::write(double* sample, int len)
{
	wxASSERT(m_file != NULL);
	wxASSERT(sample != NULL);
	wxASSERT(len > 0);

	int n = ::sf_write_double(m_file, sample, len);

	if (n != len)
		throw CException(wxT("Error writing to the sound file"));
}

void CSoundFile::close()
{
	wxASSERT(m_file != NULL);

	::sf_close(m_file);
	m_file = NULL;
}

#endif
