/* fileentry.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001-2008 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "fileentry.hh"
#include <aguix/lowlevelfunc.h>
#include "wconfig.h"
#include "grouphash.h"
#include "pdatei.h"
#include "nmrowdata.h"
#include "condparser.h"
#include "stringbuf.h"
#include "wcfiletype.hh"
#include "magic_db.hh"

#ifndef MAXPATHLEN
#define MAXPATHLEN 2048
#endif

FileEntry::FileEntry()
{
  name=NULL;
  fullname=NULL;
  filetype=NULL;
  use=true;
  isHidden=false;
  
  reclistQueued = false;
  
  memset( &statbuf, 0, sizeof( statbuf ) );
  memset( &dstatbuf, 0, sizeof( dstatbuf ) );
  
  contentBuf.content = NULL;
  contentBuf.size = 0;
  
  outputBuf = NULL;

  customcolors = false;
  fg[0] = 1;
  fg[1] = 1;
  fg[2] = 1;
  fg[3] = 1;
  bg[0] = 0;
  bg[1] = 0;
  bg[2] = 0;
  bg[3] = 0;

  _ID = -1;
}

FileEntry::~FileEntry()
{
  if(name!=NULL) _freesafe(name);
  if(fullname!=NULL) _freesafe(fullname);
  freeContentBuf();
  freeOutputBuf();
}

bool FileEntry::isDir() const
{
  bool dir=false;
  if ( isLink == true ) {
    if ( isCorrupt == false ) {
      if ( S_ISDIR( dstatbuf.mode ) ) dir = true;
    }
  } else if ( S_ISDIR( statbuf.mode ) ) dir = true;
  return dir;
}

FileEntry *FileEntry::duplicate()
{
#if 0
  FileEntry *tfe=new FileEntry();
  tfe->fullname=dupstring(fullname);
  tfe->name=dupstring(name);
  tfe->dirsize=dirsize;
  tfe->select=select;
  tfe->isLink=isLink;
  tfe->isCorrupt=isCorrupt;
  tfe->inFilter=inFilter;
  tfe->filetype=filetype;

  memcpy( &( tfe->statbuf ), &statbuf, sizeof( statbuf ) );
  memcpy( &( tfe->dstatbuf ), &dstatbuf, sizeof( dstatbuf ) );

  tfe->setCustomColors( customcolors );
  tfe->setColor( 0, fg );
  tfe->setColor( 1, bg );

  return tfe;
#endif
  return new FileEntry( *this );
}

int FileEntry::readInfos()
{
  return readInfos(false);
}

int FileEntry::readInfos(bool update)
{
  //TODO: auf update achten
  worker_struct_stat stbuf;

  select=false;
  dirsize=-1;
  filetype=NULL;

  if(fullname==NULL) return 1;

  memset( &statbuf, 0, sizeof( statbuf ) );
  memset( &dstatbuf, 0, sizeof( dstatbuf ) );

  if ( worker_lstat( fullname, &stbuf ) == 0 ) {
    statbuf.size = stbuf.st_size;
    statbuf.lastaccess = stbuf.st_atime;
    statbuf.lastmod = stbuf.st_mtime;
    statbuf.lastchange = stbuf.st_ctime;
    statbuf.mode = stbuf.st_mode;
    statbuf.userid = stbuf.st_uid;
    statbuf.groupid = stbuf.st_gid;
    statbuf.inode = stbuf.st_ino;
    statbuf.nlink = stbuf.st_nlink;
    statbuf.blocks = stbuf.st_blocks;
    statbuf.rdev = stbuf.st_rdev;
    statbuf.dev = stbuf.st_dev;
  } else return 2;
  isCorrupt=false;
  if ( S_ISLNK( statbuf.mode ) ) {
    isLink=true;
    if ( worker_stat( fullname, &stbuf ) == 0 ) {
      dstatbuf.size = stbuf.st_size;
      dstatbuf.lastaccess = stbuf.st_atime;
      dstatbuf.lastmod = stbuf.st_mtime;
      dstatbuf.lastchange = stbuf.st_ctime;
      dstatbuf.mode = stbuf.st_mode;
      dstatbuf.userid = stbuf.st_uid;
      dstatbuf.groupid = stbuf.st_gid;
      dstatbuf.inode = stbuf.st_ino;
      dstatbuf.nlink = stbuf.st_nlink;
      dstatbuf.blocks = stbuf.st_blocks;
      dstatbuf.rdev = stbuf.st_rdev;
      dstatbuf.dev = stbuf.st_dev;
    } else {
      /* korrupt */
      isCorrupt=true;
      dstatbuf.mode = 0;  // to prevent corrupt links treated as dirs
      if(errno==ENOENT) {
        // Eventl. fuer spaetere Betrachtungen
      }
    }
  } else {
    isLink=false;
  }
  return 0;
}

char *FileEntry::getDestination() const
{
  char *str=NULL,*tstr;
  if((isLink==true)&&(fullname!=NULL)) {
    tstr=(char*)_allocsafe((MAXPATHLEN+1)*sizeof(char));
    int x = worker_readlink( fullname, tstr, MAXPATHLEN );
    if(x>=0) {
      tstr[x]=0;
      str=dupstring(tstr);
    }
    _freesafe(tstr);
  }
  return str;
}

int FileEntry::writeDestination2Buf(char *buf,int size2,bool terminated)
{
  int x=-1;
  if((isLink==true)&&(fullname!=NULL)) {
    x = worker_readlink( fullname, buf, size2 );
    if((terminated==true)&&(x>=0)) {
      buf[x]=0;
    }
  }
  return x;
}

void FileEntry::setLVC( FieldListView *lv, int row, const std::vector<WorkerTypes::listcol_t> *dis )
{
  int mlen=0;
  char *str=NULL;
  struct tm *timeptr;
  loff_t dissize, li;
  bool sfds=false;
  std::string s1;
  int ost, x;
  char nbuf[ A_BYTESFORNUMBER( loff_t ) ];
  int cur_field;
  
  if ( ( lv == NULL ) || ( dis == NULL ) ) return;
  
  cur_field = 0;
  for ( int i = 0; i < (int)dis->size(); i++ ) {
    if ( i > 0 ) cur_field++;
    
    switch( (*dis)[i]) {
      case WorkerTypes::LISTCOL_NAME:
        s1 = "";
        if ( isLink == true ) {
          if ( isCorrupt == true ) s1 += '!';
          else {
            if ( S_ISDIR( dstatbuf.mode ) ) s1 += '~';
            else s1 += '@';
          }
        } else {
          if ( S_ISDIR( statbuf.mode ) ) s1 += '/';
          else if ( S_ISFIFO( statbuf.mode ) ) s1 += '|';
          else if ( S_ISSOCK( statbuf.mode ) ) s1 += '=';
          else if ( S_ISCHR( statbuf.mode ) ) s1 += '-';
          else if ( S_ISBLK( statbuf.mode ) ) s1 += '+';
          else if ( isExe() == true ) s1 += '*';
          else s1 += ' ';
        }
	s1 += name;
        
        // I only set update_set for name to avoid the overhead for
        // every field as the name is probably always visible
        lv->setText( row, cur_field, s1, true );
        break;
      case WorkerTypes::LISTCOL_SIZE:
        if ( S_ISCHR( statbuf.mode ) ||
             S_ISBLK( statbuf.mode ) ) {
          x = (int)statbuf.rdev;
          snprintf( nbuf, sizeof( nbuf ), "%d", x >> 8 );
          s1 = nbuf;
          s1 += ", ";
          snprintf( nbuf, sizeof( nbuf ), "%3d", x & 0xff );
          s1 += nbuf;
          lv->setText( row, cur_field, s1 );
        } else {
          if ( ( isDir() == true ) &&
               ( isLink == false ) &&
               ( dirsize >= 0 ) ) dissize = dirsize;
          else dissize = statbuf.size;
  
          if ( isDir() == true ) {
            if ( ! ( ( isLink == false ) && ( dirsize >= 0 ) ) ) {
              if(wconfig->getShowStringForDirSize()==true)
                sfds=true;
            }
          }
          if(sfds==true) {
            s1 = wconfig->getStringForDirSize();
            lv->setText( row, cur_field, s1 );
          } else {
	    MakeLong2NiceStr( dissize, s1 );
            lv->setText( row, cur_field, s1 );
          }
        }
        break;
      case WorkerTypes::LISTCOL_TYPE:
        if ( filetype != NULL ) {
          s1 = filetype->getName();
        } else {
          if ( isDir() == true ) {
            // Dir
            s1 = wconfig->getdirtype()->getName();
          } else {
            s1 = wconfig->getnotyettype()->getName();
          }
        }
        lv->setText( row, cur_field, s1 );
        break;
      case WorkerTypes::LISTCOL_PERM:
        s1 = getPermissionString();
        lv->setText( row, cur_field, s1 );
        break;
      case WorkerTypes::LISTCOL_OWNER:
        // will be 1 for Owner.group, 0 else
        ost=wconfig->getOwnerstringtype();

        s1 = ugdb->getUserNameS( statbuf.userid );
        if(ost==1) {
          s1 += '.';
        } else {
          s1 += " @ ";
        }
        s1 += ugdb->getGroupNameS( statbuf.groupid );
        lv->setText( row, cur_field, s1 );
        break;
      case WorkerTypes::LISTCOL_DEST:
        str = getDestination();
        if ( str != NULL ) {
          // only if there will be something to print
          lv->setText( row, cur_field, str );
          _freesafe( str );
        } else {
          lv->setText( row, cur_field, "" );
        }
        break;
      case WorkerTypes::LISTCOL_MOD:
        timeptr = localtime( &( statbuf.lastmod ) );
        mlen = wconfig->getDatelen( timeptr );
        str = (char*)_allocsafe( mlen + 2 );
        memset( str, ' ', mlen + 1 );
        x = wconfig->writeDateToString( str , mlen + 1, timeptr );
        str[ x ] = '\0';
        s1 = str;
        _freesafe( str );
        lv->setText( row, cur_field, s1 );
        break;
      case WorkerTypes::LISTCOL_ACC:
        timeptr = localtime( &( statbuf.lastaccess ) );
        mlen = wconfig->getDatelen( timeptr );
        str = (char*)_allocsafe( mlen + 2 );
        memset( str, ' ', mlen + 1 );
        x = wconfig->writeDateToString( str , mlen + 1, timeptr );
        str[ x ] = '\0';
        s1 = str;
        _freesafe( str );
        lv->setText( row, cur_field, s1 );
        break;
      case WorkerTypes::LISTCOL_CHANGE:
        timeptr = localtime( &( statbuf.lastchange ) );
        mlen = wconfig->getDatelen( timeptr );
        str = (char*)_allocsafe( mlen + 2 );
        memset( str, ' ', mlen + 1 );
        x = wconfig->writeDateToString( str , mlen + 1, timeptr );
        str[ x ] = '\0';
        s1 = str;
        _freesafe( str );
        lv->setText( row, cur_field, s1 );
        break;
      case WorkerTypes::LISTCOL_INODE:
        li = (loff_t)statbuf.inode;
	MakeLong2NiceStr( li, s1, false );
        lv->setText( row, cur_field, s1 );
        break;
      case WorkerTypes::LISTCOL_NLINK:
        li = (loff_t)statbuf.nlink;
	MakeLong2NiceStr( li, s1, false );
        lv->setText( row, cur_field, s1 );
        break;
      case WorkerTypes::LISTCOL_BLOCKS:
        li = (loff_t)statbuf.blocks;
	MakeLong2NiceStr( li, s1, false );
        lv->setText( row, cur_field, s1 );
        break;
      default:
        break;
    }
    cur_field++;
  }
  
  if ( customcolors == true ) {
    lv->setFG( row, FieldListView::CC_NORMAL, fg[0] );
    lv->setBG( row, FieldListView::CC_NORMAL, bg[0] );
    lv->setFG( row, FieldListView::CC_SELECT, fg[1] );
    lv->setBG( row, FieldListView::CC_SELECT, bg[1] );
    lv->setFG( row, FieldListView::CC_ACTIVE, fg[2] );
    lv->setBG( row, FieldListView::CC_ACTIVE, bg[2] );
    lv->setFG( row, FieldListView::CC_SELACT, fg[3] );
    lv->setBG( row, FieldListView::CC_SELACT, bg[3] );
  } else {
    if ( ( isDir() == true ) && ( isCorrupt == false ) ) {
      lv->setFG( row, FieldListView::CC_NORMAL, wconfig->getUnselDir( 0 ) );
      lv->setBG( row, FieldListView::CC_NORMAL, wconfig->getUnselDir( 1 ) );
      lv->setFG( row, FieldListView::CC_SELECT, wconfig->getSelDir( 0 ) );
      lv->setBG( row, FieldListView::CC_SELECT, wconfig->getSelDir( 1 ) );
      lv->setFG( row, FieldListView::CC_ACTIVE, wconfig->getUnselDirAct( 0 ) );
      lv->setBG( row, FieldListView::CC_ACTIVE, wconfig->getUnselDirAct( 1 ) );
      lv->setFG( row, FieldListView::CC_SELACT, wconfig->getSelDirAct( 0 ) );
      lv->setBG( row, FieldListView::CC_SELACT, wconfig->getSelDirAct( 1 ) );
    } else {
      lv->setFG( row, FieldListView::CC_NORMAL, wconfig->getUnselFile( 0 ) );
      lv->setBG( row, FieldListView::CC_NORMAL, wconfig->getUnselFile( 1 ) );
      lv->setFG( row, FieldListView::CC_SELECT, wconfig->getSelFile( 0 ) );
      lv->setBG( row, FieldListView::CC_SELECT, wconfig->getSelFile( 1 ) );
      lv->setFG( row, FieldListView::CC_ACTIVE, wconfig->getUnselFileAct( 0 ) );
      lv->setBG( row, FieldListView::CC_ACTIVE, wconfig->getUnselFileAct( 1 ) );
      lv->setFG( row, FieldListView::CC_SELACT, wconfig->getSelFileAct( 0 ) );
      lv->setBG( row, FieldListView::CC_SELACT, wconfig->getSelFileAct( 1 ) );
    }
  }
  if ( isCorrupt == true ) {
    lv->setFG( row, FieldListView::CC_NORMAL, 4 );
    lv->setFG( row, FieldListView::CC_ACTIVE, 4 );
  }

}

const char *FileEntry::getPermissionString()
{
  static char str[WORKER_PERMCHARS + 1];

  if ( isLink == true ) str[0] = 'l';
  else {
    if ( S_ISDIR( statbuf.mode ) ) str[0] = 'd';
    else if ( S_ISFIFO( statbuf.mode ) ) str[0] = 'p';
    else if ( S_ISSOCK( statbuf.mode ) ) str[0] = 's';
    else if ( S_ISCHR( statbuf.mode ) ) str[0] = 'c';
    else if ( S_ISBLK( statbuf.mode ) ) str[0] = 'b';
    else str[0] = '-';
  }
  if ( ( statbuf.mode & S_IRUSR ) == S_IRUSR ) str[1] = 'r'; else str[1] = '-';
  if ( ( statbuf.mode & S_IWUSR ) == S_IWUSR ) str[2] = 'w'; else str[2] = '-';
  if ( ( statbuf.mode & S_ISUID ) == S_ISUID ) {
    if ( ( statbuf.mode & S_IXUSR ) == S_IXUSR ) str[3] = 's'; else str[3] = 'S';
  } else {
    if ( ( statbuf.mode & S_IXUSR ) == S_IXUSR ) str[3] = 'x'; else str[3] = '-';
  }
  if ( ( statbuf.mode & S_IRGRP ) == S_IRGRP ) str[4] = 'r'; else str[4] = '-';
  if ( ( statbuf.mode & S_IWGRP ) == S_IWGRP ) str[5] = 'w'; else str[5] = '-';
  if ( ( statbuf.mode & S_ISGID ) == S_ISGID ) {
    if ( ( statbuf.mode & S_IXGRP ) == S_IXGRP ) str[6] = 's'; else str[6] = 'S';
  } else {
    if ( ( statbuf.mode & S_IXGRP ) == S_IXGRP ) str[6] = 'x'; else str[6] = '-';
  }
  if ( ( statbuf.mode & S_IROTH ) == S_IROTH ) str[7] = 'r'; else str[7] = '-';
  if ( ( statbuf.mode & S_IWOTH ) == S_IWOTH ) str[8] = 'w'; else str[8] = '-';
  if ( ( statbuf.mode & S_ISVTX ) == S_ISVTX ) {
    if ( ( statbuf.mode & S_IXOTH ) == S_IXOTH ) str[9] = 't'; else str[9] = 'T';
  } else {
    if ( ( statbuf.mode & S_IXOTH ) == S_IXOTH ) str[9] = 'x'; else str[9] = '-';
  }
  str[10]=0;
  return str;
}

bool FileEntry::isExe( bool fastTest ) const
{
  if ( fastTest == true ) {
    if ( ( statbuf.mode & S_IXUSR ) == S_IXUSR ) return true;
    if ( ( statbuf.mode & S_IXGRP ) == S_IXGRP ) return true;
    if ( ( statbuf.mode & S_IXOTH ) == S_IXOTH ) return true;
  } else {
    uid_t uid;
    gid_t gid;
    mode_t m;
    bool e = false;

    if ( isLink == false ) {
      uid = userid();
      gid = groupid();
      m = mode();
    } else if ( isCorrupt == false ) {
      uid = duserid();
      gid = dgroupid();
      m = dmode();
    } else {
      // corrupt symlink is not executable
      return false;
    }
    
    if ( uid == geteuid() ) {
      e = ( ( m & S_IXUSR ) != 0 ) ? true : false;
    } else if ( ugdb->isInGroup( gid ) == true ) {
      e = ( ( m & S_IXGRP ) != 0 ) ? true : false;
    } else {
      e = ( ( m & S_IXOTH ) != 0 ) ? true : false;
    }
    return e;
  }
  return false;
}

FileEntry::typecandidate_t FileEntry::runTests( WCFiletype *type,
						bool skipcontent,
						CondParser *condparser,
						unsigned char fbuffer[64],
						int reads,
                                                typecheck_temp_results_t &temp_results )
{
  typecandidate_t erg;
  int minsize, x, y;
  char *tstr1;
  
  if ( type == NULL ) return erg;
  if ( condparser == NULL ) return erg;
  
  if ( skipcontent == false ) {
    short *filedesc;
    if ( type->getinternID() == NORMALTYPE ) {
      if ( type->getUseFiledesc() == true ) {
	erg.tests_done++;
	if ( reads > 0 ) {
	  filedesc = type->getFiledesc();
	  minsize = 0;
	  for ( x = 0; x < 64; x++ ) {
	    if ( filedesc[x] > -1 ) minsize = x + 1;
	  }
	  if ( ( reads >= minsize ) && ( minsize > 0 ) ) {
	    y = 0;
	    for ( x = 0; x < reads; x++ ) {
	      if ( filedesc[x] >= 0 ) {
		if ( filedesc[x] != (short)fbuffer[x] ) {
		  y = 1;
		  break;
		}
	      }
	    }
	    if ( y == 0 ) {
	      erg.prio = 3;
	      erg.type = type;
	    }
	  }
	}
      }

      if ( erg.type == NULL && type->getUseMagic() == true ) {
          erg.tests_done++;
          if ( type->getMagicCompressed() == true ) {
              if ( temp_results.magic_avail_compressed == false ) {
                  if ( worker_islocal( fullname ) == true ) {
                      temp_results.magic_result_compressed = MagicDB::getInstance().getInfo( fullname, true );
                  } else {
                      temp_results.magic_result_compressed = MagicDB::getInstance().getInfo( fbuffer, reads, true );
                  }
                  temp_results.magic_avail_compressed = true;
              }
          } else {
              if ( temp_results.magic_avail == false ) {
                  if ( worker_islocal( fullname ) == true ) {
                      temp_results.magic_result = MagicDB::getInstance().getInfo( fullname );
                  } else {
                      temp_results.magic_result = MagicDB::getInstance().getInfo( fbuffer, reads );
                  }
                  temp_results.magic_avail = true;
              }
          }

          if ( type->magicMatch( ( type->getMagicCompressed() == true ) ?
                                 temp_results.magic_result_compressed :
                                 temp_results.magic_result ) == true ) {
              erg.prio = 3;
              erg.type = type;
          }
      }
    }
  }

  /* if no type found then check extended pattern */
  if ( erg.type == NULL ) {
    if ( type->getinternID() == NORMALTYPE ) {
      if ( type->getUseExtCond() == true ) {
	erg.tests_done++;
	condparser->setIgnoreContent( skipcontent );
	if ( condparser->parse( type->getExtCond(), this ) == 1 ) {
	  erg.prio = 2;
	  erg.type = type;
	}
      }
    }
  }

  /* if no type found then check the pattern */
  if ( erg.type == NULL ) {
    if ( type->getinternID() == NORMALTYPE ) {
      if ( type->getUsePattern() == true ) {
	erg.tests_done++;
	if ( type->getPatternUseFullname() == true ) {
	  tstr1 = fullname;
	} else {
	  tstr1 = name;
	}
	if ( type->patternMatchString( tstr1 ) == true ) {
	  erg.prio = 1;
	  erg.type = type;
	}
      }
    }
  }
  return erg;
}

FileEntry::typecandidate_t FileEntry::checkFiletype( const std::list<WCFiletype*> *ftlist,
						     bool skipcontent,
						     CondParser *condparser,
						     unsigned char fbuffer[64],
						     int reads,
                                                     typecheck_temp_results_t &temp_results )
{
  WCFiletype *ft;
  typecandidate_t cand1, erg, cand2;
  const std::list<WCFiletype*> *subtypes;
  std::list<WCFiletype*>::const_iterator it1;

  if ( ftlist == NULL ) return erg;
  if ( condparser == NULL ) return erg;
  if ( fbuffer == NULL ) return erg;
  
  if ( ftlist->size() < 1 ) {
    return erg;
  }
  if ( ( S_ISDIR( statbuf.mode ) ) && ( isCorrupt == false ) ) {
    // Wirklich ein Verzeichnis
    return erg;
  }
  
  for ( it1 = ftlist->begin(); it1 != ftlist->end(); it1++ ) {
    ft = *it1;
    cand1 = runTests( ft, skipcontent, condparser, fbuffer, reads, temp_results );
    if ( ( cand1.type != NULL ) || ( cand1.tests_done < 1 ) ) {
      // check subtypes if we had a hit or did no tests
      subtypes = ft->getSubTypeList();
      if ( subtypes != NULL ) {
	if ( subtypes->size() > 0 ) {
            cand2 = checkFiletype( subtypes, skipcontent, condparser, fbuffer, reads, temp_results );
	  if ( cand2.prio > 0 ) {
	    // subtypes are always more important
	    cand1.prio = cand2.prio;
	    cand1.type = cand2.type;
	  }
	}
      }
    }
    if ( cand1.type != NULL ) {
      if ( cand1.prio > erg.prio ) {
	erg.prio = cand1.prio;
	erg.type = cand1.type;
      }
    }
  }

  return erg;
}

WCFiletype* FileEntry::checkFiletype( List *ftlist, bool skipcontent, CondParser *condparser )
{
  unsigned char fbuffer[64];
  int reads = 0;
  int id;
  WCFiletype *ft,*unknownft;
  CondParser *localparser = NULL;
  PDatei *pfp;
  typecandidate_t cand1, erg, cand2;
  const std::list<WCFiletype*> *subtypes;
  typecheck_temp_results_t temp_results;

  if ( ftlist == NULL ) return NULL;
  
  // TODO: Momentan durchsuche ich jedesmal die Liste nach unknowntyp
  //       Da ich nun dazubergegangen bin, die Position nicht mehr zu
  //       garantieren, muss ich suchen
  //       Vielleicht wre eine Variable mglich die bei nderung der Liste 
  //       gendert wird und so Sofortzugriff erlaubt
  //       wconfig->getunknowntype() geht AUF JEDEN FALL nicht, die ftlist
  //       NICHT die gleiche wie im wconfig ist (wegen Thread-behandlung)
  //       (zumindest im Moment nicht, bei einer thread-safe liste und FTs
  //        knnte dies wieder zurckgenommen werden)
  id = ftlist->initEnum();
  unknownft = (WCFiletype*)ftlist->getFirstElement( id );
  while ( unknownft != NULL ) {
    if ( unknownft->getinternID() == UNKNOWNTYPE ) break;
    unknownft = (WCFiletype*)ftlist->getNextElement( id );
  }
  ftlist->closeEnum( id );
  // Vorbesetzen, damit immer unknown typ kommt, selbst wenn Datei nicht den Anforderungen
  //  fuer Dateityperkennung genuegt
  filetype=unknownft;
  if(ftlist->size()<4) {
    return filetype;
  }
  if ( ( S_ISDIR( statbuf.mode ) ) && ( isCorrupt == false ) ) {
    // Wirklich ein Verzeichnis
    return filetype;
  }
  
  if ( isReg() == false ) skipcontent = true;
  if ( statbuf.size < 1 ) skipcontent = true;

  if(skipcontent==false) {
    reads = -1;
    pfp = new PDatei();
    if ( pfp->open( fullname ) == 0 ) {
      reads = pfp->read( fbuffer, 64 );
    }
    delete pfp;
  }

  if ( condparser == NULL ) {
    localparser = new CondParser();
    condparser = localparser;
  }

  id = ftlist->initEnum();
  ft = (WCFiletype*)ftlist->getFirstElement( id );
  while ( ft != NULL ) {
      cand1 = runTests( ft, skipcontent, condparser, fbuffer, reads, temp_results );
    if ( ( cand1.type != NULL ) || ( cand1.tests_done < 1 ) ) {
      // check subtypes if we had a hit or did no tests
      subtypes = ft->getSubTypeList();
      if ( subtypes != NULL ) {
	if ( subtypes->size() > 0 ) {
            cand2 = checkFiletype( subtypes, skipcontent, condparser, fbuffer, reads, temp_results );
	  if ( cand2.prio > 0 ) {
	    // subtypes are always more important
	    cand1.prio = cand2.prio;
	    cand1.type = cand2.type;
	  }
	}
      }
    }
    if ( cand1.type != NULL ) {
      if ( cand1.prio > erg.prio ) {
	erg.prio = cand1.prio;
	erg.type = cand1.type;
      }
    }
    ft = (WCFiletype*)ftlist->getNextElement( id );
  }

  if ( erg.type  == NULL ) {
    filetype = unknownft;
  } else {
    filetype = erg.type;
  }
  ftlist->closeEnum( id );
  
  if ( filetype != NULL ) {
    ft = filetype;
    while ( ft->getColorMode() == ft->PARENTCOL ) {
      ft = ft->getParentType();
      if ( ft == NULL ) break;
    }
    if ( ft != NULL ) {
      if ( ft->getColorMode() == ft->CUSTOMCOL ) {
	setCustomColors( true );
	setColor( 0, ft->getCustomColor( 0 ) );
	setColor( 1, ft->getCustomColor( 1 ) );
      } else if ( ft->getColorMode() == ft->EXTERNALCOL ) {
	char *o1 = condparser->getOutput( ft->getColorExt(), this );
	int tfg[4], tbg[4], z;

	if ( o1 != NULL ) {
	  z = sscanf( o1, "%d %d %d %d %d %d %d %d", &tfg[0], &tbg[0], &tfg[1], &tbg[1], &tfg[2], &tbg[2], &tfg[3], &tbg[3] );
	  if ( z == 8 ) {
	   setCustomColors( true );
	   setColor( 0, tfg );
	   setColor( 1, tbg );
	  }
	  _freesafe( o1 );
	}
      }
    }
  }

  // free temporary buffers
  if ( localparser != NULL ) delete localparser;
  freeContentBuf();
  freeOutputBuf();
  
  return filetype;
}

bool FileEntry::match(char* pattern)
{
  if(fnmatch(pattern,name,0)==0) return true;
  return false;
}

void FileEntry::setLVC_DND( FieldListViewDND *lv, int row, const std::vector<WorkerTypes::listcol_t> *dis )
{
  if ( lv == NULL ) return;

  setLVC( lv, row, dis );
  lv->replacedndtext( row, name );
  lv->setDataExt( row, new NMRowData( row, this, fullname ) );
}

loff_t FileEntry::size() const
{
  return statbuf.size;
}

time_t FileEntry::lastaccess() const
{
  return statbuf.lastaccess;
}

time_t FileEntry::lastmod() const
{
  return statbuf.lastmod;
}

time_t FileEntry::lastchange() const
{
  return statbuf.lastchange;
}

mode_t FileEntry::mode() const
{
  return statbuf.mode;
}

uid_t FileEntry::userid() const
{
  return statbuf.userid;
}

gid_t FileEntry::groupid() const
{
  return statbuf.groupid;
}

ino_t FileEntry::inode() const
{
  return statbuf.inode;
}

nlink_t FileEntry::nlink() const
{
  return statbuf.nlink;
}

loff_t FileEntry::blocks() const
{
  return statbuf.blocks;
}

loff_t FileEntry::dsize() const
{
  return dstatbuf.size;
}

time_t FileEntry::dlastaccess() const
{
  return dstatbuf.lastaccess;
}

time_t FileEntry::dlastmod() const
{
  return dstatbuf.lastmod;
}

time_t FileEntry::dlastchange() const
{
  return dstatbuf.lastchange;
}

mode_t FileEntry::dmode() const
{
  return dstatbuf.mode;
}

uid_t FileEntry::duserid() const
{
  return dstatbuf.userid;
}

gid_t FileEntry::dgroupid() const
{
  return dstatbuf.groupid;
}

ino_t FileEntry::dinode() const
{
  return dstatbuf.inode;
}

nlink_t FileEntry::dnlink() const
{
  return dstatbuf.nlink;
}

loff_t FileEntry::dblocks() const
{
  return dstatbuf.blocks;
}

dev_t FileEntry::rdev() const
{
  return statbuf.rdev;
}

dev_t FileEntry::drdev() const
{
  return dstatbuf.rdev;
}

dev_t FileEntry::dev() const
{
  return statbuf.dev;
}

dev_t FileEntry::ddev() const
{
  return dstatbuf.dev;
}

bool FileEntry::checkAccess( int m )
{
  if ( fullname == NULL ) return false;
  return ( worker_access( fullname, m ) == 0 ) ? true : false;
}

char *FileEntry::getContentStr( int pos, int len )
{
  char *resstr = NULL;
  int mylen, i;

  if ( pos < 0 ) return dupstring( "" );
  if ( len < 1 ) return dupstring( "" );

  if ( contentBuf.content == NULL ) readContentBuf();
  if ( contentBuf.content != NULL ) {
    mylen = a_min( len, contentBuf.size - pos );
    if ( mylen > 0 ) {
      resstr = (char*)_allocsafe( mylen + 1 );
      for ( i = 0; i < mylen; i++ ) {
        resstr[i] = contentBuf.content[i+pos];
      }
      resstr[i] = '\0';
    }
  }
  
  return ( resstr != NULL ) ? resstr : dupstring( "" );
}

int FileEntry::getContentNum( int pos, int len )
{
  int mylen, i;
  int res = -1;

  if ( pos < 0 ) return -1;
  if ( len < 1 ) return -1;
  
  if ( contentBuf.content == NULL ) readContentBuf();
  if ( contentBuf.content != NULL ) {
    mylen = a_min( len, contentBuf.size - pos );
    if ( mylen > 0 ) {
      res = 0;
      for ( i = 0; i < mylen; i++ ) {
        res <<= 8;
        res += (unsigned char)contentBuf.content[i+pos];
      }
    }
  }
  
  return res;
}

void FileEntry::freeContentBuf()
{
  if ( contentBuf.content != NULL ) {
    _freesafe( contentBuf.content );
    contentBuf.content = NULL;
    contentBuf.size = 0;
  }
}

void FileEntry::readContentBuf( int len )
{
  int reads, i;
  unsigned char fbuffer[64];
  PDatei *pfp;
  
  if ( isReg() == false ) return;
  if ( ( len < 1 ) || ( len > 64 ) ) return;
  if ( contentBuf.content == NULL ) {
    reads = -1;
    pfp = new PDatei();
    if ( pfp->open( fullname ) == 0 ) {
      reads = pfp->read( fbuffer, len );
    }
    delete pfp;
    if ( ( reads > 0 ) && ( reads <= len ) ) {
      contentBuf.content = (char*)_allocsafe( len + 1 );
      contentBuf.size = reads;
      for ( i = 0; i < reads; i++ ) {
        contentBuf.content[i] = (char)fbuffer[i];
      }
      for ( i = reads; i < ( len + 1 ); i++ ) {
        contentBuf.content[i] = '\0';
      }
    }
  }
}

void FileEntry::freeOutputBuf()
{
  if ( outputBuf != NULL ) {
    delete outputBuf;
    outputBuf = NULL;
  }
}

const char *FileEntry::findOutput( const char *tcommand )
{
  if ( outputBuf == NULL ) {
    outputBuf = new StringBuf();
  } else {
    return outputBuf->find( tcommand );
  }
  return NULL;
}

void FileEntry::addOutput( const char *tcommand, const char *toutput, int value )
{
  if ( outputBuf == NULL ) outputBuf = new StringBuf();
  
  outputBuf->add( tcommand, toutput, value );
}

int FileEntry::findReturnvalue( const char *tcommand, int *return_value )
{
  if ( outputBuf == NULL ) {
    outputBuf = new StringBuf();
  } else {
    return outputBuf->findValue( tcommand, return_value );
  }
  return -1;
}

int FileEntry::findOutputAndRV( const char *tcommand,
                                const char **return_output,
                                int *return_value )
{
  if ( outputBuf == NULL ) {
    outputBuf = new StringBuf();
  } else {
    return outputBuf->find( tcommand, return_output, return_value );
  }
  return -1;
}

bool FileEntry::isReg() const
{
  if ( isLink == false ) {
    if ( S_ISREG( statbuf.mode ) ) return true;
    else return false;
  }
  if ( isCorrupt == true ) return false;
  if ( S_ISREG( dstatbuf.mode ) ) return true;
  return false;
}

void FileEntry::setCustomColors( bool nv )
{
  customcolors = nv;
}

bool FileEntry::getCustomColors() const
{
  return customcolors;
}

void FileEntry::setColor( int m, const int *v )
{
  int i;

  if ( v == NULL ) return;
  if ( m == 0 ) {
    for ( i = 0; i < 4; i++ ) {
      fg[i] = v[i];
    }
  } else if ( m == 1 ) {
    for ( i = 0; i < 4; i++ ) {
      bg[i] = v[i];
    }
  }
}

void FileEntry::setColor( int m, int pos, int v )
{
  if ( ( pos >= 0 ) && ( pos < 4 ) ) {
    if ( m == 0 ) {
      fg[pos] = v;
    } else if ( m == 1 ) {
      bg[pos] = v;
    }
  }
}

int FileEntry::getColor( int m, int pos ) const
{
  if ( ( pos >= 0 ) && ( pos < 4 ) ) {
    if ( m == 0 ) {
      return fg[pos];
    } else if ( m == 1 ) {
      return bg[pos];
    }
  }
  return -1;
}

const int *FileEntry::getColor( int m ) const
{
  if ( m == 0 ) {
    return fg;
  } else if ( m == 1 ) {
    return bg;
  }
  return NULL;
}

/*
 * Method to check for same destination file
 * it currently only check the destination itself, not the possible
 * destination of a symlink
 */
bool FileEntry::isSameFile( const char *othername, bool follow_symlinks ) const
{
  worker_struct_stat buf;

  if ( othername == NULL ) return false;
  if ( worker_stat( othername, &buf ) != 0 ) return false;

  if ( ( isLink == true ) && ( follow_symlinks == true ) && ( isCorrupt == false ) ) {
    if ( ( dinode() == buf.st_ino ) && ( ddev() == buf.st_dev ) ) return true;
  } else {
    if ( ( inode() == buf.st_ino ) && ( dev() == buf.st_dev ) ) return true;
  }
  return false;
}

FileEntry::FileEntry( const FileEntry &other )
{
  fullname = ( other.fullname != NULL ) ? dupstring( other.fullname ) : NULL;
  name = (other.name != NULL ) ? dupstring( other.name ) : NULL;
  filetype = other.filetype;
  use = other.use;
  isHidden = other.isHidden;
  reclistQueued = other.reclistQueued;

  memcpy( &statbuf, &( other.statbuf ), sizeof( statbuf ) );
  memcpy( &dstatbuf, &( other.dstatbuf ), sizeof( dstatbuf ) );
  customcolors = other.customcolors;
  setColor( 0, other.fg );
  setColor( 1, other.bg );

  dirsize = other.dirsize;
  select = other.select;
  isLink = other.isLink;
  isCorrupt = other.isCorrupt;

  // not copied
  contentBuf.content = NULL;
  contentBuf.size = 0;
  
  outputBuf = NULL;

  _ID = other._ID;
}

/*
 * Method to test for equal fileentry
 *
 * Does same extented test to ensure that the fileentry
 * is a duplicated version
 */
bool FileEntry::equals( const FileEntry *fe ) const
{
  if ( fe == NULL ) return false;
  if ( fe == this ) return true;
  return equals( *fe );
}

bool FileEntry::equals( const FileEntry &fe ) const
{
  // require name
  if ( ( fullname == NULL ) || ( fe.fullname == NULL ) ) return false;
  
  // require same inode/device

  //Currently disabled because AVFS can set 0
  //  if ( statbuf.inode == 0 ) return false;
  if ( ( statbuf.inode != fe.statbuf.inode ) || ( statbuf.dev != fe.statbuf.dev ) ) return false;

  // require same size
  if ( statbuf.size != fe.statbuf.size ) return false;

  // require same name
  if ( strcmp( fullname, fe.fullname ) != 0 ) return false;

  // finally require same times
  if ( statbuf.lastaccess != fe.statbuf.lastaccess ) return false;
  if ( statbuf.lastmod != fe.statbuf.lastmod ) return false;
  if ( statbuf.lastchange != fe.statbuf.lastchange ) return false;

  return true;
}

int FileEntry::getID() const
{
    return _ID;
}

void FileEntry::setID( int ID )
{
    _ID = ID;
}
