/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "mainwindow.h"

#include <QAction>
#include <QCloseEvent>
#include <QApplication>
#include <QSettings>
#include <QSplitter>
#include <QTabWidget>
#include <QFileDialog>
#include <QMessageBox>
#include <QToolBar>
#include <QLayout>
#include <QTimer>

#include "data/datamanager.h"
#include "data/connectioninfo.h"
#include "data/loginbatch.h"
#include "data/bookmark.h"
#include "commands/commandmanager.h"
#include "widgets/statusbar.h"
#include "widgets/panewidget.h"
#include "dialogs/startpage.h"
#include "dialogs/logindialog.h"
#include "dialogs/userdialogs.h"
#include "dialogs/finditemdialog.h"
#include "dialogs/preferencesdialog.h"
#include "dialogs/settingsdialog.h"
#include "dialogs/connectioninfodialog.h"
#include "views/projectsview.h"
#include "views/dashboardview.h"
#include "views/folderview.h"
#include "views/issueview.h"
#include "xmlui/builder.h"
#include "viewmanager.h"
#include "connectionmanager.h"
#include "connectionmanager.h"
#include "iconloader.h"

#if defined( HAVE_OPENSSL )
#include "dialogs/sslerrorsdialog.h"
#endif

#if defined( Q_WS_WIN )
static const int TrayIconSize = 16;
#else
static const int TrayIconSize = 22;
#endif

MainWindow::MainWindow() :
    m_view( NULL ),
    m_dashboard( NULL ),
    m_folderView( NULL ),
    m_issueView( NULL ),
    m_activeView( NULL ),
    m_folderPane( NULL ),
    m_issuePane( NULL ),
    m_layout( LayoutNone ),
    m_selectedFolderId( 0 ),
    m_selectedIssueId( 0 ),
    m_supressFilter( false )
{
    setWindowTitle( tr( "WebIssues" ) );

    QAction* action;

    action = new QAction( IconLoader::icon( "file-quit" ), tr( "&Quit" ), this );
    action->setMenuRole( QAction::QuitRole );
    action->setShortcut( tr( "Ctrl+Q" ) );
    connect( action, SIGNAL( triggered() ), this, SLOT( quit() ) );
    setAction( "quit", action );

    action = new QAction( IconLoader::icon( "connection-open" ), tr( "&New Connection..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( newConnection() ) );
    setAction( "newConnection", action );

    action = new QAction( IconLoader::icon( "connection-close" ), tr( "&Close Connection" ), this );
    action->setVisible( false );
    connect( action, SIGNAL( triggered() ), this, SLOT( closeConnection() ) );
    setAction( "closeConnection", action );

    action = new QAction( IconLoader::icon( "status-info" ), tr( "Connection &Details..." ), this );
    action->setVisible( false );
    connect( action, SIGNAL( triggered() ), this, SLOT( connectionInfo() ) );
    setAction( "connectionInfo", action );

    action = new QAction( IconLoader::icon( "configure-import" ), tr( "&Import..." ), this );
    action->setVisible( false );
    connect( action, SIGNAL( triggered() ), this, SLOT( importSettings() ) );
    setAction( "importSettings", action );

    action = new QAction( IconLoader::icon( "configure-export" ), tr( "&Export..." ), this );
    action->setVisible( false );
    connect( action, SIGNAL( triggered() ), this, SLOT( exportSettings() ) );
    setAction( "exportSettings", action );

    action = new QAction( IconLoader::icon( "view-users" ), tr( "&User Accounts" ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( showUsers() ) );
    setAction( "showUsers", action );

    action = new QAction( IconLoader::icon( "view-types" ), tr( "Issue &Types" ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( showTypes() ) );
    setAction( "showTypes", action );

    action = new QAction( IconLoader::icon( "edit-goto" ), tr( "&Go To Item..." ), this );
    action->setShortcut( tr( "Ctrl+G" ) );
    connect( action, SIGNAL( triggered() ), this, SLOT( gotoItem() ) );
    setAction( "gotoItem", action );

    action = new QAction( IconLoader::icon( "edit-password" ), tr( "Change &Password..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( changePassword() ) );
    setAction( "changePassword", action );

    action = new QAction( IconLoader::icon( "preferences" ), tr( "User P&references..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( userPreferences() ) );
    setAction( "userPreferences", action );

    action = new QAction( IconLoader::icon( "configure" ), tr( "&Configure WebIssues..." ), this );
    action->setMenuRole( QAction::PreferencesRole );
    connect( action, SIGNAL( triggered() ), this, SLOT( configure() ) );
    setAction( "configure", action );

    action = new QAction( IconLoader::icon( "help-contents" ), tr( "WebIssues &Manual" ), this );
    action->setShortcut( QKeySequence::HelpContents );
    connect( action, SIGNAL( triggered() ), qApp, SLOT( manual() ) );
    setAction( "manual", action );

    action = new QAction( IconLoader::icon( "webissues" ), tr( "&About WebIssues" ), this );
    action->setMenuRole( QAction::AboutRole );
    connect( action, SIGNAL( triggered() ), qApp, SLOT( about() ) );
    setAction( "about", action );

    action = new QAction( IconLoader::icon( "about-qt" ), tr( "About &Qt" ), this );
    action->setMenuRole( QAction::AboutQtRole );
    connect( action, SIGNAL( triggered() ), qApp, SLOT( aboutQt() ) );
    setAction( "aboutQt", action );

    setTitle( "menuWebIssues", tr( "&WebIssues" ) );
    setTitle( "menuServerSettings", tr( "Cli&ent Configuration" ) );
    setTitle( "menuEdit", tr( "&Edit" ) );
    setTitle( "menuTools", tr( "&Tools" ) );
    setTitle( "menuHelp", tr( "&Help" ) );
    setTitle( "mainToolBar", tr( "Main Toolbar" ) );

    setButtonStyle( "gotoItem", Qt::ToolButtonTextBesideIcon );

    loadXmlUiFile( ":/resources/mainwindow.xml" );

    m_builder = new XmlUi::Builder( this );
    connect( m_builder, SIGNAL( toolBarCreated( QToolBar* ) ), this, SLOT( toolBarCreated( QToolBar* ) ) );

    updateActions( false );

    m_builder->setToolbarVisible( "mainToolBar", false );
    m_builder->addClient( this );

    setStatusBar( new ::StatusBar( this ) );

    qRegisterMetaType<Bookmark>( "Bookmark" );

    showStartPage();

    QSettings* settings = configData->settings();
    settings->beginGroup( "MainWindow" );
    if ( settings->contains( "Geometry" ) ) {
        QString geometry = settings->value( "Geometry" ).toString();
        restoreGeometry( QByteArray::fromHex( geometry.toAscii() ) );
    } else {
        resize( settings->value( "Size", QSize( 750, 500 ) ).toSize() );
        QPoint position = settings->value( "Position" ).toPoint();
        if ( !position.isNull() )
            move( position );
    }
    settings->endGroup();

    connect( configData, SIGNAL( settingsChanged() ), this, SLOT( settingsChanged() ) );

    m_trayIcon = new QSystemTrayIcon( this );
    m_trayIcon->setIcon( IconLoader::pixmap( "webissues", TrayIconSize ) );
    m_trayIcon->setToolTip( tr( "WebIssues - Not connected" ) );

    connect( m_trayIcon, SIGNAL( activated( QSystemTrayIcon::ActivationReason ) ), this, SLOT( trayIconActivated( QSystemTrayIcon::ActivationReason ) ) );

    connect( m_builder, SIGNAL( reset() ), this, SLOT( builderReset() ) );
    builderReset();

    settingsChanged();

    statusBar()->showInfo( tr( "Not connected to server" ) );

    m_selectionTimer = new QTimer( this );
    m_selectionTimer->setInterval( 400 );
    m_selectionTimer->setSingleShot( true );

    connect( m_selectionTimer, SIGNAL( timeout() ), this, SLOT( updateSelection() ) );
}

MainWindow::~MainWindow()
{
    if ( m_layout != LayoutNone )
        closeView( false );

    QSettings* settings = configData->settings();
    settings->beginGroup( "MainWindow" );
    settings->setValue( "Geometry", QString( saveGeometry().toHex() ) );
    settings->setValue( "Size", size() );
    settings->setValue( "Position", pos() );
    settings->endGroup();
}

void MainWindow::closeEvent( QCloseEvent* e )
{
    if ( m_trayIcon->isVisible() )
        hide();
    else
        quit();

    e->ignore();
}

QMenu* MainWindow::createPopupMenu()
{
    return NULL;
}

void MainWindow::reconnect()
{
    BookmarkSettings settings;
    if ( settings.bookmarks().count() > 0 )
       openConnection( settings.bookmarks().at( 0 ) ); 
}

void MainWindow::quit()
{
    if ( viewManager && !viewManager->queryCloseViews() )
        return;

    qApp->quit();
}

void MainWindow::newConnection()
{
    LoginDialog dialog( this );
    if ( dialog.exec() != QDialog::Accepted )
        return;

    BookmarkSettings settings;
    settings.addBookmark( dialog.createBookmark() );

    initializeView();
}

void MainWindow::closeConnection()
{
    if ( !viewManager->queryCloseViews() )
        return;

    closeView( true );

    connectionManager->closeConnection();
}

void MainWindow::initializeView()
{
    m_startPage = NULL;

    m_layout = configData->windowLayout();

    QWidget* parent = NULL;

    if ( m_layout == LayoutSinglePane ) {
        parent = this;
    } else if ( m_layout == LayoutThreePanes ) {
        parent = new QSplitter( Qt::Horizontal, this );
    }

    QWidget* widget = new QWidget( parent );
    QTabWidget* tabWidget = new QTabWidget( widget );

    QVBoxLayout* layout = new QVBoxLayout( widget );
    layout->setContentsMargins( 0, 1, 0, 0 );
    layout->addWidget( tabWidget );

    m_view = new ProjectsView( this, tabWidget );
    viewManager->addView( m_view );

    m_dashboard = new DashboardView( this, tabWidget );
    viewManager->addView( m_dashboard );

    tabWidget->addTab( m_view->mainWidget(), IconLoader::icon( "project" ), tr( "Projects" ) );
    tabWidget->addTab( m_dashboard->mainWidget(), IconLoader::icon( "view-dashboard" ), tr( "Dashboard" ) );

    if ( m_layout == LayoutSinglePane ) {
        setCentralWidget( widget );
        widget->show();
    } else if ( m_layout == LayoutThreePanes ) {
        QSplitter* horizSplitter = (QSplitter*)parent;

        horizSplitter->addWidget( widget );

        QSplitter* vertSplitter = new QSplitter( Qt::Vertical, horizSplitter );

        m_folderPane = new PaneWidget( vertSplitter );
        m_folderPane->setPlaceholderText( tr( "No folder selected" ) );
        vertSplitter->addWidget( m_folderPane );

        m_folderView = new FolderView( this, m_folderPane );
        m_folderView->mainWidget()->setFocusPolicy( Qt::ClickFocus );
        viewManager->addView( m_folderView );

        m_folderPane->setMainWidget( m_folderView->mainWidget() );

        m_issuePane = new PaneWidget( vertSplitter );
        m_issuePane->setPlaceholderText( tr( "No issue selected" ) );
        vertSplitter->addWidget( m_issuePane );

        m_issueView = new IssueView( this, m_issuePane );
        m_issueView->mainWidget()->setFocusPolicy( Qt::ClickFocus );
        viewManager->addView( m_issueView );

        m_issuePane->setMainWidget( m_issueView->mainWidget() );

        horizSplitter->addWidget( vertSplitter );

        setCentralWidget( horizSplitter );
        horizSplitter->show();

        QSettings* settings = configData->settings();
        settings->beginGroup( "MainWindow" );
        QString hex = settings->value( "HorizSplitter" ).toString();
        horizSplitter->restoreState( QByteArray::fromHex( hex.toAscii() ) );
        hex = settings->value( "VertSplitter" ).toString();
        vertSplitter->restoreState( QByteArray::fromHex( hex.toAscii() ) );
        settings->endGroup();
    }

    m_builder->supressUpdate();

    updateActions( true );

    connect( m_view, SIGNAL( captionChanged( const QString& ) ), this, SLOT( captionChanged( const QString& ) ) );

    connect( m_view, SIGNAL( statusChanged( const QPixmap&, const QString& ) ), statusBar(), SLOT( showStatus( const QPixmap&, const QString& ) ) );
    connect( m_view, SIGNAL( summaryChanged( const QPixmap&, const QString& ) ), statusBar(), SLOT( showSummary( const QPixmap&, const QString& ) ) );

    m_activeView = m_view;

    m_builder->setToolbarVisible( "mainToolBar", true );
    m_builder->addClient( m_view );

    m_builder->resumeUpdate();

    qApp->installEventFilter( this );

    m_trayIcon->setIcon( IconLoader::overlayedPixmap( "webissues", "overlay-connected", TrayIconSize ) );
    m_trayIcon->setToolTip( tr( "WebIssues - Connected to %1" ).arg( connectionManager->connectionInfo()->serverName() ) );

    m_view->initialUpdate();
    m_dashboard->initialUpdate();

    if ( m_layout == LayoutThreePanes ) {
        connect( m_folderView, SIGNAL( enabledChanged( bool ) ), this, SLOT( folderEnabledChanged( bool ) ) );
        connect( m_issueView, SIGNAL( enabledChanged( bool ) ), this, SLOT( issueEnabledChanged( bool ) ) );

        connect( m_folderView, SIGNAL( captionChanged( const QString& ) ), this, SLOT( captionChanged( const QString& ) ) );

        connect( m_view, SIGNAL( selectedFolderChanged( int ) ), this, SLOT( selectedFolderChanged( int ) ) );
        connect( m_dashboard, SIGNAL( selectedWatchChanged( int, const QString& ) ), this, SLOT( selectedWatchChanged( int, const QString& ) ) );

        connect( m_folderView, SIGNAL( selectedFilterChanged( const QString& ) ), this, SLOT( selectedFilterChanged( const QString& ) ) );
        connect( m_folderView, SIGNAL( selectedIssueChanged( int ) ), this, SLOT( selectedIssueChanged( int ) ) );

        connect( m_issueView, SIGNAL( gotoIssue( int, int ) ), this, SLOT( gotoIssue( int, int ) ), Qt::QueuedConnection );
        connect( m_folderView, SIGNAL( gotoItem( int ) ), this, SLOT( gotoItem( int ) ) );

        folderEnabledChanged( m_folderView->isEnabled() );
        issueEnabledChanged( m_issueView->isEnabled() );

        m_folderView->initialUpdate();
        m_issueView->initialUpdate();
    }
}

void MainWindow::closeView( bool startPage )
{
    if ( m_layout == LayoutThreePanes ) {
        QSplitter* horizSplitter = (QSplitter*)centralWidget();
        QSplitter* vertSplitter = (QSplitter*)horizSplitter->widget( 1 );

        QSettings* settings = configData->settings();
        settings->beginGroup( "MainWindow" );
        settings->setValue( "HorizSplitter", QString( horizSplitter->saveState().toHex() ) );
        settings->setValue( "VertSplitter", QString( vertSplitter->saveState().toHex() ) );
        settings->endGroup();
    }

    qApp->removeEventFilter( this );

    m_selectionTimer->stop();

    m_builder->supressUpdate();

    delete m_view;
    m_view = NULL;

    delete m_dashboard;
    m_dashboard = NULL;

    delete m_folderView;
    m_folderView = NULL;

    delete m_issueView;
    m_issueView = NULL;

    m_activeView = NULL;

    m_folderPane = NULL;
    m_issuePane = NULL;

    m_layout = LayoutNone;

    updateActions( false );
    action( "newConnection" )->setEnabled( true );

    m_builder->setToolbarVisible( "mainToolBar", false );

    m_builder->resumeUpdate();

    m_selectedFolderId = 0;
    m_selectedFilter.clear();
    m_selectedIssueId = 0;

    if ( startPage ) {
        showStartPage();

        m_trayIcon->setIcon( IconLoader::pixmap( "webissues", TrayIconSize ) );
        m_trayIcon->setToolTip( tr( "WebIssues - Not connected" ) );

        setWindowTitle( tr( "WebIssues" ) );

        statusBar()->showInfo( tr( "Connection closed" ) );
        statusBar()->showSummary( QPixmap(), QString() );
    }
}

void MainWindow::updateActions( bool connected )
{
    action( "newConnection" )->setVisible( !connected );
    action( "closeConnection" )->setVisible( connected );
    action( "importSettings" )->setVisible( connected );
    action( "exportSettings" )->setVisible( connected );
    action( "connectionInfo" )->setVisible( connected );
    action( "showUsers" )->setVisible( connected );
    action( "showTypes" )->setVisible( connected );
    action( "gotoItem" )->setVisible( connected );
    action( "changePassword" )->setVisible( connected );
    action( "userPreferences" )->setVisible( connected && connectionManager->connectionInfo()->checkVersion( "0.8.4" ) );

    m_builder->rebuildAll();
}

bool MainWindow::eventFilter( QObject* object, QEvent* e )
{
    if ( e->type() != QEvent::MouseButtonPress &&
        e->type() != QEvent::MouseButtonDblClick &&
        e->type() != QEvent::FocusIn &&
        e->type() != QEvent::ContextMenu )
        return false;

    if ( !object->isWidgetType() )
        return false;

    QWidget* widget = static_cast<QWidget*>( object );

    if ( ( ( widget->windowFlags().testFlag( Qt::Dialog ) ) && widget->isModal() ) ||
        ( widget->windowFlags().testFlag( Qt::Popup ) ) || ( widget->windowFlags().testFlag( Qt::Tool ) ) )
        return false;

    while ( widget ) {
        if ( widget->topLevelWidget() != this )
            return false;

        View* view = NULL;
        if ( m_view->mainWidget() == widget )
            view = m_view;
        else if ( m_dashboard->mainWidget() == widget )
            view = m_dashboard;
        else if ( m_folderView && m_folderView->mainWidget() == widget )
            view = m_folderView;
        else if ( m_issueView && m_issueView->mainWidget() == widget )
            view = m_issueView;

        if ( view ) {
            setActiveView( view );
            return false;
        }

        widget = widget->parentWidget();
    }

    return false;
}

void MainWindow::setActiveView( View* view )
{
    if ( m_activeView == view )
        return;

    m_builder->supressUpdate();

    if ( m_activeView ) {
        disconnect( m_activeView, NULL, statusBar(), NULL );
        if ( m_activeView->isEnabled() )
            m_builder->removeClient( m_activeView );
    }

    m_activeView = view;

    connect( view, SIGNAL( statusChanged( const QPixmap&, const QString& ) ), statusBar(), SLOT( showStatus( const QPixmap&, const QString& ) ) );
    connect( view, SIGNAL( summaryChanged( const QPixmap&, const QString& ) ), statusBar(), SLOT( showSummary( const QPixmap&, const QString& ) ) );

    statusBar()->showStatus( view->statusPixmap(), view->statusText() );
    statusBar()->showSummary( view->summaryPixmap(), view->summaryText() );

    if ( view->isEnabled() )
        m_builder->addClient( view );

    m_builder->resumeUpdate();
}

void MainWindow::showStartPage()
{
    m_startPage = new StartPage( this );

    connect( m_startPage, SIGNAL( openConnection( const Bookmark& ) ), this, SLOT( openConnection( const Bookmark& ) ), Qt::QueuedConnection );
    connect( m_startPage, SIGNAL( editConnection( const Bookmark& ) ), this, SLOT( editConnection( const Bookmark& ) ), Qt::QueuedConnection );
    connect( m_startPage, SIGNAL( newConnection() ), this, SLOT( newConnection() ), Qt::QueuedConnection );
    connect( m_startPage, SIGNAL( abort() ), this, SLOT( abortLogin() ) );

    setCentralWidget( m_startPage );
    m_startPage->show();

    m_startPage->setFocus();
}

void MainWindow::folderEnabledChanged( bool enabled )
{
    m_folderPane->setMainWidgetVisible( enabled );

    if ( m_folderView == m_activeView ) {
        if ( enabled )
           m_builder->addClient( m_view );
        else
            m_builder->removeClient( m_view );
    }
}

void MainWindow::issueEnabledChanged( bool enabled )
{
    m_issuePane->setMainWidgetVisible( enabled );

    if ( m_issueView == m_activeView ) {
        if ( enabled )
           m_builder->addClient( m_issueView );
        else
            m_builder->removeClient( m_issueView );
    }
}

void MainWindow::selectedFolderChanged( int folderId )
{
    if ( m_selectedFolderId != folderId ) {
        m_selectedFolderId = folderId;
        m_selectedFilter.clear();
        m_selectionTimer->start();
    }
}

void MainWindow::selectedWatchChanged( int folderId, const QString& filter )
{
    if ( m_selectedFolderId != folderId || m_selectedFilter != filter ) {
        m_selectedFolderId = folderId;
        m_selectedFilter = filter;
        m_selectionTimer->start();
    }
}

void MainWindow::selectedFilterChanged( const QString& filter )
{
    if ( !m_supressFilter && m_selectedFilter != filter ) {
        m_selectedFilter = filter;
        m_dashboard->setSelectedWatch( m_selectedFolderId, m_selectedFilter );
    }
}

void MainWindow::selectedIssueChanged( int issueId )
{
    if ( m_selectedIssueId != issueId ) {
        m_selectedIssueId = issueId;
        m_selectionTimer->start();
    }
}

void MainWindow::updateSelection()
{
    m_view->setSelectedFolderId( m_selectedFolderId );
    m_dashboard->setSelectedWatch( m_selectedFolderId, m_selectedFilter );

    if ( m_folderView->id() != m_selectedFolderId ) {
        m_supressFilter = true;

        m_folderView->setId( m_selectedFolderId );
        m_folderView->initialUpdate();

        m_supressFilter = false;
        m_selectedIssueId = 0;
    }

    if ( m_folderView->selectedFilter() != m_selectedFilter )
        m_folderView->setSelectedFilter( m_selectedFilter );

    if ( m_issueView->id() != m_selectedIssueId ) {
        m_issueView->setId( m_selectedIssueId );
        m_issueView->initialUpdate();
    }
}

void MainWindow::configure()
{
    SettingsDialog dialog( this );
    dialog.exec();
}

void MainWindow::settingsChanged()
{
    m_trayIcon->setVisible( configData->dockWindow() );

    if ( m_layout != LayoutNone && m_layout != configData->windowLayout() ) {
        closeView( false );
        initializeView();
    }
}

void MainWindow::openConnection( const Bookmark& bookmark )
{
    if ( bookmark.password().isEmpty() ) {
        showLoginDialog( bookmark, false );
        return;
    }

    m_bookmark = bookmark;

    connectionManager->createCommandManager();

#if defined( HAVE_OPENSSL )
    connect( commandManager, SIGNAL( sslErrors( const QList<QSslError>& ) ),
        this, SLOT( sslErrors( const QList<QSslError>& ) ) );
#endif

    m_batch = new LoginBatch( bookmark.login(), bookmark.password() );

    connect( m_batch, SIGNAL( completed( bool ) ), this, SLOT( loginCompleted( bool ) ) );

    commandManager->setServerUrl( bookmark.url() ); 
    commandManager->execute( m_batch );

    statusBar()->showBusy( tr( "Connecting to server..." ) );

    m_startPage->setConnecting( true );

    action( "newConnection" )->setEnabled( false );    
}

void MainWindow::loginCompleted( bool successful )
{
    if ( successful ) {
        ConnectionInfo* info = m_batch->connectionInfo();
        connectionManager->openConnection( info );

        Bookmark bookmark( info->serverName(), info->serverUuid(),
            m_bookmark.url(), m_bookmark.login(), m_bookmark.password() );

        BookmarkSettings settings;
        settings.replaceBookmark( m_bookmark, bookmark );

        initializeView();
    } else {
        QString message = commandManager->errorMessage( tr( "Connection failed" ) );

        connectionManager->delayedCloseConnection();

        statusBar()->showWarning( message );

        m_startPage->setConnecting( false );

        action( "newConnection" )->setEnabled( true );
    }
}

void MainWindow::abortLogin()
{
    commandManager->abort( m_batch );
}

void MainWindow::editConnection( const Bookmark& bookmark )
{
    showLoginDialog( bookmark, true );
}

void MainWindow::showLoginDialog( const Bookmark& bookmark, bool editable )
{
    LoginDialog dialog( this );
    dialog.loadBookmark( bookmark, editable );

    if ( dialog.exec() != QDialog::Accepted )
        return;

    BookmarkSettings settings;
    settings.replaceBookmark( bookmark, dialog.createBookmark() );

    initializeView();
}

void MainWindow::importSettings()
{
    QString path = QFileDialog::getOpenFileName( this, tr( "Import Configuration" ),
        QDir::homePath(), tr( "Configuration Files (*.conf)" ) );

    if ( !path.isEmpty() ) {
        QSettings settings( path, QSettings::IniFormat );
        
        if ( settings.status() == QSettings::NoError ) {
            if ( connectionManager->canImportSettings( &settings ) ) {

                if ( QMessageBox::warning( this, tr( "Warning" ),
                    tr( "Are you sure you want to import selected client configuration?\n"
                        "All existing column settings, filters and watches will be overwritten." ),
                    QMessageBox::Ok | QMessageBox::Cancel ) == QMessageBox::Ok ) {

                    connectionManager->importSettings( &settings );

                    viewManager->postViewEvent( NULL, ViewEvent::UpdateSettings );

                    QMessageBox::information( this, tr( "Information" ),
                        tr( "Client configuration has been successfully imported." ) );
                }
            } else {
                QMessageBox::warning( this, tr( "Error" ),
                    tr( "Selected configuration is not valid for current server." ) );
            }
        } else {
            QMessageBox::warning( this, tr( "Error" ),
                tr( "The configuration file could not be loaded." ) );
        }
    }
}

void MainWindow::exportSettings()
{
    QString name = connectionManager->connectionInfo()->serverName();

    const char reserved[ 9 ] = { '\\', '/', ':', '*', '?', '"', '<', '>', '|' };
    for ( int i = 0; i < 9; i++ )
        name.remove( reserved[ i ] );

    while ( name.startsWith( '.' ) )
        name.remove( 0, 1 );

    if ( name.isEmpty() )
        name = "connection";

    QFileInfo fileInfo( QDir::home(), name + ".conf" );

    QString path = QFileDialog::getSaveFileName( this, tr( "Export Configuration" ),
         fileInfo.absoluteFilePath(), tr( "Configuration Files (*.conf)" ) );

    if ( !path.isEmpty() ) {
        bool ok = false;

        if ( !QFile::exists( path ) || QFile::remove( path ) ) {
            QSettings settings( path, QSettings::IniFormat );

            if ( settings.isWritable() ) {
                connectionManager->exportSettings( &settings );

                if ( settings.status() == QSettings::NoError )
                    ok = true;
            }
        }

        if ( ok ) {
            QMessageBox::information( this, tr( "Information" ),
                tr( "Client configuration has been successfully exported." ) );
        } else {
            QMessageBox::warning( this, tr( "Error" ),
                tr( "The configuration file could not be saved." ) );
        }
    }
}

void MainWindow::connectionInfo()
{
    ConnectionInfoDialog dialog( this );
    dialog.exec();
}

void MainWindow::showUsers()
{
    viewManager->openUsersView();
}

void MainWindow::showTypes()
{
    viewManager->openTypesView();
}

void MainWindow::gotoItem()
{
    FindItemDialog dialog( this );
    if ( dialog.exec() == QDialog::Accepted )
        gotoIssue( dialog.issueId(), dialog.itemId() );
}

void MainWindow::gotoIssue( int issueId, int itemId )
{
    const IssueRow* issue = dataManager->issues()->find( issueId );
    int folderId = issue ? issue->folderId() : 0;

    if ( m_layout == LayoutThreePanes && folderId != 0 ) {
        if ( folderId != m_folderView->id() ) {
            m_selectedFolderId = folderId;
            m_view->setSelectedFolderId( folderId );
            m_folderView->setId( folderId );
            m_folderView->initialUpdate();
        }
        if ( issueId != m_issueView->id() )
            m_folderView->gotoIssue( issueId, itemId );
        else
            m_issueView->gotoItem( itemId );
    } else {
        viewManager->openIssueView( issueId, itemId );
    }
}

void MainWindow::gotoItem( int itemId )
{
    updateSelection();
    m_issueView->gotoItem( itemId );
}

void MainWindow::changePassword()
{
    SetPasswordDialog dialog( connectionManager->connectionInfo()->userId(), this );
    dialog.exec();
}

void MainWindow::userPreferences()
{
    PreferencesDialog dialog( connectionManager->connectionInfo()->userId(), this );
    dialog.exec();
}

void MainWindow::trayIconActivated( QSystemTrayIcon::ActivationReason reason )
{
    if ( reason == QSystemTrayIcon::Trigger ) {
        if (isMinimized()) {
            // hide needed when changing desktop
            hide();
            showNormal();
            raise();
            activateWindow();
        } else if ( !isVisible() ) {
            show();
            raise();
            activateWindow();
        } else {
            hide();
        }
    }
}

void MainWindow::captionChanged( const QString& /*caption*/ )
{
    QString caption = m_view->caption();

    if ( m_layout == LayoutThreePanes && m_folderView->isEnabled() )
        caption = QString( "%1 - %2" ).arg( m_folderView->caption(), caption );

    setWindowTitle( tr( "%1 - WebIssues" ).arg( caption ) );
}

void MainWindow::toolBarCreated( QToolBar* toolBar )
{
    toolBar->setIconSize( QSize( 16, 16 ) );
}

void MainWindow::builderReset()
{
    m_trayIcon->setContextMenu( m_builder->contextMenu( "contextTray" ) );
}

#if defined( HAVE_OPENSSL )

void MainWindow::sslErrors( const QList<QSslError>& errors )
{
    SslErrorsDialog dialog( this );
    dialog.setErrors( errors );
    dialog.setCertificates( commandManager->certificateChain() );

    if ( dialog.exec() == QDialog::Accepted )
        commandManager->ignoreSslErrors();
}

#endif // defined( HAVE_OPENSSL )
