/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef DATAMANAGER_H
#define DATAMANAGER_H

#include <QObject>
#include <QList>

#include "rdb/rdb.h"
#include "datarows.h"
#include "staterows.h"
#include "updateevent.h"
#include "preferencesdata.h"

class Command;
class Reply;
class ReplyLine;

/**
* A table of users.
*/
typedef RDB::SimpleTable<UserRow> UsersTable;

/**
* A table of project members.
*/
typedef RDB::CrossTable<MemberRow> MembersTable;

/**
* A table of issue types.
*/
typedef RDB::SimpleTable<TypeRow> TypesTable;

/**
* A table of attribute definitions.
*/
typedef RDB::ChildTable<AttributeRow> AttributesTable;

/**
* A table of projects.
*/
typedef RDB::SimpleTable<ProjectRow> ProjectsTable;

/**
* A table of folders.
*/
typedef RDB::ChildTable<FolderRow> FoldersTable;

/**
* A table of issues.
*/
typedef RDB::ChildTable<IssueRow> IssuesTable;

/**
* A table of attribute values.
*/
typedef RDB::CrossTable<ValueRow> ValuesTable;

/**
* A table of comments.
*/
typedef RDB::ChildTable<CommentRow> CommentsTable;

/**
* A table of attachments.
*/
typedef RDB::ChildTable<AttachmentRow> AttachmentsTable;

/**
* A table of value changes.
*/
typedef RDB::ChildTable<ChangeRow> ChangesTable;

/**
* An internal table storing the state of the folders.
*/
typedef RDB::SimpleTable<FolderState> FoldersStateTable;

/**
* An internal table storing the state of the issues.
*/
typedef RDB::SimpleTable<IssueState> IssuesStateTable;

/**
* Class for storing data retrieved from the Webissues server.
*
* This class stores data tables which reflect the structure of the server's database.
* It also handles commands for updating the data and notifies about data updates.
*
* Each command updates a logical unit of data. The three global units are users with
* their membership, types with their attributes and projects with folders. In addition,
* each folder unit contains issues from that folder and their attribute values.
* Each issue unit contains a signle issue and its attributes, comments, attachments
* and history of changes.
*
* Each unit of data is updated independently. The folder and issue units are updated
* incrementally, i.e. only the changes since the last update are retrieved.
*
* Only a limited number of issue units is stored. The least recently used issues are
* removed from memory when the limit is exceeded. To prevent an issue from being removed
* you have to lock it and unlock it when you no longer need it.
*
* Folder units are cached in data files. When the folder is updated for the first time,
* its previous contents is automatically read from the cache and only the changed issues
* are retrieved from the server.
*
* The instance of this class is available using the dataManager global variable.
* It is created and owned by the ConnectionManager.
*/
class DataManager : public QObject
{
    Q_OBJECT
public:
    /**
    * Default constructor.
    */
    DataManager();

    /**
    * Destructor.
    */
    ~DataManager();

public:
    /**
    * Return the table of users.
    */
    const UsersTable* users() const { return &m_users; }

    /**
    * Return the table of project members.
    */
    const MembersTable* members() const { return &m_members; }

    /**
    * Return the table of issue types.
    */
    const TypesTable* types() const { return &m_types; }

    /**
    * Return the table of attribute definitions.
    */
    const AttributesTable* attributes() const { return &m_attributes; }

    /**
    * Return the table of projects.
    */
    const ProjectsTable* projects() const { return &m_projects; }

    /**
    * Return the table of folders.
    */
    const FoldersTable* folders() const { return &m_folders; }

    /**
    * Return the table of issues.
    */
    const IssuesTable* issues() const { return &m_issues; }

    /**
    * Return the table of attribute values.
    */
    const ValuesTable* values() const { return &m_values; }

    /**
    * Return the table of comments.
    */
    const CommentsTable* comments() const { return &m_comments; }

    /**
    * Return the table of attachments.
    */
    const AttachmentsTable* attachments() const { return &m_attachments; }

    /**
    * Return the table of value changes.
    */
    const ChangesTable* changes() const { return &m_changes; }

    /**
    * Return the preferences of the current user.
    */
    const PreferencesData* preferences() const { return &m_preferences; }

    /**
    * Add a data observer.
    *
    * The observer receives UpdateEvent events when a unit of data is updated.
    */
    void addObserver( QObject* observer );

    /**
    * Remove the data observer.
    */
    void removeObserver( QObject* observer );

    /**
    * Check if the folder needs updating.
    *
    * This method compares the stamp of the last modification of the folder with
    * the stamp of the last update of its issues.
    *
    * @param folderId Identifier of the folder.
    *
    * @return @c True if the folder needs updating.
    */
    bool folderUpdateNeeded( int folderId );

    /**
    * Check if the issue needs updating.
    *
    * This method compares the stamp of the last modification of the issue with
    * the stamp of the last update of its details.
    *
    * @param issueId Identifier of the issue.
    *
    * @return @c True if the issue needs updating.
    */
    bool issueUpdateNeeded( int issueId );

    /**
    * Lock the issue to prevent removing it from the cache.
    *
    * @param issueId Identifier of the issue to lock.
    */
    void lockIssue( int issueId );

    /**
    * Release a previous issue lock.
    *
    * When the lock count of the issue is zero, it can be removed from memory.
    *
    * @param issueId Identifier of the issue to unlock.
    */
    void unlockIssue( int issueId );

    /**
    * Set the watch state of the folder.
    */
    void setFolderWatchState( int folderId, WatchState state );

    /**
    * Return the watch state of the folder.
    */
    WatchState folderWatchState( int folderId );

    /**
    * Set the watch stamp of the issue to the current stamp.
    */
    void updateIssueWatchStamp( int issueId );

    /**
    * Reset the watch stamp of the issue to zero.
    */
    void resetIssueWatchStamp( int issueId );

    /**
    * Calculate the watch state of the issue.
    *
    * The state is based on the watch stamp and the state of the folder.
    */
    WatchState issueWatchState( int issueId, bool ignoreFolder = false );

    /**
    * Return the notification state of the folder.
    */
    bool isFolderNotify( int folderId );

    /**
    * Return the notification state of the issue.
    */
    bool isIssueNotify( int issueId );

    /**
    * Create a command for updating users and their membership.
    */
    Command* updateUsers();

    /**
    * Create a command for updating types and attributes.
    */
    Command* updateTypes();

    /**
    * Create a command for updating projects and folders.
    */
    Command* updateProjects();

    /**
    * Create a command for updating the given folder.
    */
    Command* updateFolder( int folderId );

    /**
    * Create a command for updating the given issue.
    */
    Command* updateIssue( int issueId );

    /**
    * Create a command for updating the user preferences.
    */
    Command* updatePreferences();

    /**
    * Create a command for updating the state of notifications.
    */
    Command* updateNotifications();

    /**
    * Find the issue containing the given item.
    *
    * @param itemId Identifier of an issue, comment or attachment.
    *
    * @return Identifier of the issue or 0 if the item wasn't found.
    */
    int findItem( int itemId );

private:
    void notifyObservers( UpdateEvent::Unit unit, int id = 0 );

    FolderState* folderState( int folderId );
    IssueState* issueState( int issueId );

    void flushIssueCache();

    UserRow* readUserRow( const ReplyLine& line );
    MemberRow* readMemberRow( const ReplyLine& line );
    TypeRow* readTypeRow( const ReplyLine& line );
    AttributeRow* readAttributeRow( const ReplyLine& line );
    ProjectRow* readProjectRow( const ReplyLine& line );
    FolderRow* readFolderRow( const ReplyLine& line );
    IssueRow* readIssueRow( const ReplyLine& line );
    ValueRow* readValueRow( const ReplyLine& line );
    CommentRow* readCommentRow( const ReplyLine& line );
    AttachmentRow* readAttachmentRow( const ReplyLine& line );
    ChangeRow* readChangeRow( const ReplyLine& line );

    void updateFolderCache( int folderId );
    void readFolderCache( int folderId );
    void saveFolderCache();
    void writeFolderCache( int folderId );

    void loadWatchCache();
    void saveWatchCache();

private slots:
    void updateUsersReply( const Reply& reply );
    void updateTypesReply( const Reply& reply );
    void updateProjectsReply( const Reply& reply );
    void updateFolderReply( const Reply& reply );
    void updateIssueReply( const Reply& reply );
    void updatePreferencesReply( const Reply& reply );
    void updateNotificationsReply( const Reply& reply );

private:
    UsersTable m_users;
    MembersTable m_members;
    TypesTable m_types;
    AttributesTable m_attributes;
    ProjectsTable m_projects;
    FoldersTable m_folders;
    IssuesTable m_issues;
    ValuesTable m_values;
    CommentsTable m_comments;
    AttachmentsTable m_attachments;
    ChangesTable m_changes;

    FoldersStateTable m_foldersState;
    IssuesStateTable m_issuesState;

    PreferencesData m_preferences;

    QList<QObject*> m_observers;
};

/**
* Global pointer used to access the DataManager.
*/
extern DataManager* dataManager;

#endif
