/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef FREQUENCYINFO_H
#define FREQUENCYINFO_H

#include <QTime>
#include <QString>

/**
* Structure storing notification frequency definition.
*/
class FrequencyInfo
{
public:
    /**
    * Type of frequency definition.
    */
    enum Type
    {
        /** Invalid frequency definition. */
        Invalid,
        /** Send notifications immediately. */
        Immediately,
        /** Send notifications daily at given hour. */
        Daily,
        /** Send notifications weekly at given day and hour. */
        Weekly
    };

public:
    /**
    * Default constructor.
    *
    * Construct an invalid frequency definition.
    */
    FrequencyInfo();

    /**
    * Destructor.
    */
    ~FrequencyInfo();

public:
    /**
    * Check if the definition is valid.
    *
    * @return @c true if the definition is valid.
    */
    bool isValid() const { return m_type != Invalid; }

    /**
    * Set the frequency type.
    */
    void setType( Type type ) { m_type = type; }

    /**
    * Return the frequency type.
    */
    Type type() const { return m_type; }

    /**
    * Set the day of the week (from 0 - Sunday to 6 - Saturday).
    */
    void setDay( int day ) { m_day = day; }

    /**
    * Return the day of the week.
    */
    int day() const { return m_day; }

    /**
    * Set the time of the day.
    */
    void setTime( const QTime& time ) { m_time = time; }

    /**
    * Return the time of the day.
    */
    const QTime& time() const { return m_time; }

    /**
    * Format the frequency as a string in WebIssues format.
    */
    QString toString() const;

public:
    /**
    * Parse the frequency string in WebIssues format.
    */
    static FrequencyInfo fromString( const QString& text );

private:
    Type m_type;
    int m_day;
    QTime m_time;
};

#endif
