/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "typesview.h"

#include <QTreeView>
#include <QAction>
#include <QMenu>
#include <QMessageBox>

#include "models/tableitemmodel.h"
#include "models/tablemodels.h"
#include "models/treeviewsettings.h"
#include "models/treeviewhelper.h"
#include "data/rdb/tableiterators.h"
#include "data/datamanager.h"
#include "data/connectioninfo.h"
#include "data/updatebatch.h"
#include "dialogs/typedialogs.h"
#include "xmlui/builder.h"
#include "connectionmanager.h"
#include "iconloader.h"

using namespace WebIssues;

/* TRANSLATOR WebIssues::TypesView */

TypesView::TypesView( QObject* parent, QWidget* parentWidget ) : View( parent )
{
    m_systemAdmin = connectionManager->connectionInfo()->access() == AdminAccess;

    QAction* action;

    action = new QAction( IconLoader::icon( "file-reload" ), tr( "&Update Types" ), this );
    action->setShortcut( QKeySequence::Refresh );
    connect( action, SIGNAL( triggered() ), this, SLOT( updateTypes() ) );
    setAction( "updateTypes", action );

    if ( m_systemAdmin ) {
        action = new QAction( IconLoader::icon( "type-new" ), tr( "Add &Type..." ), this );
        connect( action, SIGNAL( triggered() ), this, SLOT( addType() ) );
        setAction( "addType", action );

        action = new QAction( IconLoader::icon( "attribute-new" ), tr( "Add &Attribute..." ), this );
        action->setShortcut( QKeySequence::New );
        connect( action, SIGNAL( triggered() ), this, SLOT( addAttribute() ) );
        setAction( "addAttribute", action );

        action = new QAction( IconLoader::icon( "edit-rename" ), tr( "&Rename..." ), this );
        action->setShortcut( tr( "F2" ) );
        connect( action, SIGNAL( triggered() ), this, SLOT( editRename() ) );
        setAction( "editRename", action );

        action = new QAction( IconLoader::icon( "edit-modify" ), tr( "&Modify Attribute..." ), this );
        connect( action, SIGNAL( triggered() ), this, SLOT( editModify() ) );
        setAction( "editModify", action );

        action = new QAction( IconLoader::icon( "edit-delete" ), tr( "&Delete" ), this );
        action->setShortcut( QKeySequence::Delete );
        connect( action, SIGNAL( triggered() ), this, SLOT( editDelete() ) );
        setAction( "editDelete", action );
    }

    setTitle( "menuMain", tr( "Ty&pes" ) );
    setTitle( "menuEdit", tr( "&Edit" ) );

    loadXmlFile( ":/resources/typesview.xml" );

    m_list = new QTreeView( parentWidget );
    m_list->setRootIsDecorated( true );
    m_list->setSortingEnabled( true );
    m_list->setContextMenuPolicy( Qt::CustomContextMenu );

    connect( m_list, SIGNAL( customContextMenuRequested( const QPoint& ) ),
        this, SLOT( contextMenu( const QPoint& ) ) );
    connect( m_list, SIGNAL( activated( const QModelIndex& ) ),
        this, SLOT( activated( const QModelIndex& ) ) );

    setMainWidget( m_list );

    setViewerSizeHint( QSize( 700, 500 ) );

    setAccess( NormalAccess );
}

TypesView::~TypesView()
{
    TreeViewSettings settings;
    settings.openTypesTree();

    settings.saveColumnWidths( TreeViewHelper::readColumnWidths( m_list ) );
    settings.saveExpandedNodes( TreeViewHelper::readExpandedNodes( m_list ) );
}

void TypesView::initialUpdate()
{
    m_model = new TableItemModel( this );
    m_model->setRootTableModel( new TypesTableModel( m_model ), dataManager->types()->index() );
    m_model->addChildTableModel( new AttributesTableModel( m_model ), dataManager->attributes()->parentIndex() );

    TreeViewSettings settings;
    settings.openTypesTree();

    m_model->setColumns( settings.loadColumns() );

    m_list->setModel( m_model );

    m_list->sortByColumn( 0, Qt::AscendingOrder );

    TreeViewHelper::applyColumnWidths( m_list, settings.loadColumnWidths() );
    TreeViewHelper::applyExpandedNodes( m_list, settings.loadExpandedNodes() );

    setCaption( tr( "Issue Types" ) );

    connect( m_list->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ),
        this, SLOT( updateActions() ) );
    connect( m_model, SIGNAL( layoutChanged() ), this, SLOT( updateActions() ) );

    updateActions();
}

void TypesView::updateActions()
{
    m_selectedTypeId = 0;
    m_selectedAttributeId = 0;
    m_currentTypeId = 0;

    QModelIndex index = selectedIndex();
    if ( index.isValid() ) {
        int level = m_model->data( index, TableItemModel::LevelRole ).toInt();
        int rowId = m_model->data( index, TableItemModel::RowIdRole ).toInt();
        if ( level == 0 ) {
            m_selectedTypeId = rowId;
            m_currentTypeId = rowId;
        } else {
            m_selectedAttributeId = rowId;
            const AttributeRow* attribute = dataManager->attributes()->find( rowId );
            if ( attribute )
                m_currentTypeId = attribute->typeId();
        }
    }

    if ( m_systemAdmin ) {
        action( "addAttribute" )->setEnabled( m_currentTypeId != 0 );
        action( "editRename" )->setEnabled( m_selectedTypeId != 0 || m_selectedAttributeId != 0 );
        action( "editModify" )->setEnabled( m_selectedAttributeId != 0 );
        action( "editDelete" )->setEnabled( m_selectedTypeId != 0 || m_selectedAttributeId != 0 );
    }
}

void TypesView::updateTypes()
{
    if ( !isUpdating() ) {
        UpdateBatch* batch = new UpdateBatch();
        batch->updateTypes();

        executeUpdate( batch );
    }
}

void TypesView::addType()
{
    if ( m_systemAdmin ) {
        AddTypeDialog dialog( mainWidget() );
        dialog.exec();
    }
}

void TypesView::addAttribute()
{
    if ( m_currentTypeId != 0 && m_systemAdmin ) {
        AddAttributeDialog dialog( m_currentTypeId, mainWidget() );
        if ( dialog.exec() == QDialog::Accepted )
            m_list->expand( selectedIndex() );
    }
}

void TypesView::editRename()
{
    if ( m_selectedTypeId != 0 && m_systemAdmin ) {
        RenameTypeDialog dialog( m_selectedTypeId, mainWidget() );
        dialog.exec();
    } else if ( m_selectedAttributeId != 0 && m_systemAdmin ) {
        RenameAttributeDialog dialog( m_selectedAttributeId, mainWidget() );
        dialog.exec();
    }
}

void TypesView::editModify()
{
    if ( m_selectedAttributeId != 0 && m_systemAdmin ) {
        ModifyAttributeDialog dialog( m_selectedAttributeId, mainWidget() );
        dialog.exec();
    }
}

void TypesView::editDelete()
{
    if ( m_selectedTypeId != 0 && m_systemAdmin ) {
        RDB::TableConstIterator<FolderRow> it( dataManager->folders() );
        while ( it.next() ) {
            const FolderRow* folder = it.get();
            if ( folder->typeId() == m_selectedTypeId ) {
                const TypeRow* type = dataManager->types()->find( m_selectedTypeId );
                QString name = type ? type->name() : QString();
                QMessageBox::warning( mainWidget(), tr( "Delete Type" ),
                    tr( "<qt>Type <b>%1</b> cannot be deleted because it is has associated folders.</qt>" ).arg( name ) );
                return;
            }
        }
        DeleteTypeDialog dialog( m_selectedTypeId, mainWidget() );
        dialog.exec();
    } else if ( m_selectedAttributeId != 0 && m_systemAdmin ) {
        DeleteAttributeDialog dialog( m_selectedAttributeId, mainWidget() );
        dialog.exec();
    }
}

void TypesView::contextMenu( const QPoint& pos )
{
    QModelIndex index = m_list->indexAt( pos );

    QString menuName;
    if ( index.isValid() ) {
        int level = m_model->data( index, TableItemModel::LevelRole ).toInt();
        if ( level == 0 )
            menuName = "contextType";
        else
            menuName = "contextAttribute";
    } else {
        menuName = "contextNull";
    }

    QMenu* menu = builder()->contextMenu( menuName );
    if ( menu )
        menu->exec( m_list->viewport()->mapToGlobal( pos ) );
}

void TypesView::activated( const QModelIndex& index )
{
    if ( index.isValid() && m_systemAdmin ) {
        int level = m_model->data( index, TableItemModel::LevelRole ).toInt();
        int rowId = m_model->data( index, TableItemModel::RowIdRole ).toInt();

        if ( level == 1 ) {
            ModifyAttributeDialog dialog( rowId, mainWidget() );
            dialog.exec();
        }
    }
}

QModelIndex TypesView::selectedIndex()
{
    if ( !m_list->selectionModel() )
        return QModelIndex();

    QModelIndexList selection = m_list->selectionModel()->selectedRows();
    if ( selection.isEmpty() )
        return QModelIndex();

    return selection.at( 0 );
}
