/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "watchesdialog.h"

#include <QLayout>
#include <QLabel>
#include <QHeaderView>
#include <QPushButton>
#include <QListWidget>
#include <QDialogButtonBox>

#include "data/datamanager.h"
#include "models/columnconditionsettings.h"
#include "models/treeviewsettings.h"
#include "models/tablemodelshelper.h"
#include "widgets/gradientwidget.h"
#include "filterdialog.h"
#include "viewmanager.h"
#include "iconloader.h"

using namespace WebIssues;

/* TRANSLATOR WebIssues::WatchesDialog */

WatchesDialog::WatchesDialog( int folderId, QWidget* parent ) : QDialog( parent ),
    m_folderId( folderId )
{
    setWindowTitle( tr( "Folder Watches" ) );

    QVBoxLayout* topLayout = new QVBoxLayout( this );
    topLayout->setMargin( 0 );
    topLayout->setSpacing( 0 );

    QWidget* promptWidget = new GradientWidget( this );
    topLayout->addWidget( promptWidget );

    QHBoxLayout* promptLayout = new QHBoxLayout( promptWidget );
    promptLayout->setSpacing( 10 );

    QLabel* promptPixmap = new QLabel( promptWidget );
    promptPixmap->setPixmap( IconLoader::pixmap( "folder-watch", 22 ) );
    promptLayout->addWidget( promptPixmap, 0 );

    QLabel* promptLabel = new QLabel( promptWidget );
    promptLabel->setWordWrap( true );
    promptLabel->setMinimumWidth( 250 );
    promptLayout->addWidget( promptLabel, 1 );

    QFrame* separator = new QFrame( this );
    separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    topLayout->addWidget( separator );

    QVBoxLayout* mainLayout = new QVBoxLayout();
    mainLayout->setMargin( 9 );
    mainLayout->setSpacing( 6 );
    topLayout->addLayout( mainLayout );

    m_list = new QListWidget( this );
    mainLayout->addWidget( m_list, 1 );

    mainLayout->addSpacing( 10 );

    QDialogButtonBox* buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Cancel, Qt::Horizontal, this );
    mainLayout->addWidget( buttonBox );

    buttonBox->button( QDialogButtonBox::Ok )->setText( tr( "&OK" ) );
    buttonBox->button( QDialogButtonBox::Cancel )->setText( tr( "&Cancel" ) );

    connect( buttonBox, SIGNAL( accepted() ), this, SLOT( accept() ) );
    connect( buttonBox, SIGNAL( rejected() ), this, SLOT( reject() ) );

    QPushButton* newButton = buttonBox->addButton( tr( "&New Filter..." ), QDialogButtonBox::ResetRole );

    connect( newButton, SIGNAL( clicked() ), this, SLOT( newFilter() ) );

    const FolderRow* folder = dataManager->folders()->find( folderId );
    QString name = folder ? folder->name() : QString();
    int typeId = folder ? folder->typeId() : 0;

    promptLabel->setText( tr( "Select active watches for folder <b>%1</b>:" ).arg( name ) );
    promptLabel->setFixedHeight( promptLabel->heightForWidth( promptLabel->minimumWidth() ) );

    m_settings = new ColumnConditionSettings();
    m_settings->openIssueFilters( typeId );

    populateList();

    QStringList watches = m_settings->loadWatches( folderId );
    setSelectedWatches( watches );

    resize( 100, 300 );
}

WatchesDialog::~WatchesDialog()
{
    delete m_settings;
}

void WatchesDialog::accept()
{
    QStringList watches = selectedWatches();
    m_settings->saveWatches( m_folderId, watches );

    viewManager->postViewEvent( NULL, ViewEvent::RecalculateWatches, m_folderId );

    QDialog::accept();
}

void WatchesDialog::populateList()
{
    m_list->clear();

    m_list->addItem( tr( "(All Issues)" ) );

    QStringList filters = m_settings->filterNames();

    if ( filters.count() > 0 ) {
        filters.sort();
        m_list->addItems( filters );
    }
}

void WatchesDialog::setSelectedWatches( const QStringList& watches )
{
    for ( int i = 0; i < m_list->count(); i++ ) {
        QString name = ( i == 0 ) ? QString() : m_list->item( i )->text();
        if ( watches.contains( name ) )
            m_list->item( i )->setCheckState( Qt::Checked );
        else
            m_list->item( i )->setCheckState( Qt::Unchecked );
    }
}

QStringList WatchesDialog::selectedWatches()
{
    QStringList watches;

    for ( int i = 0; i < m_list->count(); i++ ) {
        if ( m_list->item( i )->checkState() == Qt::Checked ) {
            QString name = ( i == 0 ) ? QString() : m_list->item( i )->text();
            watches.append( name );
        }
    }

    return watches;
}

void WatchesDialog::newFilter()
{
    const FolderRow* folder = dataManager->folders()->find( m_folderId );
    int typeId = folder ? folder->typeId() : 0;
    int projectId = folder ? folder->projectId() : 0;

    const TypeRow* type = dataManager->types()->find( typeId );
    QString name = type ? type->name() : QString();

    FilterDialog dialog( projectId, this );

    dialog.setPrompt( tr( "Create a new filter for folders of type <b>%1</b>:" ).arg( name ) );

    TreeViewSettings viewSettings;
    viewSettings.openIssuesList( typeId );

    dialog.setAvailableColumns( viewSettings.availableColumns() );
    dialog.setConditions( QList<ColumnCondition>() );

    dialog.setFilterMode( true );
    dialog.setExistingFilters( m_settings->filterNames() );

    if ( dialog.exec() == QDialog::Accepted ) {
        m_settings->saveFilter( dialog.filterName(), dialog.conditions() );

        QStringList watches = selectedWatches();
        populateList();
        setSelectedWatches( watches );

        viewManager->postViewEvent( NULL, ViewEvent::UpdateFilters, typeId ); 
    }
}
