/*****************************************************************************
* sdladump.c	WANPIPE(tm) Adapter Memeory Dump Utility.
*
* Author:	Gene Kozin	<genek@compuserve.com>
*
* Copyright:	(c) 1995-1996 Sangoma Technologies Inc.
*
*		This program is free software; you can redistribute it and/or
*		modify it under the terms of the GNU General Public License
*		as published by the Free Software Foundation; either version
*		2 of the License, or (at your option) any later version.
* ----------------------------------------------------------------------------
* Nov 29, 1996	Gene Kozin	Initial version based on WANPIPE configurator.
*****************************************************************************/

/*****************************************************************************
* Usage:
*   sdladump [{switches}] {device} [{offset} [{length}]]
*
* Where:
*   {device}	WANPIPE device name in /proc/net/wanrouter directory
*   {offset}	address of adapter's on-board memory. Default is 0.
*   {length}	size of memory to dump. Default is 0x100 (256 bytes).
*   {switches}	one of the following:
*			-v	verbose mode
*			-h | -?	show help
*****************************************************************************/

#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>

#include <linux/wanpipe.h>	/* WANPIPE user API definitions */

/****** Defines *************************************************************/

#ifndef	min
#define	min(a,b)	(((a)<(b))?(a):(b))
#endif

/* Error exit codes */
enum ErrCodes
{
	ERR_SYSTEM = 1,		/* System error */
	ERR_SYNTAX,		/* Command line syntax error */
	ERR_LIMIT
};

/* Command line parsing stuff */
#define ARG_SWITCH	'-'	/* switch follows */
#define	SWITCH_GETHELP	'h'	/* show help screen */
#define	SWITCH_ALTHELP	'?'	/* same */
#define	SWITCH_VERBOSE	'v'	/* enable verbose output */

/* Defaults */
#define	DEFAULT_OFFSET	0
#define	DEFAULT_LENGTH	0x100

/****** Data Types **********************************************************/

/****** Function Prototypes *************************************************/

int arg_proc	(int argc, char* argv[]);
int do_dump	(int argc, char* argv[]);
void show_dump	(char* buf, unsigned long len, unsigned long addr);
void show_error	(int err);
int hexdump	(char* str, unsigned char* data, int length, int limit);

extern	int close (int);

/****** Global Data *********************************************************/

/*
 * Strings & message tables.
 */
char progname[] = "sdladump";
char wandev_dir[] = "/proc/net/wanrouter";	/* location of WAN devices */

char banner[] =
	"WANPIPE Memory Dump Utility. v3.0.0 "
	"(c) 1995-1996 Sangoma Technologies Inc."
;
char helptext[] =
	"Usage:\tsdladump [{switches}] {device} [{offset} [{length}]]\n"
	"\nWhere:"
	"\t{device}\tWANPIPE device name from /proc/net/wanrouter directory\n"
	"\t{offset}\taddress of adapter's on-board memory (default is 0)\n"
	"\t{length}\tsize of memory to dump (default is 256 bytes)\n"
	"\t{switches}\tone of the following:\n"
	"\t\t\t-v\tverbose mode\n"
	"\t\t\t-h|?\tshow this help\n"
;

char* err_messages[] =		/* Error messages */
{
	"Invalid command line syntax",	/* ERR_SYNTAX */
	"Unknown error code",		/* ERR_LIMIT */
};

enum				/* execution modes */
{
	DO_DUMP,
	DO_HELP
} action;
int verbose;			/* verbosity level */

/****** Entry Point *********************************************************/

int main (int argc, char *argv[])
{
	int err = 0;	/* return code */
	int skip;

	/* Process command line switches */
	for (skip = 0, --argc, ++argv;
	     argc && (**argv == ARG_SWITCH);
	     argc -= skip, argv += skip)
	{
		skip = arg_proc(argc, argv);
		if (skip == 0)
		{
			err = ERR_SYNTAX;
			show_error(err);
			break;
		}
	}

	/* Perform requested action */
	if (verbose) puts(banner);
	if (!err) switch (action)
	{
	case DO_DUMP:
		err = do_dump(argc, argv);
		break;

	default:
		err = ERR_SYNTAX;
	}
	if (err == ERR_SYNTAX) puts(helptext);
	return err;
}

/*============================================================================
 * Process command line.
 *	Return number of successfully processed arguments, or 0 in case of
 *	syntax error.
 */
int arg_proc (int argc, char *argv[])
{
	int cnt = 0;

	switch (argv[0][1])
	{
	case SWITCH_GETHELP:
	case SWITCH_ALTHELP:
		action = DO_HELP;
		cnt = 1;
		break;

	case SWITCH_VERBOSE:
	  	verbose = 1;
		cnt = 1;
		break;
	}
	return cnt;
}

/*============================================================================
 * Dump adapter memory.
 *	argv[0]: device name
 *	argv[1]: offset
 *	argv[2]: length
 */
int do_dump (int argc, char *argv[])
{
	int err = 0;
	int dev;
	char filename[sizeof(wandev_dir) + WAN_DRVNAME_SZ + 2];
	sdla_dump_t dump;

	if (!argc || (strlen(argv[0]) > WAN_DRVNAME_SZ))
	{
		show_error(ERR_SYNTAX);
		return ERR_SYNTAX;
	}

	dump.magic  = WANPIPE_MAGIC;
	dump.offset = (argc > 1) ? strtoul(argv[1], NULL, 0) : DEFAULT_OFFSET;
	dump.length = (argc > 2) ? strtoul(argv[2], NULL, 0) : DEFAULT_LENGTH;
	if (!dump.length)
	{
		show_error(ERR_SYNTAX);
		return ERR_SYNTAX;
	}

	if (verbose) printf(
		"Dumping %lu bytes from offset %lu from device %s ...\n",
		dump.length, dump.offset, argv[0])
	;

	dump.ptr = malloc(dump.length);
	if (dump.ptr == NULL)
	{
		show_error(ERR_SYSTEM);
		return ERR_SYSTEM;
	}

	sprintf(filename, "%s/%s", wandev_dir, argv[0]);
	dev = open(filename, O_RDONLY);
	if ((dev < 0) || (ioctl(dev, WANPIPE_DUMP, &dump) < 0))
	{
		err = ERR_SYSTEM;
		show_error(err);
	}
	else show_dump(dump.ptr, dump.length, dump.offset);
	if (dev >= 0) close(dev);
	free(dump.ptr);
	return err;
}

/*============================================================================
 * Print hex memory dump to standard output.
 *	argv[0]: device name
 *	argv[1]: offset
 *	argv[2]: length
 */
void show_dump (char* buf, unsigned long len, unsigned long addr)
{
	char str[80];
	int cnt;

	if (len > 16)
	{
		/* see if ajustment is needed and adjust to 16-byte
		 * boundary, if necessary.
		 */ 
		cnt = 16 - addr % 16;
		if (cnt)
		{
			printf("%05lX: ", addr);
			hexdump(str, buf, cnt, 16);
			puts(str);
		}
	}
	else cnt = 0;

	while (cnt < len)
	{
		printf("%05lX: ", addr + cnt);
		cnt += hexdump(str, &buf[cnt], min(16, len - cnt), 16);
		puts(str);
	}
}

/*============================================================================
 * Dump data into a string in the following format:
 *	XX XX XX XX XX XX XX XX XX XX XX XX XX XX XX XX AAAAAAAAAAAAAAAA
 *
 * Return number of bytes dumped.
 * NOTE: string buffer must be at least (limit * 4 + 2) bytes long.
 */
int hexdump (char* str, unsigned char* data, int length, int limit)
{
	int i, n;

	n = min(limit, length);
	if (n)
	{
		for (i = 0; i < n; ++i)
			str += sprintf(str, "%02X ", data[i])
		;
		for (i = 0; i < limit - n; ++i)
			str += sprintf(str, "   ")
		;
		str += sprintf(str, " ");
		for (i = 0; i < n; ++i) str += sprintf(
			str, "%c", (isprint(data[i])) ? data[i] : '.')
		;
		for (i = 0; i < limit - n; ++i)
			str += sprintf(str, " ")
		;
	}
	*str = '\0';
	return n;
}

/*============================================================================
 * Show error message.
 */
void show_error (int err)
{
	if (err == ERR_SYSTEM) fprintf(stderr, "%s: SYSTEM ERROR %d: %s!\n",
		progname, errno, strerror(errno))
	;
	else fprintf(stderr, "%s: %s!\n", progname,
		err_messages[min(err, ERR_LIMIT) - 2])
	;
}

/****** End *****************************************************************/
