/*
 * twclock:  A world clock implemented with openMotif widgets
 * Copyright (C) 1997 - 2006 Ted Williams WA0EIR
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have recieved a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge. MA 02139, USA.
 * See the COPYING file in this directory
 *
 * Version: 2.5 - Jan 2006
 */

/*
 * CALLBACKS FOR TWCLOCK 
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>

#include "twclock.h"
#include "cwlib.h"

char timepath[100];
Pixel saveColor;
Boolean newTimeFlag;

/*
 * CATCH INPUT EVENTS
 * This catches all input events in the GUI.  If it was a
 * button 3 press, then popup the menu, else go away.
 */
void input_event (Widget w, XtPointer client_data, XEvent *event, Boolean *cont)
{
   Widget popup_menu = (Widget) client_data;

   if(event->type == ButtonPress && event->xbutton.button == Button3)
   {
      XmMenuPosition(popup_menu, &(event->xbutton));
      XtManageChild(popup_menu); 
   }
}


/*
 * POPUP MENU CALLBACK
 * All popup menu button presses come here.  The switch checks for
 * which menu button was pressed and either sets the TZ variable 
 * to Local, GMT or, in the case of the Others: button, pops
 * up the file selection box dialog.  The tzset() forces the
 * new value in TZ to be recognized.  The clock will adjust
 * to the new time on the next update.
 */
void popup_cb(Widget w, XtPointer client_data, XtPointer cbs)
{
   int set_zone = (int) client_data;
   Widget zone_fsb;

   switch (set_zone)
   {
      case LOCAL_BTN:
         strcpy (timepath, "");
         doit = 1;     /* force the date label to be updated next time */
         break;

      case GMT_BTN:
         strcpy (timepath, "TZ=GMT");
         doit = 1;     /* force the date label to be updated next time */
         break;

      case OTHERS_BTN:
         zone_fsb = XmCreateFileSelectionDialog(clock_shell, "zone_fsb",
            (ArgList) NULL, 0);

         XtVaSetValues (XtParent(zone_fsb),
            XmNtitle, "Timezone Selection",
            NULL);

         XtVaSetValues (zone_fsb,
            XmNdialogStyle, XmDIALOG_MODELESS,
            NULL);

         /* Unmanage some fsb widgets */
         XtUnmanageChild (XmFileSelectionBoxGetChild (zone_fsb,
            XmDIALOG_FILTER_TEXT));

         XtUnmanageChild (XmFileSelectionBoxGetChild (zone_fsb,
            XmDIALOG_FILTER_LABEL));

         XtUnmanageChild (XmFileSelectionBoxGetChild (zone_fsb,
            XmDIALOG_HELP_BUTTON));

         /* Add callbacks for file selection box */
         XtAddCallback (zone_fsb, XmNokCallback, fsbOkCB,
               (XtPointer) NULL);

         XtAddCallback (zone_fsb, XmNcancelCallback, fsbCancelCB,
            (XtPointer) NULL);

         XtManageChild (zone_fsb);   /* Show the file selection box */
         break;

      case SET_TIMER_BTN:
         settimer();
         break;

      case ID_NOW_BTN:
         cw_set_volume (appRes.cwLevel);
         cw_set_send_speed (appRes.cwSpeed);
         cw_send_string ((const unsigned char *) appRes.cwStr);
         break;

      case QRT:
         exit(0);
         break;

      default:
         fprintf (stderr, "Error: invalid menu button\n");
         break;
   }
}
 

/*
 * GETTIME FUNCTION
 * A call to this function is hardcoded near the end of widget
 * creation.  At the end of this function, it registers a timeout
 * that forces itself to be called again in one second. And we're off!
 *
 * This function calls localtime(), which bases time on TZ.  The
 * time values are then set into the widgets.  No need to put the
 * same string in the date label 86400 times a day.  So, if the day
 * of the week value has changed, change the label string.  Or, if
 * doit is true, we update the label.  popup_cb() sets doit when
 * the time zone is changed, which requires updating the label.
 */
void  gettime ()
{
   time_t current;
   struct tm *timestruct;
   char  datestr[40];
   static int oldday;
   XmString labelstr;
   unsigned long time_delta = 1000;

   /* register the time out again */
   time_id=XtAppAddTimeOut(app,
      time_delta,
      (XtTimerCallbackProc)gettime,
      (XtPointer)NULL);

   putenv (timepath);                   /* put selected time in TZ */
   time (&current);
   timestruct = localtime (&current);   /* get localtime as per TZ */

   XtVaSetValues (hr_scale,  XmNvalue, timestruct->tm_hour, NULL);
   XtVaSetValues (min_scale, XmNvalue, timestruct->tm_min,  NULL);
   XtVaSetValues (sec_scale, XmNvalue, timestruct->tm_sec,  NULL);
   if ((timestruct->tm_mday != oldday) | doit)
   {
      strftime (datestr, sizeof(datestr), "%e %b %Y %Z", timestruct);
      labelstr = XmStringCreateLocalized (datestr);

      XtVaSetValues (date_label,
         XmNlabelString, labelstr,
         NULL);

      XmStringFree (labelstr);
      oldday = timestruct->tm_mday;
      doit = 0;
   }
}


/*
 * FSB OK CALLBACK
 * This function gets data from the fsb and builds a string of
 * the form TZ=pathname.  If the pathname ends with a "/",
 * then you didn't select a city in the fsb, so exit and leave
 * TZ alone.  Otherwise, set TZ to the new timezone and set doit.
 * This forces gettime() to use the new TZ and update the date label.
 */
void fsbOkCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   XmFileSelectionBoxCallbackStruct *fsbcbs =
      (XmFileSelectionBoxCallbackStruct *) cbs;

   int  len;
   char *path;

   XmStringGetLtoR (fsbcbs->value, XmFONTLIST_DEFAULT_TAG, &path);

   len = strlen (path);
   if (path[len-1] != '/')       /* If the path ends with /, then do nothing */
   {
      strcpy (timepath, "TZ=");  /* build the new TZ value */
      strcat (timepath, path);   /* and put it in timepath */
      doit = 1;                  /* force date label to be updated next time */

   }
   /* XtUnmanageChild (w); */
   XtDestroyWidget (XtParent (w)); /* Remove the fsb */
   XtFree (path);
}


/*
 * FSB CANCEL CALLBACK
 * Cancel button was pressed, so unmanage the file selection box
 */
void fsbCancelCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   XtDestroyWidget(XtParent (w));
}


/*
 * TBCHANGECB CALLBACK FUNCTION
 * Registers or unregisters the alarm timeout, depending on the state of the
 * toggle button.
 */
void TBchangeCB(Widget w, XtPointer cdata, XtPointer cbs)
{
   XmToggleButtonCallbackStruct *alarmCBS=(XmToggleButtonCallbackStruct *)cbs;

   count = 0;        /* reset flags */
   cwIDdone = False;

   switch(alarmCBS->set)
   {
      case True:
         flashFlag = True;
         XtVaGetValues (w,
            XmNselectColor, &saveColor,
            NULL);

         alarm_id = XtAppAddTimeOut(app,
            (unsigned long) (((60 * appRes.minutes) + appRes.seconds) * 1000),
            (XtTimerCallbackProc)timer,
            NULL);

         break;

      case False:
         /* Restore original selectColor */
         XtVaSetValues (w,
            XmNselectColor, saveColor,
            NULL);

         /* Remove the alarm time out*/
         XtRemoveTimeOut (alarm_id);

         /* if auto_id != 0, then one exists, so remove it too. */
         if (auto_id != 0)
         {
            XtRemoveTimeOut (auto_id);
         }
         /* unregister and re-register the time out again */
         /* Why? - How does time_id get changed when the toggle */
         /* button is unset? This hack removes and adds a very */
         /* short time_id, which has no effect on the gui, but */
         /* keeps timer_id valid. timer_id was moved from gettime */
         /* twclock.h so it can be seen from here */
         XtRemoveTimeOut (time_id);
         time_id=XtAppAddTimeOut(app,
            1L,
            (XtTimerCallbackProc)gettime,
            (XtPointer)NULL);
               break;
      default:
         fprintf (stderr, "Error: invalid toggle button state\n");
         break;
  }
} 

/*
 * ALARM TIMER FUNCTION
 * First called when the ID timer times out.  It will cw id, blink the
 * toggle button and beep as requested in the resource file.  It then
 * changes the timer delay to 1 sec and registers another timeout.
 * Unsetting the toggle button unregisters the timeouts.
 */
void timer (void)
{
   FILE *con;

   auto_id = 0;  /* so, TBchangeCB won't try to unregister it */

   if (appRes.autoReset == True)
   {
      if (count == 0)  /* reset alarm timeout the first time, so no time skew */
      {
         alarm_id = XtAppAddTimeOut(app,
            (unsigned long) (((60 * appRes.minutes) + appRes.seconds) * 1000),
            (XtTimerCallbackProc)timer,
            (XtPointer)NULL);
      }

      if (count >= appRes.autobeeps)
      {
         /* restore bg color */
         XtVaSetValues (call_toggleB,
            XmNselectColor, saveColor,
            NULL);

         /* reset flags and wait for next alarm timeout */
         count = 0;
         cwIDdone = False;
         flashFlag = True;
         return;
      }
      else
      {
         count++;  /* autoReset is True so count. */
      }
   }

   /* If we get here, popup the clock, in case its hidden 
    * Send cw id,flash and blink if we should and then
    * set a one second timeout
    */

   if ( appRes.cwID == True && cwIDdone == False)
   {
      cw_set_volume (appRes.cwLevel);
      cw_set_send_speed (appRes.cwSpeed);
      cw_send_string ((const unsigned char*) appRes.cwStr);
      cwIDdone = True;
   }

   XtPopup( clock_shell, XtGrabNone );

   if ((con = fopen ("/dev/console", "w")) == 0)
   {
      fprintf (stderr, "can't open console\n");
   }
   else
   {
      /* do beep? */
      if (appRes.beep == True)
      {
         fprintf (con, "\a");
      }
      fclose (con);
   }

   if (appRes.blink == True && flashFlag == True)
   {
      XtVaSetValues (call_toggleB,
         XmNselectColor, appRes.flashColor,
         NULL);
      flashFlag = False;
   }
   else
   {
      XtVaSetValues (call_toggleB,
         XmNselectColor, saveColor,
         NULL);
      flashFlag = True;
      
   }
   /* reset auto reset alarm for one second */
   auto_id=XtAppAddTimeOut(app,
      1000L,
      (XtTimerCallbackProc)timer,
      NULL);   
}


/*
 * SETTIMER FUNCTION
 * Create a dialog box to input new time
 * Add callbacks for min, sec, and close
 */
void settimer (void)
{
   char minutes[10], seconds[10];
   static Widget settimerDiag = NULL;
   static Widget timeTFs[2];
   Widget setForm, label1, frame1, timeForm, min_label, sec_label;
   Widget label2, frame2, optForm, autoResetTB, cwidTB, beepTB, blinkTB;
   Widget label3, frame3, cwForm, cwSpdScale, cwLvlScale;
   XmString xms;

   /* Convert the current minutes and seconds to strings */ 
   sprintf (minutes, "%d", appRes.minutes);
   sprintf (seconds, "%d", appRes.seconds);

   /* Lazy Instantiation */   
   if (settimerDiag != NULL)
   {
     /* reload times in case we closed via Cancel button */
     /* but got to remove the callbacks first */
      XtRemoveCallback (timeTFs[0], XmNmodifyVerifyCallback, timerModVerCB,
         (XtPointer) NULL);
      XtRemoveCallback (timeTFs[1], XmNmodifyVerifyCallback, timerModVerCB,
         (XtPointer) NULL);

      XtVaSetValues (timeTFs[0],
         XmNvalue, minutes,
         NULL);

      XtVaSetValues (timeTFs[1],
         XmNvalue, seconds,
         NULL);

      /* now add the callbacks again */
      XtAddCallback (timeTFs[0], XmNmodifyVerifyCallback, timerModVerCB,
         (XtPointer) NULL);
      XtAddCallback (timeTFs[1], XmNmodifyVerifyCallback, timerModVerCB,
         (XtPointer) NULL);

      XtManageChild (settimerDiag);
      return;
   }

   /* Create a message box dialog and set the dialog shell values */   
   settimerDiag = XmCreateMessageDialog (clock_shell, "settimer",
      (ArgList) NULL, 0);
   
   XtVaSetValues (XtParent(settimerDiag),
      XmNtitle, "Twclock Options",
      NULL); 

   XtVaSetValues (settimerDiag,          
      XmNdialogStyle, XmDIALOG_MODELESS,
      NULL);

   /* Don't want a help button */
   XtUnmanageChild (XmMessageBoxGetChild (settimerDiag,XmDIALOG_HELP_BUTTON));   
   /* Add form to the message box workspace for goodies */
   setForm = XtVaCreateWidget ("setForm", xmFormWidgetClass, settimerDiag,
      XmNverticalSpacing, 5,
      NULL);

   xms = XmStringCreateLocalized ("Timer");
   label1 = XtVaCreateManagedWidget ("label1", xmLabelWidgetClass,
      setForm,
      XmNlabelString, xms,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);
   XmStringFree(xms);

   frame1 = XtVaCreateWidget ("frame1", xmFrameWidgetClass, setForm,
      XmNshadowType, XmSHADOW_OUT,
      XmNshadowThickness, 3,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, label1,
      XmNleftAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);

   timeForm = XtVaCreateWidget ("timeForm", xmFormWidgetClass, frame1,
      NULL);

   min_label = XtVaCreateManagedWidget ("Minutes", xmLabelWidgetClass,
      timeForm,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);

   timeTFs[0] = XtVaCreateManagedWidget ("settimerDiag",
      xmTextFieldWidgetClass, timeForm,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, min_label,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_NONE,
      XmNvalue, minutes,
      XmNmaxLength, 4,
      XmNcolumns, 4,
      NULL);

   sec_label = XtVaCreateManagedWidget ("Seconds",
      xmLabelWidgetClass, timeForm,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, timeTFs[0],
      XmNbottomAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);
   
   timeTFs[1] = XtVaCreateManagedWidget ("settimerDiag",
      xmTextFieldWidgetClass, timeForm,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, sec_label,
      XmNbottomAttachment, XmATTACH_FORM,
      XmNrightAttachment, XmATTACH_NONE,
      XmNvalue, seconds,
      XmNmaxLength, 4,
      XmNcolumns, 4,
      NULL);

   xms = XmStringCreateLocalized ("Alarms");
   label2 = XtVaCreateManagedWidget ("label2", xmLabelWidgetClass,
      setForm,
      XmNlabelString, xms,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, timeForm,
      XmNleftAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);
   XmStringFree(xms);

   frame2 = XtVaCreateWidget ("frame1", xmFrameWidgetClass, setForm,
      XmNshadowType, XmSHADOW_OUT,
      XmNshadowThickness, 3,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, label2,
      XmNleftAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);

   optForm = XtVaCreateWidget ("optForm", xmFormWidgetClass, frame2,
      NULL);

   xms = XmStringCreateLocalized ("Auto Reset");
   autoResetTB = XtVaCreateManagedWidget ("autoResetTB",
      xmToggleButtonWidgetClass, optForm,
      XmNlabelString, xms,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);
   XmStringFree(xms);
   /* sync auoResetTB with autoReset resource */
   XmToggleButtonSetState (autoResetTB, appRes.autoReset, False);

   xms = XmStringCreateLocalized ("CW ID");
   cwidTB = XtVaCreateManagedWidget ("cwidTB",
      xmToggleButtonWidgetClass, optForm,
      XmNlabelString, xms,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, autoResetTB,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);
   XmStringFree(xms);
   /* sync cwidTB with cwid resource */
   XmToggleButtonSetState (cwidTB, appRes.cwID, False);

   xms = XmStringCreateLocalized ("Beep");
   beepTB = XtVaCreateManagedWidget ("beepTB", xmToggleButtonWidgetClass,
      optForm,
      XmNlabelString, xms,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, cwidTB,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);
   XmStringFree(xms);
   /* sync beepTB with beep resource */
   XmToggleButtonSetState (beepTB, appRes.beep, False);

   xms = XmStringCreateLocalized ("Blink");
   blinkTB = XtVaCreateManagedWidget ("blinkTB", xmToggleButtonWidgetClass,
      optForm,
      XmNlabelString, xms,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_WIDGET,
      XmNleftWidget, beepTB,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_NONE,
      NULL);
   XmStringFree(xms);
   /* sync blinkTB with blink resource */
   XmToggleButtonSetState (blinkTB, appRes.blink, False);

   xms = XmStringCreateLocalized ("CW ID");
   label3 = XtVaCreateManagedWidget ("label3", xmLabelWidgetClass,
      setForm,
      XmNlabelString, xms,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, optForm,
      XmNleftAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);
   XmStringFree(xms);

   frame3 = XtVaCreateWidget ("frame3", xmFrameWidgetClass, setForm,
      XmNshadowType, XmSHADOW_OUT,
      XmNshadowThickness, 3,
      XmNmarginWidth, 5,
      XmNmarginHeight, 5,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, label3,
      XmNleftAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);

   cwForm = XtVaCreateWidget ("cwForm", xmFormWidgetClass, frame3,
      NULL);

   xms = XmStringCreateLocalized ("CW Speed");
   cwSpdScale = XtVaCreateManagedWidget ("cwSptScale", xmScaleWidgetClass,
      cwForm,
      XmNtitleString, xms,
      XmNscaleMultiple, 5,
      XmNorientation, XmHORIZONTAL,
      XmNshowValue, True,
      XmNminimum, 10,
      XmNmaximum, 40,
      XmNtopAttachment, XmATTACH_FORM,
      XmNleftAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);
   XmStringFree(xms);
   /* sync cwSpdScale with cwSpeed resource */
   XmScaleSetValue (cwSpdScale, appRes.cwSpeed);

   xms = XmStringCreateLocalized ("CW Level");
   cwLvlScale = XtVaCreateManagedWidget ("cwLvlScale", xmScaleWidgetClass,
      cwForm,
      XmNtitleString, xms,
      XmNorientation, XmHORIZONTAL,
      XmNshowValue, True,
      XmNtopAttachment, XmATTACH_WIDGET,
      XmNtopWidget, cwSpdScale,
      XmNleftAttachment, XmATTACH_FORM,
      XmNbottomAttachment, XmATTACH_NONE,
      XmNrightAttachment, XmATTACH_FORM,
      NULL);
   XmStringFree(xms);
   /* sync cwLvlScale with cwLevel resource */
   XmScaleSetValue (cwLvlScale, appRes.cwLevel);

   /* Add the callbacks and 2 modifyVerify */

   XtAddCallback (settimerDiag, XmNokCallback, timerOkCB,
               (XtPointer) timeTFs);

   XtAddCallback (timeTFs[0], XmNmodifyVerifyCallback, timerModVerCB,
               (XtPointer) NULL);

   XtAddCallback (timeTFs[1], XmNmodifyVerifyCallback, timerModVerCB,
               (XtPointer) NULL);

   /*
    * TB's all use the same CB function. Client data is a pointer
    * to the resourse that needs tweaking.
    */
   XtAddCallback (autoResetTB, XmNvalueChangedCallback, optionsCB,
               (XtPointer) &appRes.autoReset);

   XtAddCallback (cwidTB, XmNvalueChangedCallback, optionsCB,
               (XtPointer) &appRes.cwID);

   XtAddCallback (beepTB, XmNvalueChangedCallback, optionsCB,
               (XtPointer) &appRes.beep);

   XtAddCallback (blinkTB, XmNvalueChangedCallback, optionsCB,
               (XtPointer) &appRes.blink);

   /*
    * Scales all use the same CB function. Client data is a pointer
    * to the resourse that needs tweaking.
    */
   XtAddCallback (cwSpdScale, XmNdragCallback, scalesCB,
               (XtPointer) &appRes.cwSpeed);

   XtAddCallback (cwSpdScale, XmNvalueChangedCallback, scalesCB,
               (XtPointer) &appRes.cwSpeed);

   XtAddCallback (cwLvlScale, XmNdragCallback, scalesCB,
               (XtPointer) &appRes.cwLevel);

   XtAddCallback (cwLvlScale, XmNvalueChangedCallback, scalesCB,
               (XtPointer) &appRes.cwLevel);

   /* Pop up the timer dialog */
   XtManageChild (timeForm);
   XtManageChild (frame1);
   XtManageChild (optForm);
   XtManageChild (frame2);
   XtManageChild (cwForm);
   XtManageChild (frame3);
   XtManageChild (setForm);
   XtManageChild (settimerDiag);
}


/*
 * Callbacks for setting the timer
 */


/*
 * OK button for settimer Dialog
 * If the call_toggleB is set, and the time was changed, then
 * calculate and register the new alarm time.
 */
void timerOkCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   Widget *wids = (Widget *) client_data;
   char *minStr, *secStr;
   int min, sec;

   if (newTimeFlag == True)
   {
      minStr = XmTextFieldGetString (wids[0]);
      secStr = XmTextFieldGetString (wids[1]);

      min = atoi(minStr);
      sec = atoi(secStr);

      /* just in case */
      min = min + (sec / 60);
      sec = sec % 60;

      appRes.minutes = min;
      appRes.seconds = sec;
   }

   if (XmToggleButtonGetState (call_toggleB) == True && newTimeFlag == True)
   {
      XtRemoveTimeOut (alarm_id);
      alarm_id = XtAppAddTimeOut(app,
         (unsigned long) (((60 * appRes.minutes) + appRes.seconds) * 1000),
         (XtTimerCallbackProc)timer, (XtPointer)NULL);
   }
   /* finally, clear the flag */
   newTimeFlag = False;
}


/*
 * modify/verify callback for time textfields
 * only allow numbers in the min and sec fields
 * if time is change is valid, set the the newTimeFlag
 */
void timerModVerCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   int i;
   XmTextVerifyCallbackStruct *pt = (XmTextVerifyCallbackStruct *) cbs;

   for (i=0; i < pt->text->length; i++)
   {
      if (isdigit (pt->text->ptr[i]))  /* allow only digits, beep if not */
      {
         newTimeFlag = True;
         return;
      }
      else
      {
         pt->doit = False;
         newTimeFlag = False;
         return;
      }
   }
}


/*
 * optionsCB value changed callback for option toggle buttons
 * maintains the values of alarm option resources
 * client_data is a pointer to appRes where state is stored
 */
void optionsCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   int *ptr = (int *)client_data;
   XmToggleButtonCallbackStruct *cbsptr = (XmToggleButtonCallbackStruct *) cbs;

   *ptr = cbsptr->set;
}


/*
 * scalesCB drag/valueChanged callbacks for CW ID scales
 * maintains the values of CW ID option resources
 * client_data is a pointer to appRes where state is stored
 */
void scalesCB (Widget w, XtPointer client_data, XtPointer cbs)
{
   int *ptr = (int *)client_data;
   XmScaleCallbackStruct *cbsptr = (XmScaleCallbackStruct *) cbs;

   *ptr = cbsptr->value;
   /* doesn't work...  cw_set_send_speed (appRes.cwSpeed); */
   /* but this does */
   cw_set_volume (appRes.cwLevel);  
}
