// Copyright (C) 2002 Ronan Collobert (collober@iro.umontreal.ca)
//                
//
// This file is part of Torch. Release II.
// [The Ultimate Machine Learning Library]
//
// Torch is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Torch is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Torch; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef QC_TRAINER_INC
#define QC_TRAINER_INC

#include "Trainer.h"
#include "QCMachine.h"
#include "QCCache.h"

namespace Torch {


/** Train a #QCMachine#.
    With the conventions of QCMachine.h,
    Q is given by the class QCCache (in #cache#)

    Options:
    \begin{tabular}{lcll}
      "unshrink"      & bool  &  unshrink or not unshrink              & [false] \\
      "max unshrink"  & int   &  maximal number of unshrinking         & [1] \\
      "iter shrink"   & int   &  minimal number of iterations to shrink& [100] \\
      "eps shrink"    & real  &  shrinking accuracy                    & [1E-4 (f)  1E-9 (d)] \\
      "end accuracy"  & real  &  end accuracy                          & [0.01] \\
      "iter message"  & int   &  number of iterations between messages & [1000]
    \end{tabular}


    Note: "iter shrink" must be carefully chosen.
    Read http://www.ai.mit.edu/projects/jmlr/papers/volume1/collobert01a/collobert01a.ps.gz
    for more details.

    @author Ronan Collobert (collober@iro.umontreal.ca)
    @see QCCache
    @see QCMachine
*/
class QCTrainer : public Trainer
{
  public:
    // ohhh boy, c'est la zone
    QCMachine *qcmachine;
    QCCache *cache;

    int n_unshrink;
    int n_max_unshrink;

    real *k_xi;
    real *k_xj;
  
    real old_alpha_xi;
    real old_alpha_xj;
    real current_error;

    int *active_var_new;
    int n_active_var_new;

    int l;                  // le nb de alphas

    bool deja_shrink;
    bool unshrink_mode;

    real *y;
    real *alpha;
    real *grad;

    real eps_shrink;
    real eps_fin;
    real eps_bornes;

    int n_active_var;
    int *active_var;
    int *not_at_bound_at_iter;
    int iter;
    int n_iter_min_to_shrink;
    int n_iter_message;

    char *status_alpha;
    real *Cup;
    real *Cdown;

    //-----

    ///
    QCTrainer(QCMachine *qcmachine_, DataSet *data_, QCCache *cache_);

    /** Train it...
        Before calling this function, #grad# in #qcmachine#
        must contain the gradient of QP(alpha) with respect
        to alpha = 0.

        ( = $beta$, with the conventions of QCMachine.h)

        Moreover #alpha# in #qcmachine# has to be zero.
    */
    void train(List *measurers);

    //-----

    void prepareToLaunch();
    void atomiseAll();

    bool bCompute();
    bool selectVariables(int *i, int *j);
    int checkShrinking(real bmin, real bmax);
    void shrink();
    void unShrink();

    void analyticSolve(int xi, int xj);
    void updateStatus(int i);  
    inline bool isNotUp(int i)   {  return(status_alpha[i] != 2); };
    inline bool isNotDown(int i) {  return(status_alpha[i] != 1); };

    virtual ~QCTrainer();
};


}

#endif
