/* -*- mode: c; c-file-style: "gnu" -*-
 * queue.c -- Queue abstraction laye
 * Copyright (C) 2002, 2003, 2004 Gergely Nagy <algernon@bonehunter.rulez.org>
 *
 * This file is part of Thy.
 *
 * Thy is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2 dated June, 1991.
 *
 * Thy is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
 * License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/** @file queue.c
 * Queue abstraction layer.
 *
 * The queue abstraction layer is meant to represent a consistent API
 * to handle the session queue in Thy. At the moment, it is very
 * primitive, as the queue is simply a static-size array. However, in
 * the future, when Thy will support a (virtually) infinite ammount of
 * concurrent sessions, it might become a dynamical list of some sort.
 *
 * The assumption made here (and elsewhere in Thy, like fabs.c) is
 * that file descriptors are assigned sequentally. A system where fds
 * are assigned randomly, this part of Thy will break horribly.
 */

#include <stdlib.h>

#include "compat/compat.h"

#include "nqueue.h"
#include "queue.h"
#include "options.h"
#include "session.h"
#include "thy.h"
#include "types.h"

/** @internal Static-size array of the queued sessions.
 * Its size is determined at run-time, then it is initialized by
 * queue_init().
 */
static session_t **queue;

/** Initialise the I/O queue.
 */
void
queue_init (void)
{
  queue = (session_t **)bhc_calloc ((size_t)(_THY_MAXCONN + 2),
				    sizeof (session_t *));
}

/** Retrieve a member of the I/O queue.
 * @param idx is the index of the sought member.
 *
 * @returns A pointer to the sought session or NULL on error.
 * @warning The return value is not a copy!
 */
session_t *
queue_get (long int idx)
{
  return queue[idx];
}

/** Add a new entry to the I/O queue.
 * When adding a new entry, no sanity checking is done - none
 * needed. The index is the file descriptor of the session, which is
 * guaranteed to be unique.
 *
 * @param session is the new entry to add.
 */
void
queue_add (session_t *session)
{
  if (session)
    {
      bhc_debug ("queue_add(%p) at %d", session, session->io.in);

      queue[session->io.in] = session;
      thy_nq_fd_control (session->io.in, THY_NQ_EVENT_INPUT, 1);
    }
}

/** Delete an entry from the I/O queue.
 * Delete a given entry from the I/O queue after killing the
 * associated session.
 *
 * @param idx is the index of the entry to be deleted.
 *
 * @see session_kill().
 */
void
queue_del (long int idx)
{
  if (queue[idx])
    {
      bhc_debug ("queue_del(%p) at %ld", queue[idx], idx);

      thy_nq_fd_control (queue[idx]->io.in, THY_NQ_EVENT_NONE, 0);
      thy_nq_fd_control (queue[idx]->io.out, THY_NQ_EVENT_NONE, 0);
      thy_nq_fd_control (queue[idx]->cgi.pipes.in[0],
			 THY_NQ_EVENT_NONE, 0);
      thy_nq_fd_control (queue[idx]->cgi.pipes.out[1],
			 THY_NQ_EVENT_NONE, 0);
      session_kill (queue[idx], 1);
    }
  queue[idx] = NULL;
}

/** Readd an entry to the I/O queue.
 * Saves some selected properties (fd, origin and tls_session) of a
 * session, kills it, then reinitialises it and adds it back to the
 * queue.
 *
 * @param idx is the index of the session to re-add.
 *
 * @returns Zero on success, or -1 if an error occurred.
 */
int
queue_keep (long int idx)
{
  session_t *session = queue[idx];
  int infd, outfd, sport, ssl, rc;
  char *origin;
  void *tls_data = NULL;

  if (!session)
    return -1;

  bhc_debug ("queue_keep(%p) at %ld", session, idx);

  infd = session->io.in;
  outfd = session->io.out;
  sport = session->port;
  ssl = session->ssl;
  rc = session->request_count;
  origin = bhc_strdup (session->origin);
#if THY_OPTION_TLS
  tls_data = (void *)session->tls_session;
#endif

  thy_nq_fd_control (session->cgi.pipes.in[0], THY_NQ_EVENT_NONE, 0);
  thy_nq_fd_control (session->cgi.pipes.out[1], THY_NQ_EVENT_NONE, 0);
  session_kill (session, 0);
  session = session_init (infd, outfd, sport, ssl, origin, tls_data);
  session->request_count = rc + 1;
  queue[idx] = session;
  thy_nq_fd_control (outfd, THY_NQ_EVENT_NONE, 0);
  thy_nq_fd_control (infd, THY_NQ_EVENT_INPUT, 1);

  free (origin);

  return 0;
}
