/*
 * dev.c - a module for device control
 * by Hirotsugu Kakugawa
 */
/*
 * Copyright (C) 1996-1997 Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_STDARG_H
#  include <stdarg.h>
#else
#  include <vararg.h>
#endif
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#include <X11/Xlib.h>
#include <X11/Xresource.h>
#include "libdvi29.h"
#include "defs.h"
#include "dev.h"
#include "resource.h"
#include "window.h"


Public DVI_DEVICE
dvi_device_create(int argc, char **argv, int is_cmdline)
{
  DVI_DEVICE  dev;

  if ((dev = DVI_DEVICE_ALLOC()) == NULL)
    return NULL;
  if (resource_create(dev, argc, argv, is_cmdline) < 0)
    return NULL;

  dev->device_polling        = DEV_poll;           /* see below & ui.c */

  dev->h_dpi                 = (double)RES(dev,dpi);
  dev->v_dpi                 = (double)RES(dev,dpi);

  dev->put_bitmap            = DEV_put_bitmap;     /* see below & ui.c */
  dev->put_rectangle         = DEV_put_rectangle;  /* see below & ui.c */

  dev->put_pixmap_rgb = NULL;
  dev->put_graymap    = NULL;
  switch (RES(dev,visual_eps)){
  default:
  case VISUAL_MONO:
    break;
  case VISUAL_GRAY:
    dev->put_graymap    = DEV_put_graymap;
    break;
  case VISUAL_RGB:
    dev->put_pixmap_rgb = DEV_put_pixmap_rgb;
    break;
  }
  if (x_visual_color() == 0){
    dev->put_graymap    = NULL;
    dev->put_pixmap_rgb = NULL;
  }

  dev->color_gray   = NULL;
  dev->bgcolor_gray = NULL;
  dev->color_rgb    = NULL;
  dev->bgcolor_rgb  = NULL;
  switch (RES(dev,visual_text)){
  default:
  case VISUAL_MONO:
    break;
  case VISUAL_GRAY:
    dev->color_gray   = DEV_color_gray;
    dev->bgcolor_gray = DEV_bgcolor_gray;
    break;
  case VISUAL_RGB:
    dev->color_rgb    = DEV_color_rgb;
    dev->bgcolor_rgb  = DEV_bgcolor_rgb;
    break;
  }
  if (x_visual_color() == 0){
    dev->color_gray = dev->bgcolor_gray = NULL;
    dev->color_rgb  = dev->bgcolor_rgb  = NULL;
  }
  
  dev->special_command_undef = DEV_special_command_undef; /* see below */
  dev->draw_ps_figures       = DEV_draw_ps_figures;       /* see below */
  dev->before_ps_figure      = DEV_before_ps_figure;      /* see below */
  dev->after_ps_figure       = DEV_after_ps_figure;       /* see below */

  dev->gs_program_path       = DEV_gs_program_path;       /* see below */
  dev->gs_timeout_value      = DEV_gs_timeout_value;      /* see below */
  dev->gs_timeout_give_up    = DEV_gs_timeout_giveup;     /* see below */

  dev->message_advice        = DEV_message_advice;        /* see below */
  dev->message_warning       = DEV_message_warning;       /* see below */
  dev->message_error         = DEV_message_error;         /* see below */
  dev->message_fatal         = DEV_message_fatal;         /* see below */

  return dev;
}

Public void
dvi_device_dispose(DVI_DEVICE dev)
{
  if (dev != NULL){
    resource_dispose(GET_RES(dev));
    free(dev);
  }
}

Public void
dvi_device_change_visual_eps(DVI_DEVICE dev, int mode)
{
  if (x_visual_color() == 0)
    mode = VISUAL_MONO;

  DviVisualEPS = mode;
  switch (mode){
  default:
  case VISUAL_MONO:
    dev->put_pixmap_rgb    = NULL;
    dev->put_graymap       = NULL;
    break;
  case VISUAL_GRAY:
    dev->put_pixmap_rgb    = NULL;
    dev->put_graymap       = DEV_put_graymap;
    break;
  case VISUAL_RGB:
    dev->put_pixmap_rgb    = DEV_put_pixmap_rgb;
    dev->put_graymap       = NULL;
    break;
  }
}

Public void
dvi_device_change_visual_text(DVI_DEVICE dev, int mode)
{
  if (x_visual_color() == 0)
    mode = VISUAL_MONO;

  DviVisualText = mode;
  switch (mode){
  default:
  case VISUAL_MONO:
    dev->color_gray        = NULL;
    dev->bgcolor_gray      = NULL;
    dev->color_rgb         = NULL;
    dev->bgcolor_rgb       = NULL;
    break;
  case VISUAL_GRAY:
    dev->color_gray        = DEV_color_gray;
    dev->bgcolor_gray      = DEV_bgcolor_gray;
    dev->color_rgb         = NULL;
    dev->bgcolor_rgb       = NULL;
    break;
  case VISUAL_RGB:
    dev->color_gray        = NULL;
    dev->bgcolor_gray      = NULL;
    dev->color_rgb         = DEV_color_rgb;
    dev->bgcolor_rgb       = DEV_bgcolor_rgb;
    break;
  }
}




void
DEV_put_bitmap(DVI_DEVICE dev, DVI dvi, DVI_BITMAP bm, 
	       int font_id, long key2, long code_point, long x, long y)
{
  if ((bm->width == 0) || (bm->height == 0)) 
    return;

  x_put_bitmap(bm, x, y);
}

void
DEV_put_rectangle(DVI_DEVICE dev, DVI dvi, 
		  long x, long y, long w, long h)
{
  x_put_rectangle(x, y, w, h);
}

void
DEV_put_pixmap_rgb(DVI_DEVICE dev, DVI dvi, DVI_PIXMAP_RGB bm_rgb, 
		   int font_id, long key2, long code_point, long x, long y)
{
  if ((bm_rgb->width == 0) || (bm_rgb->height == 0)) 
    return;

  x_put_pixmap_rgb(bm_rgb, x, y);
}

void
DEV_put_graymap(DVI_DEVICE dev, DVI dvi, DVI_GRAYMAP gm, 
		int font_id, long key2, long code_point, long x, long y)
{
  if ((gm->width == 0) || (gm->height == 0)) 
    return;

  x_put_graymap(gm, x, y);
}


void
DEV_color_rgb(DVI_DEVICE dev, DVI dvi, int page_resume_flag,
	      double r, double g, double b)
{
  x_change_color_fg(r, g, b);
}

void
DEV_bgcolor_rgb(DVI_DEVICE dev, DVI dvi, int page_resume_flag,
		double r, double g, double b)
{
  x_change_color_bg(r, g, b, page_resume_flag);
}

void
DEV_color_gray(DVI_DEVICE dev, DVI dvi, int page_resume_flag, double gs)
{
  x_change_color_gray_fg(gs);
}

void
DEV_bgcolor_gray(DVI_DEVICE dev, DVI dvi, int page_resume_flag, double gs)
{
  x_change_color_gray_bg(gs, page_resume_flag);
}



Public int
DEV_poll(DVI_DEVICE dev, DVI dvi, int poll_type)
{
  int         val;
  static int  t = -1;

  if (--t > 0)
    return 0;

  val = 0;
  switch (poll_type){
  default:
  case DVI_POLL_FILE:
  case DVI_POLL_FONT:
  case DVI_POLL_SPECIAL:
    t = 1;
    DviDisableCommand = 1;
    x_poll();
    val = 0;
    DviDisableCommand = 0;
    break;
  case DVI_POLL_PAGE:
    t = RES(DviDev,poll);
    DviDisableCommand = 0;
    val = x_poll();
    DviDisableCommand = 0;
    break;
  }
  return val;
}

Public char*
DEV_gs_program_path(DVI_DEVICE dev, DVI dvi)
{
  return RES(dev,gs_path);
}

Public int
DEV_draw_ps_figures(DVI_DEVICE dev, DVI dvi)
{
  if (DviEPSRendering == 1)
    return 1;
  return 0;
}

Public int
DEV_before_ps_figure(DVI_DEVICE dev, DVI dvi, 
		     char *special_cmd, long x, long y, long dvipos)
{
  PAGE_CACHE pc; 

  pc = x_get_cache(DviCurrentPage);

  pc->n_eps_figs = pc->n_eps_figs + 1;
  if (DviEPSRendering == 1)
    x_cursor_change(CURSOR_DRAWING_PS);
  else 
    pc->pending_eps = pc->pending_eps + 1;
  return 1;
}

Public void
DEV_after_ps_figure(DVI_DEVICE dev, DVI dvi,
		    char *special_cmd, long x, long y, long dviposxcompi)
{
  if (DviEPSRendering == 1)
    x_cursor_change(CURSOR_DRAWING);
}

Public int
DEV_special_command_undef(DVI_DEVICE dev, DVI dvi, 
			  char *cmd, long x, long y, long dvipos)
{
  DEV_message_error(dev, dvi, 
		    "This special command in DVI file is not supported:\n");
  if (strlen(cmd) < 32)
    DEV_message_error(dev, dvi, "  %s\n", cmd);
  else
    DEV_message_error(dev, dvi, "  %*s ...\n", 32, cmd);
  return 0;
}

Public int
DEV_gs_timeout_value(DVI_DEVICE dev, DVI dvi)
{
  if (RES(dev,gs_timeout) <= 0)
    return 0;
  return RES(dev,gs_timeout);
}

Public int
DEV_gs_timeout_giveup(DVI_DEVICE dev, DVI dvi)
{
  DEV_message_error(dev, dvi, "Timed out Ghostscript process...");
  return 1;
}

void 
DEV_message_advice(DVI_DEVICE dev, DVI dvi, char *fmt, ...)
{
  va_list ap;
  char    msg[512];

  va_start(ap, fmt);
  if (RES(DviDev,novice) == 1){
    vsprintf(msg, fmt, ap);
    x_message(msg);
  }
  va_end(ap);
}
void 
DEV_message_warning(DVI_DEVICE dev, DVI dvi, char *fmt, ...)
{
  va_list ap;
  char    msg[512];

  va_start(ap, fmt);
  vsprintf(msg, fmt, ap);
  x_message(msg);
  va_end(ap);
}
void 
DEV_message_error(DVI_DEVICE dev, DVI dvi, char *fmt, ...)
{
  va_list ap;
  char    msg[512];

  va_start(ap, fmt);
  vsprintf(msg, fmt, ap);
  x_error_message(msg);
  va_end(ap);
}
void 
DEV_message_fatal(DVI_DEVICE dev, DVI dvi, char *fmt, ...)
{
  va_list ap;
  char    msg[512];

  va_start(ap, fmt);
  vsprintf(msg, fmt, ap);
  x_error_message(msg);
  x_error_message("FATAL ERROR: Exiting xmdvi...");
  fprintf(stderr, "FATAL ERROR: Exiting xmdvi...");
  vfprintf(stderr, fmt, ap);
  fprintf(stderr, "\n");
  va_end(ap);

  sleep(5);
  exit(1);
}

/*EOF*/
