/*
 * xldvi  -  A very simple previewer for DVI files on X Window.
 * by Hirotsugu Kakugawa
 * 
 *  18 Dec 1996  Version 2.0 based on DVIlib 2.0 and VFlib 3.1
 *  21 Feb 1997  Version 2.3 based on DVIlib 2.2
 *               Added enlarging/shrinking window.
 *   4 Mar 1997  Upgraded for DVIlib 2.3
 *   3 Jun 1997  Version 2.4  Added vflibcap parameterlization
 *               feature for vflibcap.
 *   6 Jun 1997  Added color feature. Added -c option.
 *  28 Oct 1997  Added -eps-{mono,gray,rgb}, -text-{mono,gray,rgb} options.
 *               (-eps-mono, -eps-gray options are no longer supported.)
 *  27 Nov 1998  Added -dpi, -mode options.
 *  29 Mar 1999  Added asynchronous gs process invocation.
 */
/*
 * Copyright (C) 1996-1999  Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#if STDC_HEADERS
#  include <string.h>
#else
#  include <strings.h>
#endif

#include <X11/Xlib.h>
#include <X11/Xutil.h>

#include "libdvi29.h"
#include "defs.h"
#include "cf-xldvi.h"
#include "resource.h"
#include "compiled.h"
#include "window.h"
#include "../params.h"

#define PROGRAM   "xldvi"

#define PR1(s1)    fprintf(stderr, s1);
#define PR2(s1,s2) fprintf(stderr, s1,s2);

Public DVI          dviobj;
Public DVI_DEVICE   dev;
Public int          paper;   

Private DVI_DEVICE dvi_device_create(void);
Private char  *parse_args(int,char**);
Private void  usage(void);
Private void  license(void);
Private void  usage_key_mouse(void);


Public int
main(int argc, char **argv)
{
  DVI_PROPERTY  dvi_props;
  char          *dvi_file;
  char          params[3*256];

  current_page = 1;
  x_open_display(NULL);

  get_default_application_resources();
  get_application_resources();
  if ((dvi_file = parse_args(argc, argv)) == NULL)
    exit(0);
  resource_conv();
  resource_check();
  set_shrink(Resource.shrink);

  x_init();
  x_change_cursor(CURSOR_READING);

  sprintf(params, "%s=%d, %s=%s, %s=%s", 
	  PARAM_NAME_DPI,  Resource.dpi,
	  PARAM_NAME_MODE, Resource.kpathsea_mode,
	  PARAM_NAME_PROG, "xldvi");

  if (DVI_INIT(Resource.vflibcap, params) < 0){
    fprintf(stderr, "%s: Failed to initialize DVI interpreter system.\n",
	    PROGRAM);
    exit(1);
  }

  if ((dev = dvi_device_create()) == NULL){
    fprintf(stderr, "%s: Failed to initialize device.\n",
	    PROGRAM);
    exit(1);
  }

  dvi_props = DVI_PROPERTY_ALLOC_DEFAULT();
  if (dvi_props != NULL){
    if (Resource.eps_style == EPS_STYLE_LATEX2E_GRAPHICS_STY){
      DVI_PROPERTY_SET(dvi_props,   DVI_PROP_LATEX2E_GRAPHICS_STY);
      DVI_PROPERTY_UNSET(dvi_props, DVI_PROP_LATEX209_EPSF_STY);
    } else {
      DVI_PROPERTY_UNSET(dvi_props, DVI_PROP_LATEX2E_GRAPHICS_STY);
      DVI_PROPERTY_SET(dvi_props,   DVI_PROP_LATEX209_EPSF_STY);
    }
    DVI_PROPERTY_SET(dvi_props, DVI_PROP_ASYNC_GS_INVOCATION);
    DVI_PROPERTY_SET(dvi_props, DVI_PROP_INCREMENTAL_EPS_DISPLAY);
  }

  dviobj = DVI_CREATE(dev, dvi_file, dvi_props);
  if ((dviobj == NULL) || (dviobj->pages <= 0)) {
    fprintf(stderr, "%s: Failed to read a DVI file.\n", PROGRAM);
    exit(1);
  }

  if (DVI_OPEN_FONT(dviobj, dev) < 0){
    fprintf(stderr, "%s: Cannot find all fonts\n", PROGRAM);
    exit(1);
  }

  x_change_cursor(CURSOR_READY);
  main_loop();

  DVI_DISPOSE(dviobj, dev);
  return 0;
}

Private DVI_DEVICE
dvi_device_create(void)
{
  DVI_DEVICE  dev;

  if ((dev = DVI_device_alloc()) == NULL)
    return NULL;
  dev->h_dpi                 = (double)Resource.dpi;
  dev->v_dpi                 = (double)Resource.dpi;
  dev->device_polling        = DEV_poll;
  dev->put_bitmap            = DEV_put_bitmap;
  dev->put_rectangle         = DEV_put_rectangle;

  switch (Resource.visual_text){
  default:
  case VISUAL_MONO:
    break;
  case VISUAL_GRAY:
    dev->color_gray   = DEV_color_gray;
    dev->bgcolor_gray = DEV_bgcolor_gray;
    break;
  case VISUAL_RGB:
    dev->color_rgb    = DEV_color_rgb;
    dev->bgcolor_rgb  = DEV_bgcolor_rgb;
    break;
  }
  if (x_visual_color() == 0){
    dev->color_rgb  = dev->bgcolor_rgb  = NULL;
    dev->color_gray = dev->bgcolor_gray = NULL;
  }

  switch (Resource.visual_eps){
  default:
  case VISUAL_MONO:
    dev->put_pixmap_rgb = NULL;
    dev->put_graymap    = NULL;
    break;
  case VISUAL_GRAY:
    dev->put_pixmap_rgb = NULL;
    dev->put_graymap    = DEV_put_graymap;
    break;
  case VISUAL_RGB:
    dev->put_pixmap_rgb = NULL;
    dev->put_pixmap_rgb = DEV_put_pixmap_rgb;
    break;
  }
  if (x_visual_color() == 0){
    dev->put_pixmap_rgb = NULL;    
    dev->put_graymap    = NULL;    
  }

  dev->special_command_undef = DEV_special_command_undef;
  dev->draw_ps_figures       = DEV_draw_ps_figures;
  dev->before_ps_figure      = DEV_before_ps_figure;
  dev->after_ps_figure       = DEV_after_ps_figure;

  dev->gs_program_path       = DEV_gs_program_path;
  dev->gs_timeout_value      = DEV_gs_timeout_value;
  dev->gs_timeout_give_up    = DEV_gs_timeout_giveup;

  dev->message_advice        = DEV_message_advice;
  dev->message_warning       = DEV_message_warning;
  dev->message_error         = DEV_message_error;
  dev->message_fatal         = DEV_message_fatal;

  return dev;
}

Private char*
parse_args(int argc, char **argv)
{
  char        *dvi_name, *dvi_file;
  double      m, s;
  int         c;
  static char shrink_str[64];  /* must be 'static' */
  static char cache_str[64];   /* must be 'static' */

  --argc; argv++;
  dvi_name = NULL;
  while (argc > 0){
    if ((strcmp(*argv, "-m") == 0) && (argc > 1)){
      argc--; argv++;
      m = atof(*argv);
      s = atof(Resource.s_shrink);
      if ((m <= 0.001) || (m < MAX_MAGNIFICATION)){
	sprintf(shrink_str, "%.5f", s/m);
	Resource.s_shrink = shrink_str;
      } else {
	fprintf(stderr, "Maginification is out of range.\n");
	usage();
	exit(1);
      }
    } else if (((strcmp(*argv, "-s") == 0) || (strcmp(*argv, "-shrink") == 0))
	       && (argc > 1)){
      argc--; argv++;
      Resource.s_shrink = *argv;
    } else if ((strcmp(*argv, "-dpi") == 0) && (argc > 1)){
      argc--; argv++;
      Resource.s_dpi = *argv;
    } else if ((strcmp(*argv, "-mode") == 0) && (argc > 1)){
      argc--; argv++;
      Resource.kpathsea_mode = *argv;
    } else if (strcmp(*argv, "-cx") == 0){
      Resource.s_dpi = "300";
      Resource.kpathsea_mode = "cx";
    } else if (strcmp(*argv, "-sparcptr") == 0){
      Resource.s_dpi = "400";
      Resource.kpathsea_mode = "sparcptr";
    } else if (strcmp(*argv, "-ljfour") == 0){
      Resource.s_dpi = "600";
      Resource.kpathsea_mode = "ljfour";
    } else if ((strcmp(*argv, "-c") == 0) && (argc > 1)){
      argc--; argv++;
      if ((c = atoi(*argv)) < 1)
	c = 1;
      sprintf(cache_str, "%d", c);
      Resource.s_page_cache = cache_str;
    } else if ((strcmp(*argv, "-ver") == 0)
	       || (strcmp(*argv, "-version") == 0)){
      printf("%s %s of %s on %s\n",
	     PROG_NAME, VERSION, COMPILED_DATE, COMPILED_HOST);
      exit(0);
    } else if ((strcmp(*argv, "-v") == 0) && (argc > 1)){
      argc--;  argv++;
      Resource.vflibcap = *argv;
    } else if ((strcmp(*argv, "-fg") == 0) && (argc > 1)){
      argc--;  argv++;
      Resource.color_char = *argv;
    } else if ((strcmp(*argv, "-bg") == 0) && (argc > 1)){
      argc--;  argv++;
      Resource.color_paper = *argv;
    } else if (strcmp(*argv, "-text-mono") == 0){
      Resource.s_visual_text = VISUAL_MONO_STR;
    } else if (strcmp(*argv, "-text-gray") == 0){
      Resource.s_visual_text = VISUAL_GRAY_STR;
    } else if (strcmp(*argv, "-text-rgb") == 0){
      Resource.s_visual_text = VISUAL_RGB_STR;
    } else if ((strcmp(*argv, "-frame") == 0) && (argc > 1)){
      argc--;  argv++;
      Resource.color_frame = *argv;
    } else if ((strcmp(*argv, "-back") == 0) && (argc > 1)){
      argc--;  argv++;
      Resource.color_back = *argv;
    } else if ((strcmp(*argv, "-ox") == 0) && (argc > 1)){
      argc--;  argv++;
      Resource.s_offset_x = *argv;
    } else if ((strcmp(*argv, "-oy") == 0) && (argc > 1)){
      argc--;  argv++;
      Resource.s_offset_y = *argv;
    } else if ((strcmp(*argv, "-cache") == 0) && (argc > 1)){
      argc--;  argv++;
      Resource.s_page_cache = *argv;
    } else if (strcmp(*argv, "-l") == 0){
      Resource.s_orient = "Landscape";
    } else if (strcmp(*argv, "-epsf.sty") == 0){
      Resource.s_eps_style = EPS_STYLE_LATEX209_EPSF_STY_STR;
    } else if (strcmp(*argv, "-no-eps") == 0){
      Resource.s_draw_eps = "No";
    } else if (strcmp(*argv, "-eps-mono") == 0){
      Resource.s_visual_eps = VISUAL_MONO_STR;
    } else if (strcmp(*argv, "-eps-gray") == 0){
      Resource.s_visual_eps = VISUAL_GRAY_STR;
    } else if (strcmp(*argv, "-eps-rgb") == 0){
      Resource.s_visual_eps = VISUAL_RGB_STR;
    } else if ((strcmp(*argv, "-h") == 0) || (strcmp(*argv, "--help") == 0)){
      usage();
      exit(1);
    } else if (strcmp(*argv, "-license") == 0){
      license();
      exit(1);
    } else if (strcmp(*argv, "-help-key") == 0){
      usage_key_mouse();
      exit(1);
    } else if (paper_sym2id(&argv[0][1], 0) >= 0){
      Resource.s_paper = &argv[0][1];
    } else if (**argv != '-'){
      dvi_name = *argv;
    } else {
      fprintf(stderr, "Unknow option: %s\n", *argv);
      exit(1);
    }
    argc--;  argv++;
  }

  if (dvi_name == NULL){
    fprintf(stderr, "%s: No DVI file\n", PROGRAM);
    usage();
    exit(1);
  }
  if ((dvi_file = find_dvi_file(dvi_name)) == NULL){
    fprintf(stderr, "%s: DVI file not found: %s\n", PROGRAM, dvi_name);
    exit(1);
  }
  return dvi_file;
}


Private void
usage(void)
{
  fprintf(stderr, "%s version %s of %s on %s\n", 
	  PROG_NAME, VERSION, COMPILED_DATE, COMPILED_HOST);
  fprintf(stderr, "Usage: %s [Paper Size] [Options] dvi-file\n", PROG_NAME);
  PR1("Paper Size: ");
  PR1("  -A3, -A4, -A5, -B4, -B5, -B6, -LETTER, -US, etc\n");
  PR1("  -l            Landscape mode\n");
  PR1("Options:  (Each value enclosed by [ ] is the default.)\n");
  PR2(" -v VFLIBCAP File name of vflibcap file [%s]\n", DEFAULT_VFLIBCAP);
  PR2(" -dpi DPI    Device resolution [%s]\n", DEFAULT_DPI);
  PR2(" -mode MODE  Device mode for kpathsea [%s]\n", DEFAULT_KPATHSEA_MODE);
  PR1(" -cx         Same as `-dpi 300 -mode cx'\n");
  PR1(" -sparcptr   Same as `-dpi 400 -mode sparcptr'\n");
  PR1(" -ljfour     Same as `-dpi 600 -mode ljfour'\n");
  PR2(" -m MAG      Magnification  (0<M<%d) [1.0]\n", MAX_MAGNIFICATION);
  PR2(" -fg COL     Character color [%s]\n", DEFAULT_PAPER_CHAR_COLOR);
  PR2(" -bg COL     Paper color [%s]\n", DEFAULT_PAPER_PAPER_COLOR);
  PR2(" -frame COL  Paper frame color [%s]\n", DEFAULT_PAPER_FRAME_COLOR);
  PR2(" -back COL   Background color [%s]\n", DEFAULT_PAPER_BACKGROUND_COLOR);
  PR2(" -cache N    Set page cache size [%s]\n", DEFAULT_PAGE_CACHE_SIZE);
  PR1(" -no-eps     Disable EPS figures\n");
  PR1(" -eps-{mono,gray,rgb}     Monochrome/Grayscale/RGB EPS figures\n");
  PR1(" -text-{mono,gray,rgb}    Monochrome/Grayscale/RGB text color\n");
  PR1(" -epsf.sty    EPS figures are typeset by LaTeX 2.09 epsf.sty\n");
  PR1(" -help        Show usage\n");
  PR1(" -help-key    Show key and mouse operations\n");
  PR1(" -license     Show license\n");
  PR1("Key and Mouse operations:\n");
  PR1(" SPC (mouse right button)    Go to next page\n");
  PR1(" BS  (mouse left button)     Go to previous page\n");
  PR1(" q                           Quit\n");
}

Private void
license(void)
{
  PR1("Copyright (C) 1996-1997 Hirotsugu Kakugawa. \n"
      "All rights reserved.\n\n"
      "This program is free software; you can redistribute it and/or modify\n"
      "it under the terms of the GNU General Public License as published by\n"
      "the Free Software Foundation; either version 2, or (at your option)\n"
      "any later version.\n"
      "This program is distributed in the hope that it will be useful,\n"
      "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
      "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
      "GNU General Public License for more details.\n");
}

Private void
usage_key_mouse(void)
{
  fprintf(stderr, "%s version %s of %s on %s\n", 
	  PROG_NAME, VERSION, COMPILED_DATE, COMPILED_HOST);
  PR1("Key and Mouse operation:\n");
  PR1("  SPC (mouse right button)    Go to next page\n");
  PR1("  BS  (mouse left button)     Go to previous page\n");
  PR1("  <                           Go to the first page\n");
  PR1("  >                           Go to the last page\n");
  PR1("  ]                           Magnify\n");
  PR1("  [                           Shrink\n");
  PR1("  ^f (Right)                  Move view point right\n");
  PR1("  ^b (Left)                   Move view point left\n");
  PR1("  ^p (Up)                     Move view point up\n");
  PR1("  ^n (Down)                   Move view point down\n");
  PR1("  }                           Enlarge window\n");
  PR1("  {                           Shrink window\n");
  PR1("  q                           Quit\n");
}



/*
 * Draw Bitmaps
 */
Public void
DEV_put_bitmap(DVI_DEVICE dev, DVI dvi, 
	       DVI_BITMAP bm, 
	       int font_id, long key2, long code_point, long x, long y)
{
  x_put_bitmap(bm, x, y);
}

Public void
DEV_put_rectangle(DVI_DEVICE dev, DVI dvi,
		  long x, long y, long w, long h)
{
  x_put_rectangle(x, y, w, h);
}


Public void
DEV_put_pixmap_rgb(DVI_DEVICE dev, DVI dvi, 
		   DVI_PIXMAP_RGB pm, 
	       int font_id, long key2, long code_point, long x, long y)
{
  x_put_pixmap_rgb(pm, x, y);
}

Public void
DEV_put_graymap(DVI_DEVICE dev, DVI dvi, 
		DVI_GRAYMAP gm, 
		int font_id, long key2, long code_point, long x, long y)
{
  x_put_graymap(gm, x, y);
}


/*
 * Special command for EPS figures
 */

Public int
DEV_special_command_undef(DVI_DEVICE dev, DVI dvi, char *cmd, 
			  long x, long y, long dvipos)
{
  DEV_message_error(dev, dvi, 
		    "This special command in DVI file is not supported:");
  DEV_message_error(dev, dvi, "%s", cmd);
  DEV_message_error(dev, dvi, "");
  return 0;
}

Public int
DEV_draw_ps_figures(DVI_DEVICE dev, DVI dvi)
{
  if (Resource.draw_eps == 1)
    return 1;
  return 0;
}

static int old_cursor;

Public int
DEV_before_ps_figure(DVI_DEVICE dev, DVI dvi, char *cmd, 
		     long x, long y, long dvipos)
{
  old_cursor = x_currenct_cursor();
  x_change_cursor(CURSOR_DRAWING_PS);
  return 1;
}

Public void
DEV_after_ps_figure(DVI_DEVICE dev, DVI dvi, char *cmd, 
		    long x, long y, long dvipos)
{
  x_change_cursor(old_cursor);
}


/*
 * Ghostscript control
 */
Public char*
DEV_gs_program_path(DVI_DEVICE dev, DVI dvi)
{
#if 0
  return "/home/kakugawa/hack/GS/gs4/gs4.03";
#endif
  return Resource.gs_path;
}

Public int
DEV_gs_timeout_value(DVI_DEVICE dev, DVI dvi)
{
  return Resource.gs_timeout;
}

Public int
DEV_gs_timeout_giveup(DVI_DEVICE dev, DVI dvi)
{
  return 1;  /* Yes, I want to give up.  It takes too long to finish. */
}

/*
 * Color Change
 */
Public void
DEV_color_rgb(DVI_DEVICE dev, DVI dvi, int page_resume_flag,
	      double r, double g, double b)
{
  x_change_color_fg(page_resume_flag, r, g, b);
}

Public void
DEV_bgcolor_rgb(DVI_DEVICE dev, DVI dvi, int page_resume_flag,
		double r, double g, double b)
{
  x_change_color_bg(page_resume_flag, r, g, b);
}

Public void
DEV_color_gray(DVI_DEVICE dev, DVI dvi, int page_resume_flag, double g)
{
  x_change_color_fg(page_resume_flag, g, g, g);
}

Public void
DEV_bgcolor_gray(DVI_DEVICE dev, DVI dvi, int page_resume_flag, double g)
{
  x_change_color_bg(page_resume_flag, g, g, g);
}


/* 
 * Message
 */
Public void 
DEV_message_advice(DVI_DEVICE dev, DVI dvi, char *fmt, ...)
{
  va_list ap;

  va_start(ap, fmt);
#if 0  /* We love silence. */
  vprintf(fmt, ap);  
  printf("\n");
#endif
  va_end(ap);
}

Public void 
DEV_message_warning(DVI_DEVICE dev, DVI dvi, char *fmt, ...)
{
  va_list ap;

  va_start(ap, fmt);
  vprintf(fmt, ap);  
  printf("\n");
  va_end(ap);
}

Public void 
DEV_message_error(DVI_DEVICE dev, DVI dvi, char *fmt, ...)
{
  va_list ap;

  va_start(ap, fmt);
  vprintf(fmt, ap);
  printf("\n");
  va_end(ap);
}

Public void 
DEV_message_fatal(DVI_DEVICE dev, DVI dvi, char *fmt, ...)
{
  va_list ap;

  va_start(ap, fmt);
  vprintf(fmt, ap);
  printf("\n");
  va_end(ap);
  exit(1);
}

/*EOF*/
