/************************************************************************/
/*									*/
/*  Gif writing basic functionality.					*/
/*									*/
/*  ORIGIN: "Gif-Lib" - Yet another gif library.			*/
/*  Written by:  Gershon Elber, Ver 1.1, Aug. 1990			*/
/*  Version 3.0 by Eric S. Raymond (Full GIF89 support)			*/
/*									*/
/*  Modified to what it is now by Mark de Does				*/
/*									*/
/************************************************************************/

#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<string.h>
#   include	"bm_gif_lib.h"
#   include	"bm_gif_lib_private.h"

#   include	<appDebugon.h>
#   include	<sioBlocked.h>
#   include	<sioLzw.h>
#   include	<sioEndian.h>

/************************************************************************/
/*									*/
/*  Write a color map to file.						*/
/*									*/
/************************************************************************/

static int bmGifWriteColorMap(	GifFilePrivateType *	gfpt,
				const GifColorMap *	gcm )
    {
    int			i;
    const RGB8Color *	rgb8;
    int			count= 1 << gcm->gcmBitsPerPixel;

    if  ( gcm->gcmColorCount > count )
	{ LLDEB(gcm->gcmColorCount,count);	}

    rgb8= gcm->gcmColors;
    for ( i= 0; i < gcm->gcmColorCount && i < count; rgb8++, i++ )
	{
	sioOutPutCharacter( rgb8->rgb8Red, gfpt->gfptSos );
	sioOutPutCharacter( rgb8->rgb8Green, gfpt->gfptSos );
	sioOutPutCharacter( rgb8->rgb8Blue, gfpt->gfptSos );
	}

    for ( i= gcm->gcmColorCount; i < count; rgb8++, i++ )
	{
	sioOutPutCharacter( 255, gfpt->gfptSos );
	sioOutPutCharacter( 255, gfpt->gfptSos );
	sioOutPutCharacter( 255, gfpt->gfptSos );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Initialize GIF writing to an OutputStream.				*/
/*									*/
/*  _GifError is cleared if succesfull.					*/
/*									*/
/************************************************************************/

GifFileType * EGifOpenFileHandle(	SimpleOutputStream *	sos )
{
    GifFileType *		gft;
    GifFilePrivateType *	gfpt;

    gft= (GifFileType *)malloc(sizeof(GifFileType));
    if  ( ! gft )
	{ XDEB(gft); return (GifFileType *)0;	}

    memset( gft, '\0', sizeof(GifFileType) );

    strncpy( gft->gftVersionString, GIF87_STAMP, 6 )[6]= '\0';

    gfpt= (GifFilePrivateType *)malloc(sizeof(GifFilePrivateType));
    if  ( ! gfpt )
	{ XDEB(gfpt); free(gft); return (GifFileType *)0;	}

    gft->Private = (void *) gfpt;
    gfpt->gfptSos= sos;
    gfpt->gfptSosBlocked= (SimpleOutputStream *)0;
    gfpt->gfptSosLzw= (SimpleOutputStream *)0;
    gfpt->FileState = FILE_STATE_WRITE;
    
    _GifError = 0;

    return gft;
}

/******************************************************************************
*   Routine to set current GIF version. All files open for write will be      *
* using this version until next call to this routine. Version consists of     *
* 3 characters as "87a" or "89a". No test is made to validate the version.    *
******************************************************************************/

void bmGifSetVersion(	GifFileType *	gft,
			const char *	version )
{
    if  ( strcmp( version, GIF87_STAMP+ 3 )	&&
	  strcmp( version, GIF89_STAMP+ 3 )	)
	{ SDEB(version); return;	}

    strncpy( gft->gftVersionString+ 3, version, 3 );
}

/************************************************************************/
/*									*/
/*  Write the Screen descriptor to the output.				*/
/*									*/
/*  This routine should be called before any other EGif calls, I.E	*/
/*  immediately after the GIF file openning.				*/
/*									*/
/*  1)  First write the version prefix into the file.			*/
/*  2)  Put the screen descriptor into the file				*/
/*  3)  If we have Global color map - dump it also			*/
/*									*/
/************************************************************************/

int EGifPutScreenDesc(	GifFileType *		gft,
			int			Width,
			int			Height,
			int			bitsPerPixel,
			int			BackGround,
			const GifColorMap *	gcm )
{
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)gft->Private;
    GifScreenDescriptor *	gsd= &(gft->gftScreenDescriptor);

    if (gfpt->FileState & FILE_STATE_SCREEN) {
	/* If already has screen descriptor - something is wrong! */
	_GifError = E_GIF_ERR_HAS_SCRN_DSCR;
	return GIF_ERROR;
    }

    if (!IS_WRITEABLE(gfpt)) {
	/* This file was NOT open for writing: */
	_GifError = E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
    }

    /*  1  */
    sioOutPutString( gft->gftVersionString, gfpt->gfptSos );

    gsd->gsdScreenWide= Width;
    gsd->gsdScreenHigh= Height;
    gsd->gsdScreenBitsPerPixel= bitsPerPixel;
    gsd->gsdScreenBackgroundColor= BackGround;
    gsd->gsdScreenAspectRatio= 0;

    bmGifInitGifColorMap( &(gsd->gsdScreenColorMap) );

    if  ( gcm )
	{ gsd->gsdScreenColorMap= *gcm;	}

    /*  2  */
    sioEndianPutLeInt16( gsd->gsdScreenWide, gfpt->gfptSos );
    sioEndianPutLeInt16( gsd->gsdScreenHigh, gfpt->gfptSos );

    gsd->gsdPackedFields= 0;
    if  ( gcm && gcm->gcmColorCount > 0 )
	{
	gsd->gsdPackedFields |= 0x80;
	gsd->gsdPackedFields |= gcm->gcmBitsPerPixel -1;
	}
    gsd->gsdPackedFields |= ( bitsPerPixel -1 ) << 4;

    sioOutPutCharacter( gsd->gsdPackedFields, gfpt->gfptSos );
    sioOutPutCharacter( gsd->gsdScreenBackgroundColor, gfpt->gfptSos );
    sioOutPutCharacter( gsd->gsdScreenAspectRatio, gfpt->gfptSos );

    /*  3  */
    if  ( gcm && gcm->gcmColorCount > 0 )
	{
	if  ( bmGifWriteColorMap( gfpt, &(gsd->gsdScreenColorMap) ) )
	    { XDEB(gcm); return GIF_ERROR;	}
	}

    /* Mark this file as has screen descriptor, and no pixel written yet: */
    gfpt->FileState |= FILE_STATE_SCREEN;

    return GIF_OK;
}

/************************************************************************/
/*									*/
/*  Emit an image descriptor to file.					*/
/*									*/
/*  This routine should be called after a screen descripto is emmitted	*/
/*  and before any of the image data is emitted. If the image has	*/
/*  transparent parts, the graphics extension saying so should have	*/
/*  been emitted before.						*/
/*									*/
/*  In general, any extensions and comments relating to an image should	*/
/*  be emitted BEFORE the image.					*/
/*									*/
/*  Setup the LZ compression for this image:				*/
/*									*/
/*  2)  Find out what color map is relevant and determine BitsPerPixel.	*/
/*  3)  Determine code size and insert it in the output stream.		*/
/*  4)  Open a blocked stream to divide the compressed output in	*/
/*	packets. (Push it on the unstructured one.)			*/
/*  5)  Open an LZW compressed stream, push it on the blocked one.	*/
/*									*/
/*  The result is that pixels emitted to the image are LZW compressed,	*/
/*  the compressed data is divided in packets and the packets are	*/
/*  emitted over the output stream for the image.			*/
/*									*/
/************************************************************************/

static int bmGifSetupCompress(	GifFilePrivateType *	gfpt,
				int			codeSize )
    {
    /*  4  */
    if  ( gfpt->gfptSosBlocked )
	{ XDEB(gfpt->gfptSosBlocked);	}

    gfpt->gfptSosBlocked= sioOutBlockedOpen( gfpt->gfptSos );
    if  ( ! gfpt->gfptSosBlocked )
	{ XDEB(gfpt->gfptSosBlocked); return -1;	}

    /*  5  */
    if  ( gfpt->gfptSosLzw )
	{ XDEB(gfpt->gfptSosLzw);	}

    gfpt->gfptSosLzw= sioOutLzwOpen( gfpt->gfptSosBlocked, codeSize );
    if  ( ! gfpt->gfptSosLzw )
	{ XDEB(gfpt->gfptSosLzw); return -1;	}

    return 0;
    }

static int bmGifCleanupCompress(	GifFilePrivateType *	gfpt )
    {
    if  ( gfpt->gfptSosLzw )
	{
	if  ( sioOutClose( gfpt->gfptSosLzw ) )
	    { LDEB(1); return -1;	}
	gfpt->gfptSosLzw= (SimpleOutputStream *)0;
	}

    if  ( gfpt->gfptSosBlocked )
	{
	if  ( sioOutClose( gfpt->gfptSosBlocked ) )
	    { LDEB(1); return -1;	}
	gfpt->gfptSosBlocked= (SimpleOutputStream *)0;
	}

    return 0;
    }

int EGifPutImageDesc(	GifFileType *		gft,
			int			Left,
			int			Top,
			int			Width,
			int			Height,
			int			Interlace,
			const GifColorMap *	gcm )
{
    unsigned char		flags;
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)gft->Private;
    GifScreenDescriptor *	gsd= &(gft->gftScreenDescriptor);
    GifImageDesc *		gid= &(gft->gftCurrentImageDescriptor);

    int				codeSize;

    if (gfpt->FileState & FILE_STATE_IMAGE &&
#if defined(__MSDOS__) || defined(__GNUC__)
	gfpt->PixelCount > 0xffff0000UL) {
#else
	gfpt->PixelCount > 0xffff0000) {
#endif /* __MSDOS__ */
	/* If already has active image descriptor - something is wrong! */
	_GifError = E_GIF_ERR_HAS_IMAG_DSCR;
	return GIF_ERROR;
    }
    if (!IS_WRITEABLE(gfpt)) {
	/* This file was NOT open for writing: */
	_GifError = E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
    }

    gid->Left = Left;
    gid->Top = Top;
    gid->Width = Width;
    gid->Height = Height;
    gid->Interlace = Interlace;

    bmGifInitGifColorMap( &(gid->gidImageColorMap) );

    if  ( gcm )
	{ gid->gidImageColorMap= *gcm;	}

    /* Put the image descriptor into the file: */
    sioOutPutCharacter( ',', gfpt->gfptSos );
    sioEndianPutLeInt16( gid->Left, gfpt->gfptSos );
    sioEndianPutLeInt16( gid->Top, gfpt->gfptSos );
    sioEndianPutLeInt16( gid->Width, gfpt->gfptSos );
    sioEndianPutLeInt16( gid->Height, gfpt->gfptSos );

    flags= 0;
    if  ( gcm && gcm->gcmColorCount > 0 )
	{
	flags |= 0x80;
	flags |= gcm->gcmBitsPerPixel - 1;
	}
    if  ( Interlace )
	{
	flags |= 0x40;
	}
    sioOutPutCharacter( flags, gfpt->gfptSos );

    /* If we have Global color map - dump it also: */
    if  ( gcm && gcm->gcmColorCount > 0 )
	{
	if  ( bmGifWriteColorMap( gfpt, gcm ) )
	    { XDEB(gcm); return GIF_ERROR;	}
	}

    /* Mark this file as having an image: */
    gfpt->FileState |= FILE_STATE_IMAGE;
    gfpt->PixelCount = (long) Width * (long) Height;

    /*  2  */
    if  ( gcm && gcm->gcmColorCount > 0 )
	{ codeSize= gcm->gcmBitsPerPixel;	}
    else{
	if  ( gsd->gsdScreenColorMap.gcmColorCount > 0 )
	    { codeSize= gsd->gsdScreenColorMap.gcmBitsPerPixel;	}
	else{
	    if  ( gcm )
		{ LDEB(gcm->gcmColorCount);	}
	    else{ XDEB(gcm);			}
	    LDEB( gsd->gsdScreenColorMap.gcmBitsPerPixel );

	    _GifError = E_GIF_ERR_NO_COLOR_MAP;
	    return GIF_ERROR;
	    }
	}

    /*  3  */
    codeSize= ( codeSize < 2 ? 2 : codeSize );
    sioOutPutCharacter( codeSize, gfpt->gfptSos );

    if  ( bmGifSetupCompress( gfpt, codeSize ) )
	{ LDEB(1); return GIF_ERROR;	}

    return GIF_OK;
}

/************************************************************************/
/*									*/
/*  Emit a series of pixel values to file.. typically a scanline.	*/
/*									*/
/*  1)  File not writable.. Refuse.					*/
/*  2)  Refuse to exceed the number of pixels in the image.		*/
/*  3)  Subtract the number of pixels from that left in the file.	*/
/*  4)  Actually emit the pixels.					*/
/*  5)  When the image is finished, clean compressor.			*/
/*									*/
/************************************************************************/

int bmGifPutPixels(		GifFileType *		gft,
				const GifPixelType *	Line,
				int			LineLen )
{
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)gft->Private;

    /*  1  */
    if  ( ! IS_WRITEABLE( gfpt ) )
	{
	_GifError= E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
	}

    /*  2  */
    if  ( gfpt->PixelCount < (unsigned)LineLen )
	{
	LLDEB(gfpt->PixelCount,LineLen);
	_GifError = E_GIF_ERR_DATA_TOO_BIG;
	return GIF_ERROR;
	}

    /*  3  */
    gfpt->PixelCount -= LineLen;

    /*  4  */
    if  ( sioOutWriteBytes( gfpt->gfptSosLzw, Line, LineLen ) != LineLen )
	{
	LDEB(LineLen);
	_GifError = E_GIF_ERR_DISK_IS_FULL; return GIF_ERROR;
	}

    /*  5  */
    if  ( gfpt->PixelCount == 0			&&
	  bmGifCleanupCompress( gfpt )		)
	{ LDEB(gfpt->PixelCount); return GIF_ERROR;	}

    return GIF_OK;
}

/************************************************************************/
/*									*/
/*  Put a comment into GIF file using the GIF89 comment extension	*/
/*  block.								*/
/*									*/
/************************************************************************/

int EGifPutComment(	GifFileType *	gft,
			const char *	Comment )
{
    return EGifPutExtension(gft, COMMENT_EXT_FUNC_CODE, strlen(Comment),
								Comment);
}

/******************************************************************************
*   Put a first extension block (see GIF manual) into gif file.  Here more    *
* extensions can be dumped using EGifPutExtensionMid until		      *
* EGifPutExtensionLast is invoked.					      *
******************************************************************************/

int EGifPutExtensionFirst(	GifFileType *	gft,
				int		ExtCode,
				int		ExtLen,
				const void *	Extension)
{
    GifFilePrivateType *	gfpt= (GifFilePrivateType *) gft->Private;

    if (!IS_WRITEABLE(gfpt)) {
	/* This file was NOT open for writing: */
	_GifError = E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
    }

    if  ( ExtCode == 0 )
	{
	sioOutPutCharacter( ExtLen, gfpt->gfptSos );
	}
    else{
	sioOutPutCharacter( '!', gfpt->gfptSos );
	sioOutPutCharacter( ExtCode, gfpt->gfptSos );
	sioOutPutCharacter( ExtLen, gfpt->gfptSos );
	}

    if  ( sioOutWriteBytes( gfpt->gfptSos, Extension, ExtLen ) != ExtLen )
	{ LDEB(ExtLen); return GIF_ERROR;	}

    return GIF_OK;
}

/******************************************************************************
*   Put a middle extension block (see GIF manual) into gif file.	      *
******************************************************************************/

int EGifPutExtensionNext(	GifFileType *	gft,
				int		ExtCode,
				int		ExtLen,
				const void *	Extension)
{
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)gft->Private;

    if (!IS_WRITEABLE(gfpt)) {
	/* This file was NOT open for writing: */
	_GifError = E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
    }

    sioOutPutCharacter( ExtLen, gfpt->gfptSos );

    if  ( sioOutWriteBytes( gfpt->gfptSos, Extension, ExtLen ) != ExtLen )
	{ LDEB(ExtLen); return GIF_ERROR;	}

    return GIF_OK;
}

/******************************************************************************
*   Put a last extension block (see GIF manual) into gif file.		      *
******************************************************************************/

int EGifPutExtensionLast(	GifFileType *	gft,
				int		ExtCode,
				int		ExtLen,
				const void *	Extension )
{
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)gft->Private;

    if (!IS_WRITEABLE(gfpt)) {
	/* This file was NOT open for writing: */
	_GifError = E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
    }

    sioOutPutCharacter( ExtLen, gfpt->gfptSos );

    if  ( sioOutWriteBytes( gfpt->gfptSos, Extension, ExtLen ) != ExtLen )
	{ LDEB(ExtLen); return GIF_ERROR;	}

    sioOutPutCharacter( '\0', gfpt->gfptSos );

    return GIF_OK;
}

/******************************************************************************
*   Put an extension block (see GIF manual) into gif file.		      *
******************************************************************************/

int EGifPutExtension(	GifFileType *	gft,
			int		ExtCode,
			int		ExtLen,
			const void *	Extension )
{
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)gft->Private;

    if (!IS_WRITEABLE(gfpt)) {
	    /* This file was NOT open for writing: */
	    _GifError = E_GIF_ERR_NOT_WRITEABLE;
	    return GIF_ERROR;
    }

    if  ( ExtCode == 0 )
	{
	sioOutPutCharacter( ExtLen, gfpt->gfptSos );
	}
    else{
	sioOutPutCharacter( '!', gfpt->gfptSos );
	sioOutPutCharacter( ExtCode, gfpt->gfptSos );
	sioOutPutCharacter( ExtLen, gfpt->gfptSos );
	}

    if  ( sioOutWriteBytes( gfpt->gfptSos, Extension, ExtLen ) != ExtLen )
	{ LDEB(ExtLen); return GIF_ERROR;	}

    sioOutPutCharacter( '\0', gfpt->gfptSos );

    return GIF_OK;
}

/************************************************************************/
/*									*/
/*  Close a gif file for writing.					*/
/*									*/
/************************************************************************/

int EGifCloseFile(	GifFileType *	gft )
{
    GifFilePrivateType *	gfpt= (GifFilePrivateType *)gft->Private;

    if (!IS_WRITEABLE(gfpt)) {
	/* This file was NOT open for writing: */
	_GifError = E_GIF_ERR_NOT_WRITEABLE;
	return GIF_ERROR;
    }

    sioOutPutCharacter( ';', gfpt->gfptSos );

    if 	( gfpt )
	{
	bmGifCleanupCompress( gfpt );
	free( gfpt );
	}

    free( gft );

    return GIF_OK;
}
