

/*
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/

// $Id: Date.i,v 1.8 2002/01/16 14:51:30 nando Exp $

#ifndef quantlib_date_i
#define quantlib_date_i

%include String.i

%{
#include <cstdlib>
#include <string>
using QuantLib::Date;

using QuantLib::Day;
using QuantLib::Year;
%}

typedef int Day;
typedef int Year;

// typemap weekdays to corresponding strings

%{
using QuantLib::Weekday;
using QuantLib::Sunday;
using QuantLib::Monday;
using QuantLib::Tuesday;
using QuantLib::Wednesday;
using QuantLib::Thursday;
using QuantLib::Friday;
using QuantLib::Saturday;

using QuantLib::StringFormatter;
using QuantLib::DateFormatter;
%}

%typemap(ruby,in) Weekday (Weekday temp), const Weekday & (Weekday temp) {
    if (TYPE($source) == T_STRING) {
        std::string s(STR2CSTR($source));
        s = StringFormatter::toLowercase(s);
        if (s == "sun" || s == "sunday")
            temp = Sunday;
        else if (s == "mon" || s == "monday")
            temp = Monday;
        else if (s == "tue" || s == "tuesday")
            temp = Tuesday;
        else if (s == "wed" || s == "wednesday")
            temp = Wednesday;
        else if (s == "thu" || s == "thursday")
            temp = Thursday;
        else if (s == "fri" || s == "friday")
            temp = Friday;
        else if (s == "sat" || s == "saturday")
            temp = Saturday;
        else
            rb_raise(rb_eTypeError,"not a weekday");
    } else {
        rb_raise(rb_eTypeError,"not a weekday");
    }
    $target = &temp;
};

%typemap(ruby,out) Weekday, const Weekday & {
    switch (*$source) {
      case Sunday:      $target = rb_str_new2("Sunday");     break;
      case Monday:      $target = rb_str_new2("Monday");     break;
      case Tuesday:     $target = rb_str_new2("Tuesday");    break;
      case Wednesday:   $target = rb_str_new2("Wednesday");  break;
      case Thursday:    $target = rb_str_new2("Thursday");   break;
      case Friday:      $target = rb_str_new2("Friday");     break;
      case Saturday:    $target = rb_str_new2("Saturday");   break;
    }
};

%typemap(ruby,ret) Weekday {
    delete $source;
};



// typemap months to corresponding strings or numbers

%{
using QuantLib::Month;
using QuantLib::January;
using QuantLib::February;
using QuantLib::March;
using QuantLib::April;
using QuantLib::May;
using QuantLib::June;
using QuantLib::July;
using QuantLib::August;
using QuantLib::September;
using QuantLib::October;
using QuantLib::November;
using QuantLib::December;
%}

%typemap(ruby,in) Month (Month temp), const Month & (Month temp) {
    if (TYPE($source) == T_STRING) {
        std::string s(STR2CSTR($source));
        s = StringFormatter::toLowercase(s);
        if (s == "jan" || s == "january")
            temp = January;
        else if (s == "feb" || s == "february")
            temp = February;
        else if (s == "mar" || s == "march")
            temp = March;
        else if (s == "apr" || s == "april")
            temp = April;
	    else if (s == "may")
            temp = May;
        else if (s == "jun" || s == "june")
            temp = June;
        else if (s == "jul" || s == "july")
            temp = July;
        else if (s == "aug" || s == "august")
            temp = August;
        else if (s == "sep" || s == "september")
            temp = September;
        else if (s == "oct" || s == "october")
            temp = October;
        else if (s == "nov" || s == "november")
            temp = November;
        else if (s == "dec" || s == "december")
            temp = December;
        else
            rb_raise(rb_eTypeError,"not a month");
    } else if (TYPE($source) == T_FIXNUM) {
        int i = NUM2INT($source);
        if (i>=1 && i<=12)
            temp = Month(i);
        else
            rb_raise(rb_eTypeError,"not a month");
    } else {
        rb_raise(rb_eTypeError,"not a month");
    }
    $target = &temp;
};

%typemap(ruby,out) Month, const Month & {
    switch (*$source) {
      case January:     $target = rb_str_new2("January");    break;
      case February:    $target = rb_str_new2("February");   break;
      case March:       $target = rb_str_new2("March");      break;
      case April:       $target = rb_str_new2("April");      break;
      case May:         $target = rb_str_new2("May");        break;
      case June:        $target = rb_str_new2("June");       break;
      case July:        $target = rb_str_new2("July");       break;
      case August:      $target = rb_str_new2("August");     break;
      case September:   $target = rb_str_new2("September");  break;
      case October:     $target = rb_str_new2("October");    break;
      case November:    $target = rb_str_new2("November");   break;
      case December:    $target = rb_str_new2("December");   break;
    }
};

%typemap(ruby,ret) Month {
    delete $source;
};

// typemap time units to corresponding strings

%{
using QuantLib::TimeUnit;
using QuantLib::Days;
using QuantLib::Weeks;
using QuantLib::Months;
using QuantLib::Years;
%}

%typemap(ruby,in) TimeUnit (TimeUnit temp), const TimeUnit & (TimeUnit temp) {
    if (TYPE($source) == T_STRING) {
        std::string s(STR2CSTR($source));
        s = StringFormatter::toLowercase(s);
        if (s == "d" || s == "day" || s == "days")
            temp = Days;
        else if (s == "w" || s == "week" || s == "weeks")
            temp = Weeks;
        else if (s == "m" || s == "month" || s == "months")
            temp = Months;
        else if (s == "y" || s == "year" || s == "years")
            temp = Years;
        else
            rb_raise(rb_eTypeError,"not a time unit");
    } else {
        rb_raise(rb_eTypeError,"not a time unit");
    }
    $target = &temp;
};

%typemap(ruby,out) TimeUnit, const TimeUnit & {
    switch (*$source) {
      case Days:    $target = rb_str_new2("days");   break;
      case Weeks:   $target = rb_str_new2("weeks");  break;
      case Months:  $target = rb_str_new2("months"); break;
      case Years:   $target = rb_str_new2("years");  break;
    }
};

%typemap(ruby,ret) TimeUnit {
    delete $source;
}

%{
using QuantLib::Period;
using QuantLib::IntegerFormatter;
%}

class Period {
  public:
    Period(int n, TimeUnit units);
    ~Period();
    int length() const;
    TimeUnit units() const;
};

%addmethods Period {
    void crash() {}
    String __str__() {
        String s = IntegerFormatter::toString(self->length());
        switch (self->units()) {
          case Days:
            return s + " day(s)";
          case Weeks:
            return s + " week(s)";
          case Months:
            return s + " month(s)";
          case Years:
            return s + " year(s)";
          default:
            return "Unknown period";
        }
        QL_DUMMY_RETURN(String());
    }
}

// and finally, the Date class

class Date {
    %pragma(ruby) include = "Comparable";
  public:
    Date(Day d, Month m, Year y);
    ~Date();
    // access functions
    Weekday weekday() const;
    Day dayOfMonth() const;
    Day dayOfYear() const;        // one-based
    Month month() const;
    Year year() const;
    int serialNumber() const;
    // increment/decrement dates
    Date plusDays(int days) const;
    Date plusWeeks(int weeks) const;
    Date plusMonths(int months) const;
    Date plusYears(int years) const;
    Date plus(int units, TimeUnit) const;
    // leap years
    static bool isLeap(Year y);
    // earliest and latest allowed date
    static Date minDate();
    static Date maxDate();
};

%addmethods Date {
    void crash() {}
    int monthNumber() {
        return int(self->month());
    }
    int weekdayNumber() {
        return int(self->weekday());
    }
    Date __add__(int days) {
        return self->plusDays(days);
    }
    Date __sub__(int days) {
        return self->plusDays(-days);
    }
    int __cmp__(const Date& other) {
        if (*self < other)
            return -1;
        if (*self == other)
            return 0;
        return 1;
    }
    String __str__() {
        return DateFormatter::toString(*self);
    }
    Date succ() {
        return self->plusDays(1);
    }
}

%{
typedef std::vector<Date> DateVector;
using QuantLib::Null;
%}

class DateVector {
    %pragma(ruby) include = "Enumerable";
  public:
    ~DateVector();
};

%typemap(ruby,in) VALUE {
    $target = $source;
};

%addmethods DateVector {
    void crash() {}
    DateVector(VALUE v) {
    	if (rb_obj_is_kind_of(v,rb_cArray)) {
            int size = RARRAY(v)->len;
            DateVector* temp = new DateVector(size);
            for (int i=0; i<size; i++) {
                VALUE o = RARRAY(v)->ptr[i];
                if (o == Qnil) {
                    (*temp)[i] = Date();
                } else {
                    Date* d;
                    Get_Date(o,d);
                    (*temp)[i] = *d;
                }
            }
            return temp;
        } else {
            rb_raise(rb_eTypeError,
                "wrong argument type (expected array)");
        }
    }
    String __str__() {
        String s = "(";
        for (int i=0; i<self->size(); i++) {
            if (i != 0)
                s += ", ";
            s += DateFormatter::toString((*self)[i]);
        }
        s += ")";
        return s;
    }
    int __len__() {
        return self->size();
    }
    Date __getitem__(int i) {
        if (i>=0 && i<self->size()) {
            return (*self)[i];
        } else if (i<0 && -i<=self->size()) {
            return (*self)[self->size()+i];
        } else {
            throw IndexError("DateVector index out of range");
        }
        QL_DUMMY_RETURN(Date())
    }
    void __setitem__(int i, const Date& x) {
        if (i>=0 && i<self->size()) {
            (*self)[i] = x;
        } else if (i<0 && -i<=self->size()) {
            (*self)[self->size()+i] = x;
        } else {
            throw IndexError("DateVector index out of range");
        }
    }
    void each() {
        for (int i=0; i<self->size(); i++) {
            Date* d = new Date((*self)[i]);
            rb_yield(Wrap_Date(cDate,d));
        }
    }
};


%inline %{
    Date DateFromSerialNumber(int serialNumber) {
        return Date(serialNumber);
    }
%}


#endif

