
/*
 Copyright (C) 2000, 2001, 2002 RiskMap srl

 This file is part of QuantLib, a free-software/open-source library
 for financial quantitative analysts and developers - http://quantlib.org/

 QuantLib is free software: you can redistribute it and/or modify it under the
 terms of the QuantLib license.  You should have received a copy of the
 license along with this program; if not, please email ferdinando@ametrano.net
 The license is also available online at http://quantlib.org/html/license.html

 This program is distributed in the hope that it will be useful, but WITHOUT
 ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 FOR A PARTICULAR PURPOSE.  See the license for more details.
*/

// $Id: MarketElements.i,v 1.17 2002/03/05 16:58:17 lballabio Exp $

#ifndef quantlib_market_elements_i
#define quantlib_market_elements_i

%include Observer.i
%include Null.i
%include Functions.i

%{
using QuantLib::MarketElement;
using QuantLib::Handle;
using QuantLib::RelinkableHandle;
typedef Handle<MarketElement> MarketElementHandle;
typedef RelinkableHandle<MarketElement> MarketElementRelinkableHandle;
%}

// Python market element class
%typemap(python,in) PyObject* pyMarketElement {
	$target = $source;
}

%{
// its C++ wrapper
class PyMarketElement : public MarketElement {
  public:
	PyMarketElement(PyObject* pyMarketElement)
	: pyMarketElement_(pyMarketElement) {
	    /* make sure the Python object stays alive
	       as long as we need it */
	    Py_XINCREF(pyMarketElement_);
    }
    PyMarketElement(const PyMarketElement& me)
    : pyMarketElement_(me.pyMarketElement_) {
	    /* make sure the Python object stays alive
	       as long as we need it */
	    Py_XINCREF(pyMarketElement_);
    }
    PyMarketElement& operator=(const PyMarketElement& me) {
        if ((this != &me) && (pyMarketElement_ != me.pyMarketElement_)) {
            Py_XDECREF(pyMarketElement_);
            pyMarketElement_ = me.pyMarketElement_;
    	    Py_XINCREF(pyMarketElement_);
        }
        return *this;
    }
    ~PyMarketElement() {
        // now it can go as far as we are concerned
        Py_XDECREF(pyMarketElement_);
    }
	double value() const {
		PyObject* pyResult =
		  PyObject_CallMethod(pyMarketElement_,"value",NULL);
		QL_ENSURE(pyResult != NULL,
		  "failed to call value() on Python market element");
		double result = PyFloat_AsDouble(pyResult);
		Py_XDECREF(pyResult);
		return result;
	}
  private:
	PyObject* pyMarketElement_;
};
%}


// Export handle
%name(MarketElement) class MarketElementHandle : public ObservableHandle {
  public:
	// no constructor - forbid explicit construction
	~MarketElementHandle();
};

// replicate the MarketElement interface
%addmethods MarketElementHandle {
    MarketElementHandle(PyObject* pyMarketElement) {
        return new MarketElementHandle(new PyMarketElement(pyMarketElement));
    }
	double value() {
		return (*self)->value();
	}
    bool __nonzero__() {
        return !self->isNull();
    }
	void notifyObservers() {
	    (*self)->notifyObservers();
    }
}

AllowNoneAsInput(MarketElementHandle,MarketElement);

// export relinkable handle
%name(MarketElementHandle) class MarketElementRelinkableHandle {
  public:
    MarketElementRelinkableHandle(MarketElementHandle h);
	~MarketElementRelinkableHandle();
	void linkTo(MarketElementHandle);
};

%addmethods MarketElementRelinkableHandle {
    bool __nonzero__() {
        return !self->isNull();
    }
}


// actual market elements

%{
    using QuantLib::SimpleMarketElement;
    typedef Handle<SimpleMarketElement> SimpleMarketElementHandle;
%}

// Fake inheritance between Handles

%name(SimpleMarketElement)
class SimpleMarketElementHandle : public MarketElementHandle {
  public:
    // constructor redefined below
    ~SimpleMarketElementHandle();
};

%addmethods SimpleMarketElementHandle {
    SimpleMarketElementHandle(double value) {
        return new SimpleMarketElementHandle(
            new SimpleMarketElement(value));
    }
    void setValue(double value) {
        (*self)->setValue(value);
    }
}


%{
using QuantLib::DerivedMarketElement;
using QuantLib::CompositeMarketElement;
typedef Handle<DerivedMarketElement<PyUnaryFunction> >
    DerivedMarketElementHandle;
typedef Handle<CompositeMarketElement<PyBinaryFunction> >
    CompositeMarketElementHandle;
%}

%name(DerivedMarketElement)
class DerivedMarketElementHandle : public MarketElementHandle {
  public:
    // constructor redefined below
    ~DerivedMarketElementHandle();
};

%addmethods DerivedMarketElementHandle {
    DerivedMarketElementHandle(MarketElementRelinkableHandle h,
                               PyObject* pyFunction) {
        return new DerivedMarketElementHandle(
            new DerivedMarketElement<PyUnaryFunction>(
                h,PyUnaryFunction(pyFunction)));
    }
}


%name(CompositeMarketElement)
class CompositeMarketElementHandle : public MarketElementHandle {
  public:
    // constructor redefined below
    ~CompositeMarketElementHandle();
};

%addmethods CompositeMarketElementHandle {
    CompositeMarketElementHandle(MarketElementRelinkableHandle h1,
                                 MarketElementRelinkableHandle h2,
                                 PyObject* pyFunction) {
        return new CompositeMarketElementHandle(
            new CompositeMarketElement<PyBinaryFunction>(
                h1,h2,PyBinaryFunction(pyFunction)));
    }
}



#endif
