# --------------------------------------------------------------------------
#
# Copyright (c) Microsoft Corporation. All rights reserved.
#
# The MIT License (MIT)
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the ""Software""), to
# deal in the Software without restriction, including without limitation the
# rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
# sell copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED *AS IS*, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
# IN THE SOFTWARE.
#
# --------------------------------------------------------------------------

import json
import re
import logging
import time
import uuid

_LOGGER = logging.getLogger(__name__)
_ARMID_RE = re.compile(
    '(?i)/subscriptions/(?P<subscription>[^/]*)(/resourceGroups/(?P<resource_group>[^/]*))?'
    '(/providers/(?P<namespace>[^/]*)/(?P<type>[^/]*)/(?P<name>[^/]*)(?P<children>.*))?')

_CHILDREN_RE = re.compile('(?i)(/providers/(?P<child_namespace>[^/]*))?/'
                          '(?P<child_type>[^/]*)/(?P<child_name>[^/]*)')

def register_rp_hook(r, *args, **kwargs):
    """This is a requests hook to register RP automatically.

    You should not use this command manually, this is added automatically
    by the SDK.

    See requests documentation for details of the signature of this function.
    http://docs.python-requests.org/en/master/user/advanced/#event-hooks
    """
    if r.status_code == 409 and 'msrest' in kwargs:
        rp_name = _check_rp_not_registered_err(r)
        if rp_name:
            session = kwargs['msrest']['session']
            url_prefix = _extract_subscription_url(r.request.url)
            if not _register_rp(session, url_prefix, rp_name):
                return
            req = r.request
            # Change the 'x-ms-client-request-id' otherwise the Azure endpoint
            # just returns the same 409 payload without looking at the actual query
            if 'x-ms-client-request-id' in req.headers:
                req.headers['x-ms-client-request-id'] = str(uuid.uuid1())
            return session.send(req)

def _check_rp_not_registered_err(response):
    try:
        response = json.loads(response.content.decode())
        if response['error']['code'] == 'MissingSubscriptionRegistration':
            match = re.match(r".*'(.*)'", response['error']['message'])
            return match.group(1)
    except Exception:  # pylint: disable=broad-except
        pass
    return None

def _extract_subscription_url(url):
    """Extract the first part of the URL, just after subscription:
    https://management.azure.com/subscriptions/00000000-0000-0000-0000-000000000000/
    """
    match = re.match(r".*/subscriptions/[a-f0-9-]+/", url, re.IGNORECASE)
    if not match:
        raise ValueError("Unable to extract subscription ID from URL")
    return match.group(0)

def _register_rp(session, url_prefix, rp_name):
    """Synchronously register the RP is paremeter.
    
    Return False if we have a reason to believe this didn't work
    """
    post_url = "{}providers/{}/register?api-version=2016-02-01".format(url_prefix, rp_name)
    get_url = "{}providers/{}?api-version=2016-02-01".format(url_prefix, rp_name)
    _LOGGER.warning("Resource provider '%s' used by this operation is not "
                    "registered. We are registering for you.", rp_name)
    post_response = session.post(post_url)
    if post_response.status_code != 200:
        _LOGGER.warning("Registration failed. Please register manually.")
        return False

    while True:
        time.sleep(10)
        rp_info = session.get(get_url).json()
        if rp_info['registrationState'] == 'Registered':
            _LOGGER.warning("Registration succeeded.")
            return True

def parse_resource_id(rid):
    """Parses a resource_id into its various parts.

    Returns a dictionary with a single key-value pair, 'name': rid, if invalid resource id.

    :param rid: The resource id being parsed
    :type rid: str
    :returns: A dictionary with with following key/value pairs (if found):

        - subscription:            Subscription id
        - resource_group:          Name of resource group
        - namespace:               Namespace for the resource provider (i.e. Microsoft.Compute)
        - type:                    Type of the root resource (i.e. virtualMachines)
        - name:                    Name of the root resource
        - child_namespace_{level}: Namespace for the child resoure of that level
        - child_type_{level}:      Type of the child resource of that level
        - child_name_{level}:      Name of the child resource of that level
        - last_child_num:          Level of the last child
        - resource_parent:         Computed parent in the following pattern: providers/{namespace}\
        /{parent}/{type}/{name}
        - resource_namespace:      Same as namespace. Note that this may be different than the \
        target resource's namespace.
        - resource_type:           Type of the target resource (not the parent)
        - resource_name:           Name of the target resource (not the parent)

    :rtype: dict[str,str]
    """
    if not rid:
        return {}
    match = _ARMID_RE.match(rid)
    if match:
        result = match.groupdict()
        children = _CHILDREN_RE.finditer(result['children'] or '')
        count = None
        for count, child in enumerate(children):
            result.update({
                key + '_%d' % (count + 1): group for key, group in child.groupdict().items()})
        result['last_child_num'] = count + 1 if isinstance(count, int) else None
        result = _populate_alternate_kwargs(result)
    else:
        result = dict(name=rid)
    return {key: value for key, value in result.items() if value is not None}

def _populate_alternate_kwargs(kwargs):
    """ Translates the parsed arguments into a format used by generic ARM commands
    such as the resource and lock commands.
    """

    resource_namespace = kwargs['namespace']
    resource_type = kwargs.get('child_type_{}'.format(kwargs['last_child_num'])) or kwargs['type']
    resource_name = kwargs.get('child_name_{}'.format(kwargs['last_child_num'])) or kwargs['name']

    _get_parents_from_parts(kwargs)
    kwargs['resource_namespace'] = resource_namespace
    kwargs['resource_type'] = resource_type
    kwargs['resource_name'] = resource_name
    return kwargs

def _get_parents_from_parts(kwargs):
    """ Get the parents given all the children parameters.
    """
    parent_builder = []
    if kwargs['last_child_num'] is not None:
        parent_builder.append('{type}/{name}/'.format(**kwargs))
        for index in range(1, kwargs['last_child_num']):
            child_namespace = kwargs.get('child_namespace_{}'.format(index))
            if child_namespace is not None:
                parent_builder.append('providers/{}/'.format(child_namespace))
            kwargs['child_parent_{}'.format(index)] = ''.join(parent_builder)
            parent_builder.append(
                '{{child_type_{0}}}/{{child_name_{0}}}/'
                .format(index).format(**kwargs))
        child_namespace = kwargs.get('child_namespace_{}'.format(kwargs['last_child_num']))
        if child_namespace is not None:
            parent_builder.append('providers/{}/'.format(child_namespace))
        kwargs['child_parent_{}'.format(kwargs['last_child_num'])] = ''.join(parent_builder)
    kwargs['resource_parent'] = ''.join(parent_builder) if kwargs['name'] else None
    return kwargs

def resource_id(**kwargs):
    """Create a valid resource id string from the given parts.

    This method builds the resource id from the left until the next required id parameter
    to be appended is not found. It then returns the built up id.

    :param dict kwargs: The keyword arguments that will make up the id.

        The method accepts the following keyword arguments:
            - subscription (required): Subscription id
            - resource_group:          Name of resource group
            - namespace:               Namespace for the resource provider (i.e. Microsoft.Compute)
            - type:                    Type of the resource (i.e. virtualMachines)
            - name:                    Name of the resource (or parent if child_name is also \
            specified)
            - child_namespace_{level}: Namespace for the child resoure of that level (optional)
            - child_type_{level}:      Type of the child resource of that level
            - child_name_{level}:      Name of the child resource of that level

    :returns: A resource id built from the given arguments.
    :rtype: str
    """
    kwargs = {k: v for k, v in kwargs.items() if v is not None}
    rid_builder = ['/subscriptions/{subscription}'.format(**kwargs)]
    try:
        try:
            rid_builder.append('resourceGroups/{resource_group}'.format(**kwargs))
        except KeyError:
            pass
        rid_builder.append('providers/{namespace}'.format(**kwargs))
        rid_builder.append('{type}/{name}'.format(**kwargs))
        count = 1
        while True:
            try:
                rid_builder.append('providers/{{child_namespace_{}}}'
                                   .format(count).format(**kwargs))
            except KeyError:
                pass
            rid_builder.append('{{child_type_{0}}}/{{child_name_{0}}}'
                               .format(count).format(**kwargs))
            count += 1
    except KeyError:
        pass
    return '/'.join(rid_builder)

def is_valid_resource_id(rid, exception_type=None):
    """Validates the given resource id.

    :param rid: The resource id being validated.
    :type rid: str
    :param exception_type: Raises this Exception if invalid.
    :type exception_type: :class:`Exception`
    :returns: A boolean describing whether the id is valid.
    :rtype: bool
    """
    is_valid = False
    try:
        is_valid = rid and resource_id(**parse_resource_id(rid)).lower() == rid.lower()
    except KeyError:
        pass
    if not is_valid and exception_type:
        raise exception_type()
    return is_valid
