/*
    PaletteGeneralAttributes.m

    Implementation of the PaletteGeneralAttributes class for the
    ProjectManager application.

    Copyright (C) 2005, 2006  Saso Kiselkov

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#import "PaletteGeneralAttributes.h"

#import <Foundation/NSString.h>

#import <AppKit/NSTextField.h>
#import <AppKit/NSNibLoading.h>
#import <AppKit/NSTableView.h>

#import "PaletteProjectType.h"
#import "../../CommonFunctions/StringValidation.h"

@implementation PaletteGeneralAttributes

- (void) dealloc
{
  TEST_RELEASE(view);

  [super dealloc];
}

- initWithOwner: (PaletteProjectType *) anOwner
{
  if ((self = [self init]) != nil)
    {
      owner = anOwner;
    }

  return self;
}

- (void) awakeFromNib
{
  [view retain];
  [view removeFromSuperview];
  DESTROY(bogusWindow);

  [copyright setStringValue: [owner copyright]];
  [copyrightDescription setStringValue: [owner copyrightDescription]];
  [iconFilename setStringValue: [owner iconName]];
  [mainNibFile setStringValue: [owner mainNibFile]];
  [principalClass setStringValue: [owner principalClass]];
}


- (void) addAuthor: (id)sender
{
  NSMutableArray * newAuthors = [[[owner authors] mutableCopy] autorelease];

  [newAuthors addObject: @""];
  [owner setAuthors: newAuthors];
  [authors reloadData];
}


- (void) changeCopyright: (id)sender
{
  [owner setCopyright: NonEmptyString([copyright stringValue])];
}


- (void) changeCopyrightDescription: (id)sender
{
  [owner setCopyrightDescription: NonEmptyString([copyrightDescription
    stringValue])];
}


- (void) changeIconFilename: (id)sender
{
  [owner setIconName: NonEmptyString([iconFilename stringValue])];
}


- (void) changeMainNibFile: (id)sender
{
  [owner setMainNibFile: NonEmptyString([mainNibFile stringValue])];
}


- (void) changePrincipalClass: (id)sender
{
  [owner setPrincipalClass: NonEmptyString([principalClass stringValue])];
}


- (void) moveAuthorUp: (id)sender
{
  int row = [authors selectedRow];

  if (row > 0)
    {
      NSMutableArray * newAuthors = [[[owner authors] mutableCopy]
        autorelease];
      NSString * originalAuthor;

      originalAuthor = [[newAuthors objectAtIndex: row - 1] retain];
      [newAuthors replaceObjectAtIndex: row - 1
                            withObject: [newAuthors objectAtIndex: row]];
      [newAuthors replaceObjectAtIndex: row
                            withObject: originalAuthor];
      [owner setAuthors: newAuthors];

      [authors reloadData];
      [authors selectRow: row - 1 byExtendingSelection: NO];
    }
}


- (void) moveAuthorDown: (id)sender
{
  int row = [authors selectedRow];
  NSArray * authorsArray = [owner authors];

  if (row >= 0 && row + 1 < (int) [authorsArray count])
    {
      NSMutableArray * newAuthors = [[authorsArray mutableCopy] autorelease];
      NSString * originalAuthor;

      originalAuthor = [[newAuthors objectAtIndex: row + 1] retain];
      [newAuthors replaceObjectAtIndex: row + 1
                            withObject: [newAuthors objectAtIndex: row]];
      [newAuthors replaceObjectAtIndex: row
                            withObject: originalAuthor];
      [owner setAuthors: newAuthors];

      [authors reloadData];
      [authors selectRow: row + 1 byExtendingSelection: NO];
    }
}


- (void) removeAuthor: (id)sender
{
  int row = [authors selectedRow];

  if (row >= 0)
    {
      NSMutableArray * newAuthors = [[[owner authors] mutableCopy]
        autorelease];

      [newAuthors removeObjectAtIndex: row];
      [owner setAuthors: newAuthors];

      [authors reloadData];
    }
}

- (void) addExportedClass: sender
{
  NSMutableArray * newExportedClasses = [NSMutableArray array];

  [newExportedClasses addObjectsFromArray: [owner exportedClasses]];

  [newExportedClasses addObject: _(@"NewClass")];
  [owner setExportedClasses: newExportedClasses];
  [exportedClasses reloadData];
}

- (void) removeExportedClass: sender
{
  int row = [exportedClasses selectedRow];

  if (row >= 0)
    {
      NSMutableArray * newExportedClasses = [[[owner exportedClasses]
        mutableCopy] autorelease];

      [newExportedClasses removeObjectAtIndex: row];
      [owner setExportedClasses: newExportedClasses];
    }
}

- (NSView *) view
{
  if (view == nil)
    {
      [NSBundle loadNibNamed: @"PaletteGeneralAttributes" owner: self];
    }

  return view;
}

- (int) numberOfRowsInTableView: (NSTableView *)aTableView
{
  if (aTableView == authors)
    {
      return [[owner authors] count];
    }
  else
    {
      return [[owner exportedClasses] count];
    }
}

- (id) tableView: (NSTableView *)aTableView 
objectValueForTableColumn: (NSTableColumn *)aTableColumn 
             row: (int)rowIndex
{
  if (aTableView == authors)
    {
      return [[owner authors] objectAtIndex: rowIndex];
    }
  else
    {
      return [[owner exportedClasses] objectAtIndex: rowIndex];
    }
}

- (void) tableView: (NSTableView *)aTableView 
    setObjectValue: (id)anObject 
    forTableColumn: (NSTableColumn *)aTableColumn
               row: (int)rowIndex
{
  if (aTableView == authors)
    {
      NSMutableArray * newAuthors = [[[owner authors] mutableCopy] autorelease];

      [newAuthors replaceObjectAtIndex: rowIndex withObject: anObject];
      [owner setAuthors: newAuthors];
    }
  else
    {
      NSMutableArray * newExportedClasses = [[[owner exportedClasses]
        mutableCopy] autorelease];

      [newExportedClasses replaceObjectAtIndex: rowIndex withObject: anObject];
      [owner setExportedClasses: newExportedClasses];
    }
}

@end
