/*
 * potool is a program aiding editing of po files
 * Copyright (C) 1999-2000 Zbigniew Chyla
 *
 * see LICENSE for licensing info
 */
#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <getopt.h>
#include <string.h>
#include <ctype.h>
#include "i18n.h"
#include "po-gram.h"

typedef gboolean po_filter_func(po_entry *);

/* --- po_entry filters --- */

gboolean po_filter_translated(po_entry *po)
{
	return po->str[0] != '\0';
}

gboolean po_filter_not_translated(po_entry *po)
{
	return po->str[0] == '\0';
}

gboolean po_filter_fuzzy(po_entry *po)
{
	return po->is_fuzzy;
}

gboolean po_filter_not_fuzzy(po_entry *po)
{
	return !po->is_fuzzy;
}

/* -- */

GList *po_get_filtered(GList *po_list, po_filter_func *filter)
{
	GList *npo_list, *l;

	for ( npo_list = NULL, l = po_list; l != NULL; l = g_list_next(l) ) {
		if ( filter((po_entry *) l->data) ) {
			npo_list = g_list_prepend(npo_list, l->data);
		}
	}
	return g_list_reverse(npo_list);
}

typedef enum { FUZZY_FILTER = 1, NOT_FUZZY_FILTER = 2,
               TRANSLATED_FILTER = 4, NOT_TRANSLATED_FILTER = 8 } po_filters;

GList *po_apply_filters(GList *po_list, po_filters filters)
{
	if ( (filters & FUZZY_FILTER) != 0 ) {
		po_list = po_get_filtered(po_list, po_filter_fuzzy);
	}
	if ( (filters & NOT_FUZZY_FILTER) != 0 ) {
		po_list = po_get_filtered(po_list, po_filter_not_fuzzy);
	}
	if ( (filters & TRANSLATED_FILTER) != 0 ) {
		po_list = po_get_filtered(po_list, po_filter_translated);
	}
	if ( (filters & NOT_TRANSLATED_FILTER) != 0 ) {
		po_list = po_get_filtered(po_list, po_filter_not_translated);
	}
	return po_list;
}

/* --- */

typedef enum { NO_ID = 1, NO_STR = 2, NO_STD_COMMENT = 4,
               NO_POS_COMMENT = 8, NO_SPEC_COMMENT = 16,
               NO_RES_COMMENT = 32, NO_TRANSLATION = 64,
               NO_LINF =128 } po_write_modes;

/* FIXME!!! */
void print_multi_line(gchar *s, gint ulen)
{
	gchar **lines, **ln;
	enum { max_len = 77 };
	gchar *eol = strstr(s, "\\n");
	gboolean iseeof = strcmp(s + strlen(s) - 2, "\\n") == 0;

	if ( (eol == NULL || *(eol + 2) == '\0') && strlen(s) < (78 - ulen)) {
		printf("\"%s\"\n", s);
		return;
	}

	printf("\"\"\n");
	lines = g_strsplit(s, "\\n", 0);
	for ( ln = lines; *ln != NULL; ln++ ) {
		gchar *l;
		gint llen, wl;
		gchar sep[] = " \t";
		gboolean iseof = *(ln + 1) != NULL || iseeof;

		printf("\"");
		llen = 0;
		l = *ln;
		do {
			gint eoflen;

			wl = strcspn(l, sep);
			wl += strspn(l + wl, sep);

			eoflen = ( *(l + wl) == '\0' && !(strchr(sep, *(l + wl - 1)))
			           && iseof ) ? 2 : 0;
			if ( llen + wl + eoflen > max_len ) {
				printf("\"\n\"");
				llen = 0;
			}
			fwrite(l, 1, wl, stdout);
			llen += wl;
			l += wl;
		} while ( *l != '\0' );

		if ( iseof ) {
			if ( llen + 2 > max_len ) {
				printf("\"\n\"\\n");
			} else {
				printf("\\n");
			}
		}
		printf("\"\n");
	}
	g_strfreev(lines);


}

void po_write(GList *po_list, po_write_modes mode)
{
	GList *l;

	for ( l = po_list; l != NULL; l = g_list_next(l) ) {
		GList *ll;
		po_entry *po = (po_entry *) l->data;

		if ( l != po_list ) {
			printf("\n");
		}

		if ( !(mode & NO_STD_COMMENT) ) {
			for ( ll = po->comments_std; ll != NULL; ll = g_list_next(ll) ) {
				printf("# %s\n", (gchar *) ll->data);
			}
		}
		if ( !(mode & NO_RES_COMMENT) ) {
			for ( ll = po->comments_res; ll != NULL; ll = g_list_next(ll) ) {
				printf("#%s\n", (gchar *) ll->data);
			}
		}
		if ( !(mode & NO_POS_COMMENT) ) {
			if ( !(mode & NO_LINF) ) {
				for ( ll = po->comments_pos; ll != NULL; ll = g_list_next(ll) ) {
					printf("#:%s\n", (gchar *) ll->data);
				}
			} else {
				for ( ll = po->comments_pos; ll != NULL; ll = g_list_next(ll) ) {
					gchar *s = g_strdup((gchar *) ll->data);
					gchar *l, *r;

					l = r = s;
					while ( *r != '\0' ) {
						if ( *r == ':' ) {
							*l++ = ':';
							*l++ = '1';
							while ( isdigit(*++r) )
								;
						} else {
							*l++ = *r++;
						}
					}
					*l = '\0';
					printf("#:%s\n", s);
					g_free(s);
				}
			}
		}
		if ( !(mode & NO_SPEC_COMMENT) ) {
			for ( ll = po->comments_spec; ll != NULL; ll = g_list_next(ll) ) {
				printf("#,%s\n", (gchar *) ll->data);
			}
		}
		if ( !(mode & NO_ID) ) {
			printf("msgid ");
			print_multi_line(po->id, 6);
		}
		if ( !(mode & NO_STR) ) {
			if ( !(mode & NO_TRANSLATION) ) {
				printf("msgstr ");
				print_multi_line(po->str, 7);
			} else {
				printf("msgstr \"\"\n");
			}
		}
	}
}

/* - */

typedef GHashTable po_entry_set;

/* FIXME */
guint po_hash_id(gchar *id)
{
	guint s;
	gchar *p;

	for ( s = 0, p = id; *p != '\0'; p++ ) {
		s += *p;
	}
	return s;
}

gint po_compare_id(gchar *id1, gchar *id2)
{
	return !strcmp(id1, id2);
}

void po_set_write_entry(gchar *id, po_entry *po, void *d)
{
	printf("id: %s (%s)\n"
	       "str: %s\n",
	       id, po->id, po->str);
}

po_entry_set *po_set_create(GList *po_list)
{
	GList *l;
	GHashTable *hash = g_hash_table_new((GHashFunc) &po_hash_id,
	                                    (GCompareFunc) &po_compare_id);
	for ( l = po_list; l != NULL; l = g_list_next(l) ) {
		po_entry *po = l->data;

		g_hash_table_insert(hash, po->id, po);
	}
	return hash;
}

po_entry_set *po_set_update(po_entry_set *po_set, GList *po_list)
{
	GList *l;

	for ( l = po_list; l != NULL; l = g_list_next(l) ) {
		po_entry *po = (po_entry *) l->data, *hpo;

		if ( (hpo = g_hash_table_lookup(po_set, po->id)) != NULL ) {
			hpo->str = po->str;
			hpo->comments_std = po->comments_std;
			hpo->comments_pos = po->comments_pos;
			hpo->comments_res = po->comments_res;
			hpo->comments_spec = po->comments_spec;
			hpo->is_fuzzy = po->is_fuzzy;
			hpo->is_c_format = po->is_c_format;
		} else {
			g_warning(_("Unknown msgid: %s"), po->id);
		}
	}
	return po_set;
}

gint main(gint argc, gchar **argv)
{
	gint c;
	/* -- */
	gboolean istats = FALSE;
	po_filters ifilters = 0;
	po_write_modes write_mode = 0;

	/*
		FILENAME1 [FILENAME2]
		[-f f|nf|t|nt]
		[-s]
		[-n id|str|cmt|ucmt|pcmt|scmt|dcmt|tr|linf]...
		[-h]
	 */
	while ( (c = getopt(argc, argv, "f:n:sh")) != EOF) {
		switch ( c ) {
			case 'h' :
				printf(_(
				"Usage: %s -i FILENAME1 [FILENAME2] [FITERS] [-s] [OUTPUT_OPTIONS] [-h]\n"
				"\n"
				), argv[0]);
				exit(0);
				break;
			case 'n' :
				if ( strcmp(optarg, "id") == 0 ) {
					write_mode |= NO_ID;
				} else if ( strcmp(optarg, "str") == 0 ) {
					write_mode |= NO_STR;
				} else if ( strcmp(optarg, "cmt") == 0 ) {
					write_mode |= NO_STD_COMMENT | NO_POS_COMMENT |
					              NO_SPEC_COMMENT | NO_RES_COMMENT;
				} else if ( strcmp(optarg, "ucmt") == 0 ) {
					write_mode |= NO_STD_COMMENT;
				} else if ( strcmp(optarg, "pcmt") == 0 ) {
					write_mode |= NO_POS_COMMENT;
				} else if ( strcmp(optarg, "scmt") == 0 ) {
					write_mode |= NO_SPEC_COMMENT;
				} else if ( strcmp(optarg, "dcmt") == 0 ) {
					write_mode |= NO_RES_COMMENT;
				} else if ( strcmp(optarg, "tr") == 0 ) {
					write_mode |= NO_TRANSLATION;
				} else if ( strcmp(optarg, "linf") == 0 ) {
					write_mode |= NO_LINF;
				} else {
					g_error(_("Unknown parameter for -n option!"));
				}
				break;
			case 's' :
				istats = TRUE;
				break;
			case 'f' :
				if ( strcmp(optarg, "f") == 0 ) {
					ifilters |= FUZZY_FILTER;
				} else if ( strcmp(optarg, "nf") == 0 ) {
					ifilters |= NOT_FUZZY_FILTER;
				} else if ( strcmp(optarg, "t") == 0 ) {
					ifilters |= TRANSLATED_FILTER;
				} else if ( strcmp(optarg, "nt") == 0 ) {
					ifilters |= NOT_TRANSLATED_FILTER;
				} else {
					g_error(_("Unknown filter!"));
				}
				break;
			case ':' :
				g_error(_("Invalid parameter!"));
				break;
			case '?' :
				g_error(_("Invalid option!"));
				break;
			default :
				g_assert_not_reached();
		}
	}

	if ( optind >= argc ) {
		g_error(_("Input file not specified!"));
	}
	if ( argc - optind == 1 ) {
		GList *po_list;
		gchar *ifn = argv[optind];

		po_list = po_apply_filters(po_read(ifn), ifilters);

		if ( istats ) {
			printf(_("%d\n"), g_list_length(po_list));
		} else {
			po_write(po_list, write_mode);
		}
	} else {
		GList *bpo_list, *po_list;
		po_entry_set *bpo_set;
		gchar *bfn = argv[optind], *fn = argv[optind + 1];

		bpo_list = po_read(bfn);
		bpo_set = po_set_create(bpo_list);
		po_list = po_apply_filters(po_read(fn), ifilters);
		bpo_set = po_set_update(bpo_set, po_list);
		po_write(bpo_list, write_mode);
	}

	return 0;
}
